\name{hitandrun}
\alias{hitandrun}
\alias{har.init}
\alias{har.run}
\title{
"Hit and Run" sampler
}
\description{
The "Hit and Run" method generates a Markov Chain whose stable state converges on the uniform distribution over a convex polytope defined by a set of linear inequality constraints. \code{hitandrun} further uses the Moore-Penrose pseudo-inverse to eliminate an arbitrary set of linear equality constraints before applying the "Hit and Run" sampler.

\code{har.init} and \code{har.run} together provide a re-entrant version of \code{hitandrun} so that the Markov chain can be continued if convergence is not satisfactory.
}
\usage{
hitandrun(constr, n.samples=1E4,
    thin.fn = function(n) { ceiling(log(n + 1)/4 * n^3) }, thin = NULL,
    x0.randomize=FALSE, x0.method="slacklp", x0 = NULL)

har.init(constr,
    thin.fn = function(n) { ceiling(log(n + 1)/4 * n^3) }, thin = NULL,
    x0.randomize=FALSE, x0.method="slacklp", x0 = NULL)

har.run(state, n.samples)
}
\arguments{
  \item{constr}{Linear constraints that define the sampling space (see details)}
	\item{n.samples}{The desired number of samples to return. The sampler is run for \code{n.samples * thin} iterations}
	\item{thin.fn}{Function that specifies a thinning factor depending on the dimension of the sampling space after equality constraints have been eliminated. Will only be invoked if \code{thin} is \code{NULL}}
  \item{thin}{The thinning factor}
  \item{x0}{Seed point for the Markov Chain. Can be used to continue sampling if convergence is inadequate (see value)}
  \item{x0.randomize}{Whether to generate a random seed point if \code{x0} is unspecified}
  \item{x0.method}{Method to generate the seed point if \code{x0} is unspecified, see \code{\link{createSeedPoint}}}
  \item{state}{A state object, as generated by {\code{har.init}} (see value)}
}
\details{
The constraints are given as a list with the elements \code{constr}, \code{dir} and \code{rhs}. \code{dir} is a vector with values \code{'='} or \code{'<='}. \code{constr} is a matrix and \code{rhs} a vector, which encode the standard linear programming constraint froms \eqn{Ax = b} and \eqn{Ax \leq b} (depending on \code{dir}). The lengths of \code{rhs} and \code{dir} must match the number of rows of \code{constr}.

\code{hitandrun} applies \code{\link{solution.basis}} to generate a basis of the (translated) solution space of the linear constraints (if any). An affine transformation is generated using \code{\link{createTransform}} and applied to the constraints. Then, a seed point satisfying the inequality constraints is generated using \code{\link{createSeedPoint}}. Finally, \code{\link{har}} is used to generate the samples.
}
\value{
For \code{hitandrun}, a matrix containing the generated samples as rows.

For \code{har.init}, a state object, containing:
  \item{basis}{The basis for the sampling space. See \code{\link{solution.basis}}.}
  \item{transform}{The sampling space transformation. See \code{\link{createTransform}}.}
  \item{constr}{The linear inequality constraints translated to the sampling space. See \code{\link{transformConstraints}}.}
  \item{x0}{The generated seed point. See \code{\link{createSeedPoint}}.}
  \item{thin}{The thinning factor to be used.}

For \code{har.run}, a list containing:
	\item{samples}{A matrix containing the generated samples as rows.}
	\item{state}{A state object that can be used to continue sampling from the Markov chain (i.e. \code{x0} has been modified).}
}
\author{
Gert van Valkenhoef
}
\note{
"Hit and Run" is a Markov Chain Monte Carlo (MCMC) method, so generated samples form a correlated time series. To get a uniform sample, you need \eqn{O^*(n^3)}{O*(n^3)} samples, where n is the dimension of the sampling space.
}
\seealso{
\code{\link{harConstraints}}
\code{\link{har}}
}
\examples{
# Sample from the 3-simplex with the additional constraint that w_1/w_2 = 2
# Three inequality constraints, two equality constraints
constr <- mergeConstraints(simplexConstraints(3), exactRatioConstraint(3, 1, 2, 2))
samples <- hitandrun(constr, n.samples=1000)
stopifnot(dim(samples) == c(1000, 3))
stopifnot(all.equal(apply(samples, 1, sum), rep(1, 1000)))
stopifnot(all.equal(samples[,1]/samples[,2], rep(2, 1000)))

# Sample from the unit rectangle (no equality constraints)
constr <- list(
  constr = rbind(c(1,0), c(0,1), c(-1,0), c(0,-1)),
  dir=rep('<=', 4),
  rhs=c(1, 1, 0, 0))
state <- har.init(constr)
result <- har.run(state, n.samples=1000)
samples <- result$samples
stopifnot(all(samples >= 0 & samples <= 1))
# Continue sampling from the same chain:
result <- har.run(result$state, n.samples=1000)
samples <- rbind(samples, result$samples)
}
\keyword{hit-and-run}
\keyword{uniform sampling}
