% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cv_cureem.R
\name{cv_cureem}
\alias{cv_cureem}
\title{Fit penalized mixture cure model using the E-M algorithm with cross-validation for parameter tuning}
\usage{
cv_cureem(
  formula,
  data,
  subset,
  x.latency = NULL,
  model = "cox",
  penalty = "lasso",
  penalty.factor.inc = NULL,
  penalty.factor.lat = NULL,
  fdr.control = FALSE,
  fdr = 0.2,
  grid.tuning = FALSE,
  thresh = 0.001,
  scale = TRUE,
  maxit = NULL,
  inits = NULL,
  lambda.inc.list = NULL,
  lambda.lat.list = NULL,
  nlambda.inc = NULL,
  nlambda.lat = NULL,
  gamma.inc = 3,
  gamma.lat = 3,
  lambda.min.ratio.inc = 0.1,
  lambda.min.ratio.lat = 0.1,
  n_folds = 5,
  measure.inc = "c",
  one.se = FALSE,
  cure_cutoff = 5,
  parallel = FALSE,
  seed = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{formula}{an object of class "\code{formula}" (or one that can be coerced to that class): a symbolic description of the model to be fitted. The response must be a survival object as returned by the \code{Surv} function while the variables on the right side of the formula are the covariates that are included in the incidence portion of the model.}

\item{data}{a data.frame in which to interpret the variables named in the \code{formula} or in the \code{subset} argument.}

\item{subset}{an optional expression indicating which subset of observations to be used in the fitting process, either a numeric or factor variable should be used in subset, not a character variable. All observations are included by default.}

\item{x.latency}{specifies the variables to be included in the latency portion of the model and can be either a matrix of predictors, a model formula with the right hand side specifying the latency variables, or the same data.frame passed to the \code{data} parameter. Note that when using the model formula syntax for \code{x.latency} it cannot handle \code{x.latency = ~ .}.}

\item{model}{type of regression model to use for the latency portion of mixture cure model. Can be "cox", "weibull", or "exponential" (default is "cox").}

\item{penalty}{type of penalty function. Can be "lasso", "MCP", or "SCAD" (default is "lasso").}

\item{penalty.factor.inc}{vector of binary indicators representing the penalty to apply to each incidence coefficient: 0 implies no shrinkage and 1 implies shrinkage. If not supplied, 1 is applied to all incidence variables.}

\item{penalty.factor.lat}{vector of binary indicators representing the penalty to apply to each latency coefficient: 0 implies no shrinkage and 1 implies shrinkage. If not supplied, 1 is applied to all latency variables.}

\item{fdr.control}{logical, if TRUE, model-X knockoffs are used for FDR-controlled variable selection and indices of selected variables are returned (default is FALSE).}

\item{fdr}{numeric value in (0, 1) range specifying the target FDR level to use for variable selection when \code{fdr.control=TRUE} (default is 0.2).}

\item{grid.tuning}{logical, if TRUE a 2-D grid tuning approach is used to select the optimal pair of \eqn{\lambda_b} and \eqn{\lambda_{\beta}} penalty parameters for the incidence and latency portions of the model, respectively. Otherwise the \eqn{\lambda_b} and \eqn{\lambda_{\beta}} are selected from a 1-D sequence and are equal to one another (default is FALSE).}

\item{thresh}{small numeric value. The iterative process stops when the differences between successive expected penalized complete-data log-likelihoods for both incidence and latency components are less than this specified level of tolerance (default is 10^-3).}

\item{scale}{logical, if TRUE the predictors are centered and scaled.}

\item{maxit}{maximum number of passes over the data for each lambda. If not specified, 100 is applied when \code{penalty = "lasso"} and 1000 is applied when \code{penalty = "MCP"} or \code{penalty = "SCAD"}.}

\item{inits}{an optional list specifiying the initial value for the incidence intercept (\code{itct}), a numeric vector for the unpenalized incidence coefficients (\code{b_u}), and a numeric vector for unpenalized latency coefficients (\code{beta_u}).  For parametric models, it should also include a numeric value for the rate parameter (\code{lambda}) when \code{model = "weibull"} or \code{model = "exponential"}, and a numeric value for the shape parameter (\code{alpha}) when \code{model = "weibull"}. When \code{model = "cox"}, it should also include a numeric vector for the latency survival probabilities \eqn{S_u(t_i|w_i)} for i=1,...,N (\code{survprob}). Penalized coefficients are initialized to zero. If \code{inits} is not specified or improperly specified, initialization is automatically provided by the function.}

\item{lambda.inc.list}{a numeric vector used to search for the optimal \eqn{\lambda_b} tuning parameter. If not supplied, the function computes a \eqn{\lambda_b} sequence based on \code{nlambda.inc} and \code{lambda.min.ratio.inc}. If \code{grid.tuning=FALSE}, the same sequence should be used for both \eqn{\lambda_b} and \eqn{\lambda_{\beta}}.}

\item{lambda.lat.list}{a numeric vector used to search for the optimal \eqn{\lambda_{\beta}} tuning parameter. If not supplied, the function computes a \eqn{\lambda_{\beta}} sequence based on \code{nlambda.lat} and \code{lambda.min.ratio.lat}. If \code{grid.tuning=FALSE}, the same sequence should be used for both \eqn{\lambda_b} and \eqn{\lambda_{\beta}}.}

\item{nlambda.inc}{an integer specifying the number of values to search for the optimal \eqn{\lambda_b} tuning parameter; default is 10 if \code{grid.tuning=TRUE} and 50 otherwise.}

\item{nlambda.lat}{an integer specifying the number of values to search for the optimal \eqn{\lambda_{\beta}} tuning parameter; default is 10 if \code{grid.tuning=TRUE} and 50 otherwise.}

\item{gamma.inc}{numeric value for the penalization parameter \eqn{\gamma} for variables in the incidence portion of the model when \code{penalty = "MCP"} or \code{penalty = "SCAD"} (default is 3).}

\item{gamma.lat}{numeric value for the penalization parameter \eqn{\gamma} for variables in the latency portion of the model when \code{penalty = "MCP"} or \code{penalty = "SCAD"} (default is 3).}

\item{lambda.min.ratio.inc}{numeric value in (0,1) representing the smallest value for \eqn{\lambda_b} as a fraction of \code{lambda.max.inc}, the data-derived entry value at which essentially all penalized variables in the incidence portion of the model have a coefficient estimate of 0 (default is 0.1).}

\item{lambda.min.ratio.lat}{numeric value in (0.1) representing the smallest value for \eqn{\lambda_{\beta}} as a fraction of \code{lambda.max.lat}, the data-derived entry value at essentially all penalized variables in the latency portion of the model have a coefficient estimate of 0 (default is 0.1).}

\item{n_folds}{an integer specifying the number of folds for the k-fold cross-valiation procedure (default is 5).}

\item{measure.inc}{character string specifying the evaluation criterion used in selecting the optimal \eqn{\lambda_b}. Can be "c" or "auc"; default is "c". If \code{measure.inc="c"}, the C-statistic using the cure status weighting (CSW) method proposed by Asano and Hirakawa (2017) is used to select both \eqn{\lambda_b} and \eqn{\lambda_{\beta}}. If \code{measure.inc="auc"}, the AUC for cure prediction using the mean score imputation (MSI) method proposed by Asano et al. (2014) is used to select \eqn{\lambda_b} while the C-statistic with CSW is used for \eqn{\lambda_{\beta}}.}

\item{one.se}{logical, if TRUE then the one standard error rule is applied for selecting the optimal parameters. The one standard error rule selects the most parsimonious model having evaluation criterion no more than one standard error worse than that of the best evaluation criterion (default is FALSE).}

\item{cure_cutoff}{numeric value representing the cutoff time value that represents subjects not experiencing the event by this time are cured. This value is used to produce a proxy for the unobserved cure status when calculating C-statistic and AUC (default is 5 representing 5 years). Users should be careful to note the time scale of their data and adjust this according to the time scale and clinical application.}

\item{parallel}{logical. If TRUE, parallel processing is performed for K-fold CV using \code{foreach} and the \pkg{doMC} package is required.}

\item{seed}{optional integer representing the random seed. Setting the random seed fosters reproducibility of the results.}

\item{verbose}{logical, if TRUE running information is printed to the console (default is FALSE).}

\item{...}{additional arguments.}
}
\value{
\item{b0}{Estimated intercept for the incidence portion of the model.}

\item{b}{Estimated coefficients for the incidence portion of the model.}

\item{beta}{Estimated coefficients for the latency portion of the model.}

\item{alpha}{Estimated shape parameter if the Weibull model is fit.}

\item{rate}{Estimated rate parameter if the Weibull or exponential model is fit.}

\item{logLik.inc }{Expected penalized complete-data log-likelihood for the incidence portion of the model.}

\item{logLik.lat }{Expected penalized complete-data log-likelihood for the latency portion of the model.}

\item{selected.lambda.inc }{Value of \eqn{\lambda_b} selected using cross-validation. NULL when fdr.control is TRUE.}

\item{selected.lambda.lat }{Value of \eqn{\lambda_{\beta}} selected using cross-validation. NULL when fdr.control is TRUE.}

\item{max.c}{Maximum C-statistic achieved.}

\item{max.auc}{Maximum AUC for cure prediction achieved; only output when \code{measure.inc="auc"}.}

\item{selected.index.inc }{Indices of selected variables for the incidence portion of the model when \code{fdr.control=TRUE}. If no variables are selected, \code{int(0)} will be returned.}

\item{selected.index.lat }{Indices of selected variables for the latency portion of the model when \code{fdr.control=TRUE}. If no variables are selected, \code{int(0)} will be returned.}

\item{call}{the matched call.}
}
\description{
Fits a penalized parametric and semi-parametric mixture cure model (MCM) using the E-M algorithm with with k-fold cross-validation for parameter tuning. The lasso (L1), MCP and SCAD penalty are supported for the Cox MCM while only lasso is currently supported for parametric MCMs. When FDR controlled variable selection is used, the model-X knockoffs method is applied and indices of selected variables are returned.
}
\examples{
library(survival)
set.seed(1234)
temp <- generate_cure_data(N = 200, J = 25, nTrue = 5, A = 1.8)
training <- temp$Training
fit.cv <- cv_cureem(Surv(Time, Censor) ~ ., data = training,
                 x.latency = training, fdr.control = FALSE,
                 grid.tuning = FALSE, nlambda.inc = 10, nlambda.lat = 10,
                 n_folds = 2, seed = 23, verbose = TRUE)
fit.cv.fdr <- cv_cureem(Surv(Time, Censor) ~ ., data = training,
                 x.latency = training, model = "weibull", penalty = "lasso",
                 fdr.control = TRUE, grid.tuning = FALSE, nlambda.inc = 10,
                 nlambda.lat = 10, n_folds = 2, seed = 23, verbose = TRUE)
}
\references{
Archer, K. J., Fu, H., Mrozek, K., Nicolet, D., Mims, A. S., Uy, G. L., Stock, W., Byrd, J. C., Hiddemann, W., Braess, J., Spiekermann, K., Metzeler, K. H., Herold, T., Eisfeld, A.-K. (2024) Identifying long-term survivors and those at higher or lower risk of relapse among patients with cytogenetically normal acute myeloid leukemia using a high-dimensional mixture cure model. \emph{Journal of Hematology & Oncology}, \bold{17}:28.
}
\seealso{
\code{\link{cureem}}
}
\keyword{models}
\keyword{regression}
