% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/data-slice.R
\name{data_slice}
\alias{data_slice}
\alias{data_slice.default}
\alias{data_slice.data.frame}
\alias{data_slice.gam}
\alias{data_slice.gamm}
\alias{data_slice.list}
\alias{data_slice.scam}
\title{Prepare a data slice through model covariates}
\usage{
data_slice(object, ...)

\method{data_slice}{default}(object, ...)

\method{data_slice}{data.frame}(object, ..., .observed_only = FALSE)

\method{data_slice}{gam}(object, ..., data = NULL, envir = NULL, .observed_only = FALSE)

\method{data_slice}{gamm}(object, ...)

\method{data_slice}{list}(object, ...)

\method{data_slice}{scam}(object, ...)
}
\arguments{
\item{object}{an R model object.}

\item{...}{<\code{\link[rlang:dyn-dots]{dynamic-dots}}> User supplied variables
defining the data slice. Arguments passed via \code{...} need to be \emph{named}.}

\item{.observed_only}{logical or character; should the data slice be trimmed
to those combinations of the variables specified that are observed in
\code{object}. If \code{TRUE}, the observed combinations of variables mentioned in
\code{...} are matched against those in \code{object} and filtered to return only
those combinations. If \code{FALSE}, no filtering is done. If \code{.observed_only}
is a character vector, on those variables named in the vector are used to
in the comparison with the combinations in \code{object}.}

\item{data}{an alternative data frame of values containing all the variables
needed to fit the model. If \code{NULL}, the default, the data used to fit the
model will be recovered using \code{model.frame}. User-supplied expressions
passed in \code{...} will be evaluated in \code{data}.}

\item{envir}{the environment within which to recreate the data used to fit
\code{object}.}
}
\description{
Prepare a data slice through model covariates
}
\details{
A data slice is the data set that results where one (or more
covariates) is varied systematically over some or all of its (their) range or
at a specified subset of values of interest, while any remaining covariates
in the model are held at fixed, representative values. This is known as a
\emph{reference grid} in package \strong{emmeans} and a \emph{data grid} in the
\strong{marginaleffects} package.

For GAMs, any covariates not specified via \code{...} will take representative
values determined from the data used to fit the model as follows:
\itemize{
\item for numeric covariates, the value in the fitting data that is closest to
the median value is used,
\item for factor covariates, the modal (most frequently observed) level is used,
or the first level (sorted as per the vector returned by \code{\link[base:levels]{base::levels()}}
if several levels are observed the same number of times.
}

These values are already computed when calling \code{gam()} or \code{bam()} for example
and can be found in the \code{var.summary} component of the fitted model. Function
\code{\link[=typical_values]{typical_values()}} will extract these values for you if you are interested.

Convenience functions \code{\link[=evenly]{evenly()}}, \code{\link[=ref_level]{ref_level()}}, and \code{\link[=level]{level()}} are provided
to help users specify data slices. \code{\link[=ref_level]{ref_level()}}, and \code{\link[=level]{level()}} also ensure
that factor covariates have the correct levels, as needed by
\code{\link[mgcv:predict.gam]{mgcv::predict.gam()}} for example.

For an extended discussion of \code{\link[=data_slice]{data_slice()}} and further examples, see
\code{vignette("data-slices", package = "gratia")}.
}
\examples{
\dontshow{
op <- options(pillar.sigfig = 3)
}
load_mgcv()

# simulate some Gaussian data
df <- data_sim("eg1", n = 50, seed = 2)

# fit a GAM with 1 smooth and 1 linear term
m <- gam(y ~ s(x2, k = 7) + x1, data = df, method = "REML")

# Want to predict over f(x2) while holding `x1` at some value.
# Default will use the observation closest to the median for unspecified
# variables.
ds <- data_slice(m, x2 = evenly(x2, n = 50))
ds

# for full control, specify the values you want
ds <- data_slice(m, x2 = evenly(x2, n = 50), x1 = 0.3)

# or provide an expression (function call) which will be evaluated in the
# data frame passed to `data` or `model.frame(object)`
ds <- data_slice(m, x2 = evenly(x2, n = 50), x1 = mean(x1))
}
\seealso{
The convenience functions \code{\link[=evenly]{evenly()}}, \code{\link[=ref_level]{ref_level()}}, and \code{\link[=level]{level()}}.
\code{\link[=typical_values]{typical_values()}} for extracting the representative values used for
covariates in the model but not named in the slice.
}
