##' Multiple sequence alignment layer for ggplot2. It creats background frames with/without sequence labels.
##'
##' @title geom_msa
##' @param data multiple sequence alignment in tidy data frame, generated by tidy_fasta()
##' @param font font families, possible values are 'helvetical', 'mono', and 'DroidSansMono', 'TimesNewRoman'. Defaults is 'helvetical'. If font = NULL, only plot the background tile 
##' @param color A Color scheme. One of 'Clustal', 'Chemistry_AA', 'Shapely_AA', 'Zappo_AA', 'Taylor_AA', 'Chemistry_NT', 'Shapely_NT', 'Zappo_NT', 'Taylor_NT'. Defaults is 'Clustal'.
##' @param char_width characters width. Defaults is 0.9.
##' @param none_bg a logical value indicating whether backgroud should be produced.  Defaults is FALSE.
##' @param posHighligthed A numeric vector of the position that need to be highlighted.
##' @param seq_name a logical value indicating whether seqence names should be displayed.
##'  Defaults is 'NULL' which indicates that the sequence name is displayed when 'font = null', but 'font = char' will not be displayed.
##'  If 'seq_name = TRUE' the sequence name will be displayed in any case. If 'seq_name = FALSE' the sequence name will not be displayed under any circumstances.
##' @param ... additional parameter
##' @return A list
##' @importFrom utils modifyList
##' @export
##' @author Guangchuang Yu
geom_msa <- function(data, font = "helvetical", color = "Clustal", char_width = 0.9, none_bg = FALSE, posHighligthed = NULL, seq_name = NULL, ... ) {

    data <- msa_data(data, font = font, color = color, char_width = char_width )
    bg_data <- data
    
    mapping <- aes_(x = ~position, y = ~name, fill = ~I(color))
    
    if  (!isTRUE(seq_name)) {
        if ('y' %in% colnames(data) | isFALSE(seq_name) ) {
            y <- as.numeric(bg_data$name)
            mapping <- modifyList(mapping, aes_(y = ~y))
        } 
    }

    if (!is.null(posHighligthed)) {
        none_bg = TRUE
        bg_data <- bg_data[bg_data$position %in% posHighligthed,]
        bg_data$postion <- as.factor(bg_data$position)
        mapping <- modifyList(mapping, aes_(x = ~position, fill = ~character, width = 1))
    } 
    
    ly_bg <- geom_tile(mapping = mapping, data = bg_data, color = 'grey', inherit.aes = FALSE)

    if (!all(c("yy", "order", "group") %in% colnames(data))) {
        return(ly_bg)
    }

    #data <- data[order(data$order),]

    if ('y' %in% colnames(data)) {
        data$yy = data$yy - as.numeric(data$name) + data$y
    }

    ly_label <- geom_polygon(aes_(x = ~x, y = ~yy,  group = ~group ),
                             data = data, inherit.aes = FALSE)
    
    if (none_bg & is.null(posHighligthed) ) {
        return(ly_label)
    }
    
    list(ly_bg, ly_label)

}

