% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/binning_methods.R
\name{bin_dots}
\alias{bin_dots}
\title{Bin data values using a dotplot algorithm}
\usage{
bin_dots(
  x,
  y,
  binwidth,
  heightratio = 1,
  layout = c("bin", "weave", "swarm"),
  side = c("topright", "top", "right", "bottomleft", "bottom", "left", "topleft",
    "bottomright", "both"),
  orientation = c("horizontal", "vertical", "y", "x")
)
}
\arguments{
\item{x}{numeric vector of x values}

\item{y}{numeric vector of y values}

\item{binwidth}{bin width}

\item{heightratio}{ratio of bin width to dot height}

\item{layout}{The layout method used for the dots:
\itemize{
\item \code{"bin"} (default): places dots on the off-axis at the midpoint of their bins as in the classic Wilkinson dotplot.
This maintains the alignment of rows and columns in the dotplot. This layout is slightly different from the
classic Wilkinson algorithm in that: (1) it nudges bins slightly to avoid overlapping bins and (2) if
the input data are symmetrical it will return a symmetrical layout.
\item \code{"weave"}: uses the same basic binning approach of \code{"bin"}, but places dots in the off-axis at their actual
positions (modulo overlaps, which are nudged out of the way). This maintains the alignment of rows but does not
align dots within columns. Does not work well when \code{side = "both"}.
\item \code{"swarm"}: uses the \code{"compactswarm"} layout from \code{beeswarm::beeswarm()}. Does not maintain alignment of rows or
columns, but can be more compact and neat looking, especially for sample data (as opposed to quantile
dotplots of theoretical distributions, which may look better with \code{"bin"} or \code{"weave"}).
}}

\item{side}{Which side to place the slab on. \code{"topright"}, \code{"top"}, and \code{"right"} are synonyms
which cause the slab to be drawn on the top or the right depending on if \code{orientation} is \code{"horizontal"}
or \code{"vertical"}. \code{"bottomleft"}, \code{"bottom"}, and \code{"left"} are synonyms which cause the slab
to be drawn on the bottom or the left depending on if \code{orientation} is \code{"horizontal"} or
\code{"vertical"}. \code{"topleft"} causes the slab to be drawn on the top or the left, and \code{"bottomright"}
causes the slab to be drawn on the bottom or the right. \code{"both"} draws the slab mirrored on both
sides (as in a violin plot).}

\item{orientation}{Whether the dots are laid out horizontally or vertically.
Follows the naming scheme of \code{geom_slabinterval()}:
\itemize{
\item \code{"horizontal"} assumes the data values for the dotplot are in the \code{x}
variable and that dots will be stacked up in the \code{y} direction.
\item \code{"vertical"} assumes the data values for the dotplot are in the \code{y}
variable and that dots will be stacked up in the \code{x} direction.
}

For compatibility with the base ggplot naming scheme for \code{orientation},
\code{"x"} can be used as an alias for \code{"vertical"} and \code{"y"} as an alias for
\code{"horizontal"}.}
}
\value{
A \code{data.frame} with three columns:
\itemize{
\item \code{x}: the x position of each dot
\item \code{y}: the y position of each dot
\item \code{bin}: a unique number associated with each bin
(supplied but not used when \code{layout = "swarm"})
}
}
\description{
Bins the provided data values using one of several dotplot algorithms.
}
\examples{

library(dplyr)
library(ggplot2)

x = qnorm(ppoints(20))
bin_df = bin_dots(x = x, y = 0, binwidth = 0.5, heightratio = 1)
bin_df

# we can manually plot the binning above, though this is only recommended
# if you are using find_dotplot_binwidth() and bin_dots() to build your own
# grob. For practical use it is much easier to use geom_dots(), which will
# automatically select good bin widths for you (and which uses
# find_dotplot_binwidth() and bin_dots() internally)
bin_df \%>\%
  ggplot(aes(x = x, y = y)) +
  geom_point(size = 4) +
  coord_fixed()

}
\seealso{
\code{\link[=find_dotplot_binwidth]{find_dotplot_binwidth()}} for an algorithm that finds good bin widths
to use with this function; \code{\link[=geom_dotsinterval]{geom_dotsinterval()}} for geometries that use
these algorithms to create dotplots.
}
