% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ccf_boot.R
\name{ccf_boot}
\alias{ccf_boot}
\title{Cross-Correlation Function of Time Series with Sieve Bootstrap p-values}
\usage{
ccf_boot(
  x,
  y,
  lag.max = NULL,
  plot = c("Pearson", "Spearman", "none"),
  level = 0.95,
  B = 1000,
  ...
)
}
\arguments{
\item{x, y}{univariate numeric time-series objects or numeric vectors for which to
compute cross-correlation. Different time attributes in \code{ts} objects are
acknowledged, see Example 2 below.}

\item{lag.max}{maximum lag at which to calculate the cross-correlation. Will be
automatically limited as in \code{\link[stats]{ccf}}.}

\item{plot}{choose whether to plot results for Pearson correlation (default, or use
\code{plot = "Pearson"}), Spearman correlation (use \code{plot = "Spearman"}), or
suppress plotting (use \code{plot = "none"}). Both Pearson's and Spearman's results are
given in the output, regardless of the \code{plot} setting.}

\item{level}{confidence level, from 0 to 1. Default is 0.95, that is, 95\% confidence.}

\item{B}{number of bootstrap simulations to obtain empirical critical values.
Default is 1000.}

\item{...}{other parameters passed to the function \code{\link{ARest}} to control
how autoregressive dependencies are estimated. The same set of parameters is used
separately on \code{x} and \code{y}.}
}
\value{
A data frame with the following columns:
\item{Lag}{lags for which the following values were obtained.}
\item{rP}{observed Pearson correlations.}
\item{pP}{bootstrap p-value for Pearson correlations.}
\item{lowerP, upperP}{lower and upper quantiles (for the confidence level set by \code{level}) of the bootstrapped Pearson correlations.}
\item{rS}{observed Spearman correlations.}
\item{pS}{bootstrap p-value for Spearman correlations.}
\item{lowerS, upperS}{lower and upper quantiles (for the confidence level set by \code{level}) of the bootstrapped Spearman correlations.}
}
\description{
Account for possible autocorrelation of time series when assessing the statistical significance
of their cross-correlation. A sieve bootstrap approach is used to generate multiple copies
of the time series with the same autoregressive dependence, under the null hypothesis of the
two time series under investigation being uncorrelated. The significance of cross-correlation
coefficients is assessed based on the distribution of their bootstrapped counterparts.
Both Pearson and Spearman types of coefficients are obtained, but a plot is provided for
only one type, with significant correlations shown using filled circles.
}
\examples{
\dontrun{
# Fix seed for reproducible simulations:
set.seed(1)

# Example 1
# Simulate independent normal time series of same lengths
x <- rnorm(100)
y <- rnorm(100)
ccf(x, y) # default CCF with parametric confidence band
tmp <- ccf_boot(x, y) # CCF with bootstrap
tmp$rP; tmp$rS # can always extract results for both Pearson and Spearman correlations

# Example 2
# Simulated ts objects of different lengths and starts (incomplete overlap)
x <- arima.sim(list(order = c(1, 0, 0), ar = 0.5), n = 30)
x <- ts(x, start = 2001)
y <- arima.sim(list(order = c(2, 0, 0), ar = c(0.5, 0.2)), n = 40)
y <- ts(y, start = 2020)
ts.plot(x, y, col = 1:2, lty = 1:2) # show how x and y are aligned
ccf(x, y)
ccf_boot(x, y, plot = "Spearman") # CCF with bootstrap
# Notice that only +-7 lags can be calculated in both cases because of the small 
# overlap of the time series. If save these time series as plain vectors, the time
# information would be lost, and time series will be misaligned. 
ccf(as.numeric(x), as.numeric(y))

# Example 3
# Box & Jenkins time series of sales and a leading indicator, see ?BJsales
plot.ts(cbind(BJsales.lead, BJsales))
# Each of the BJ time series looks as having a stochastic linear trend, so apply differences:
plot.ts(cbind(diff(BJsales.lead), diff(BJsales)))
# Get cross-correlation of the differenced series:
ccf_boot(diff(BJsales.lead), diff(BJsales), plot = "Spearman")
# The leading indicator "stands out" with significant correlations at negative lags, 
# showing it can be used to predict the sales 2-3 time steps ahead (that is,
# diff(BJsales.lead) at times t-2 and t-3 is strongly correlated with diff(BJsales) at 
# current time t).
}

}
\seealso{
\code{\link{ARest}}, \code{\link[stats]{ar}}, \code{\link[stats]{ccf}},
\code{\link{HVK}}
}
\author{
Vyacheslav Lyubchich
}
\keyword{ts}
