% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/meta.R
\name{fold.data.frame}
\alias{fold.data.frame}
\title{Fold a Data Frame}
\usage{
\method{fold}{data.frame}(x, ..., meta = obj_attr(x), simplify = TRUE,
  sort = TRUE, tol = 10)
}
\arguments{
\item{x}{data.frame}

\item{...}{unquoted names of grouping columns. See also \code{\link{fold.grouped_df}}.  Alternatively, pre-specify as a groups attribute (character vector).}

\item{meta}{a list of formulas in the form object ~ attribute. Pass something with length 0 to suppress guessing.}

\item{simplify}{set to NA any groups values that do not help distinguish values, and remove resulting duplicate records}

\item{sort}{whether to sort the result}

\item{tol}{maximum number of categories for guessing whether to encode metadata; encoding will always be attempted if metadata (attr) or its referent (obj) is a factor}
}
\value{
folded data.frame with columns VARIABLE, META, VALUE and any supplied grouping items.
}
\description{
Folds a data.frame. Stacks columns, while isolating metadata and capturing keys.
}
\details{
See \code{package?fold} for micro-vignette.

A folded data.frame is formalized re-presentation of a conventional data.frame.  Items in the conventional form are of three conceptual types: data, metadata, and keys.  Data items contain the primary values, to be described.  Metadata gives additional details about the data items or values. Keys are grouping items; combinations of grouping values should uniquely identify each conventional record.

In the result, names of data items appear in VARIABLE, while values of data items are stacked in VALUE. Data items are all columns from the input not otherwise identified as metadata or keys.

Likewise, names of metatdata items appear in META, while the name of the described data item appears in VARIABLE.  Values of metadata items appear in VALUE.  The metadata VALUE will be an encoding (see package: encode) if there is exactly one unique metadata value corresponding to each unique data value, AND one of the two is a factor (or neither factor, but there are \code{tol} or fewer unique values of data, and multiple unique values of metadata). Metadata items are identified explicitly using a list of formulas, or implicitly by means of column naming conventions.

Grouping items that are present in the input persist in the result and serve as keys.  Both data and metadata values may have keys, but neither require them.  Keys are identified explicitly by supplying unnamed, unquoted arguments (non-standard evaluation).  Use \code{dplyr::group_by} to add groups that will be respected when \code{\link{fold.grouped_df}} (or generic) is called. Or supply a \code{groups} attribute to the data.frame, e.g. \code{attr(x,'groups') <- c('USUBJID','TIME')}.

By default, superflous keys (those that do not help distinguish data items) are removed on a per-data-item basis. Column order is used to resolve ambiguities: checking proceeds right to left, preferentially discarding keys to the right.

Note that metadata items may describe other metadata items, recursively.  In practice, however, such complexity could be problematic and is best avoided if possible.

There are cases where a real grouping item may appear superfluous, e.g. for a one-record dataset.  Enforce the groups by setting \code{simplify} to FALSE.

The folded format supports mixed object types, as inferred from differences in relevant grouping items on a per record basis.  Mixed typing works best when object types form a nested hierarchy, i.e. all keys are left-subsets of the full key. Thus the order of grouping values is considered informative, e.g. for sorting.
}
\examples{
library(magrittr)
library(dplyr)
data(events)
x <- events
x \%<>\% filter(CMT == 2) \%>\% select(-EVID,-CMT,-AMT)
x \%>\% fold(USUBJID,TIME)
x \%>\% fold(USUBJID,TIME, meta = list(DV ~ BLQ, DV ~ LLOQ))
x <- events \%>\% 
  filter(CMT == 2) \%>\% 
  select(ID, TIME, TAD, DV, BLQ, LLOQ, SEX) 
x
attr(x,'groups') <- c('ID','TIME')

# less than 10 values of DV, so BLQ looks like an encoding
y <- x  \%>\% fold(meta=list(DV~BLQ,BLQ~LLOQ))
y \%>\% data.frame

# reducing the tolerance forces BLQ to match by groups (ID, TIME) instead of DV value
z <- x  \%>\% fold(meta=list(DV~BLQ,BLQ~LLOQ),tol=3)
z
\dontshow{
x <- events
x \%<>\% filter(CMT == 2) \%>\% select(-EVID,-CMT,-AMT)
attr(x,'groups') <- c('USUBJID','TIME')
y <- x \%>\% select(C, ID, TIME:PRED,USUBJID)
stopifnot(identical(fold(x),fold(y)))
}
}
\seealso{
\code{\link{obj_attr.data.frame}}  
\code{\link{fold}} 
\code{\link{print.folded}} 
\code{\link{simplify.folded}} 
\code{\link{sort.folded}} 
\code{\link{unfold.folded}}
}
