% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitcond.R, R/fitplc.R, R/fitplcs.R
\name{fitcond}
\alias{fitcond}
\alias{fitconds}
\alias{fitplc}
\alias{fitplcs}
\title{Fit a PLC curve}
\usage{
fitcond(dfr, varnames = c(K = "K", WP = "MPa"), Kmax = NULL,
  WP_Kmax = NULL, ...)

fitplc(dfr, varnames = c(PLC = "PLC", WP = "MPa"), weights = NULL,
  random = NULL, model = "Weibull", startvalues = list(Px = 3, S = 20),
  x = 50, bootci = TRUE, nboot = 999, quiet = FALSE, ...)

fitplcs(dfr, group, ...)

fitconds(dfr, group, ...)
}
\arguments{
\item{dfr}{A dataframe that contains water potential and plc or conductivity/conductance data.}

\item{varnames}{A vector specifying the names of the PLC and water potential data (WP).}

\item{Kmax}{Maximum conduct(ance)(ivity), optional (and only when using \code{fitcond}). See Examples.}

\item{WP_Kmax}{Water potential above which Kmax will be calculated from the data. Optional (and only when using \code{fitcond}). See Examples.}

\item{weights}{A variable used as weights in weighted non-linear regression that must be present in the dataframe (unquoted, see examples).}

\item{random}{Variable that specified random effects (unquoted; must be present in dfr).}

\item{model}{At the moment, only 'Weibull' is allowed.}

\item{startvalues}{A list of starting values. If set to NULL, \code{fitplc} will attempt to guess starting values.}

\item{x}{If the P50 is to be returned, x = 50. Set this value if other points of the PLC curve should be estimated (although probably more robustly done via \code{\link{getPx}}).}

\item{bootci}{If TRUE, also computes the bootstrap confidence interval.}

\item{nboot}{The number of bootstrap replicates (only relevant when \code{bootci=TRUE}).}

\item{quiet}{Logical (default FALSE), if TRUE, don't print any messages.}

\item{group}{Character; variable in the dataframe that specifies groups. The curve will be fit for every group level.}

\item{\dots}{Further parameters passed to \code{fitplc}.}
}
\description{
This function fits the Weibull curve to measurements of stem or leaf conductivity 
measurements at various water potentials. If measurements are organized as 'percent loss conductivity' (PLC), use the \code{fitplc} function. If they are organized as the actual conductance or conductivity (as is common for leaf hydraulic  conductance data, for example), use the \code{fitcond} function. See Details and Examples for more information on how to use these functions. 

It is also possible to fit multiple curves at once, for example one for each species or site, 
with the \code{fitplcs} and \code{fitconds} functions.

See \code{\link{plot.plcfit}} for documentation on plotting methods for the fitted objects, and the examples below.
}
\details{
If a variable with the name Weights is present in the dataframe, 
this variable will be used as the \code{weights} argument in \code{\link{nls}} to perform 
weighted non-linear regression. See the final example on how to use this.

If the \code{random} argument specifies a factor variable present in the dataframe, random effects will 
be estimated both for SX and PX. This affects \code{coef} as well as the confidence intervals for the fixed effects.

A plot method is available for the fitted object, see examples on how to use it.
}
\examples{

# We use the built-in example dataset 'stemvul' in the examples below. See ?stemvul.
  
# 1. Fit one species (or fit all, see next example)
dfr1 <- subset(stemvul, Species =="dpap")

# Make fit. Store results in object 'pfit'
# 'varnames' specifies the names of the 'PLC' variable in the dataframe,
# and water potential (WP). 
# In this example, we use only 50 bootstrap replicates but recommend you set this
# to 1000 or so.
pfit <- fitplc(dfr1, varnames=c(PLC="PLC", WP="MPa"), nboot=50)

# Look at fit
pfit

# Make a standard plot. The default plot is 'relative conductivity',
# (which is 1.0 where PLC = 0). For plotting options, see ?plot.plcfit
plot(pfit)

# Or plot the percent embolism
plot(pfit, what="embol")

# Get the coefficients of the fit.
coef(pfit)

# 2. Fit all species in the dataset.
# Here we also set the starting values (which is sometimes needed).
# In this example, we use only 50 bootstrap replicates but recommend you set this
# to 1000 or so. 
allfit <- fitplcs(stemvul, "Species", varnames=c(PLC="PLC", WP="MPa"), nboot=50)

# 3. Plot the fits.
plot(allfit, onepanel=TRUE, plotci=FALSE, selines="none", pxlinecol="dimgrey")

# Coefficients show the estimates and 95\% CI (given by 'lower' and 'upper')
# Based on the CI's, species differences can be decided.
coef(allfit)

# 3. Specify Weights. The default variable name is Weights, if present in the dataset
# it will be used for weighted non-linear regression
# In this example, we use only 50 bootstrap replicates but recommend you set this
# to 1000 or so. 
dfr1$Weights <- abs(50-dfr1$PLC)^1.2
pfit <- fitplc(dfr1, varnames=c(PLC="PLC", WP="MPa"), weights=Weights, nboot=50)
coef(pfit)

# 4. Fit the Weibull curve directly to the raw conductance data. 
# Use this option when you don't want to transform your data to PLC. 
# You have two options: specify the 'maximum' conductance yourself (and provide Kmax), 
# or set the threshold water potential (Kmax_WP), which is then used to calculate Kmax
# (from the average of the conductance values where WP > Kmax_WP).

# Option 1 : maximum conductivity (i.e. at full hydration) is known, and used as input.
kfit1 <- fitcond(dfr1, varnames=c(K="Cond", WP="MPa"), Kmax=7.2, nboot=50)

# Option 2 : calculate maximum cond. from data where water potential : -0.3 MPa.
# In this example, we use only 50 bootstrap replicates but recommend you set this
# to 1000 or so. 
kfit2 <- fitcond(dfr1, varnames=c(K="Cond", WP="MPa"), WP_Kmax = -0.3, nboot=50)
# Use plot(kfit1) as for fitplc, as well as coef() etc.

# Fit multiple conductivity curves at once (bootstrap omitted for speed).
kfits3 <- fitconds(stemvul, "Species", varnames=list(K="Cond", WP="MPa"), WP_Kmax=-0.3, boot=FALSE)
plot(kfits3, onepanel=TRUE, ylim=c(0,12), selines="none")

# 5. Random effects.
# This example takes into account the fact that the individual data points for a species are not 
# independent, but rather clustered by branch. 
fitr <- fitplc(dfr1, random=Branch)

# Visualize the random effects.
plot(fitr, plotrandom=TRUE)
}

