#############################################################################
#   Copyright (c) 2009 Marie Laure Delignette-Muller                                                                                                  
#                                                                                                                                                                        
#   This program is free software; you can redistribute it and/or modify                                               
#   it under the terms of the GNU General Public License as published by                                         
#   the Free Software Foundation; either version 2 of the License, or                                                   
#   (at your option) any later version.                                                                                                            
#                                                                                                                                                                         
#   This program is distributed in the hope that it will be useful,                                                             
#   but WITHOUT ANY WARRANTY; without even the implied warranty of                                          
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                                 
#   GNU General Public License for more details.                                                                                    
#                                                                                                                                                                         
#   You should have received a copy of the GNU General Public License                                           
#   along with this program; if not, write to the                                                                                           
#   Free Software Foundation, Inc.,                                                                                                              
#   59 Temple Place, Suite 330, Boston, MA 02111-1307, USA                                                             
#                                                                                                                                                                         
#############################################################################
### plot functions for non-censored data
###
###         R functions
### 

plotdist <- function(data, distr, para, breaks="default", discrete=FALSE, ...){
    def.par <- par(no.readonly = TRUE)
    if (missing(data) || !is.vector(data, mode="numeric"))
        stop("data must be a numeric vector")
    if ((missing(distr) & !missing(para)) || 
    (missing(distr) & !missing(para)))
        stop("distr and para must defined")
    xlim <- c(min(data), max(data)) # for plot of discrete distributions
    if (missing(distr)) { ## Plot of data only
        par(mfrow=c(1, 2))
        s <- sort(data)
        n <- length(data)
        if (!discrete) {
            # plot for continuous data
            obsp <- ppoints(s)
            if (breaks=="default") 
                h <- hist(data, freq=FALSE, xlab="Data", main=paste("Histogram"), ...)
            else 
                h <- hist(data, freq=FALSE, xlab="Data", main=paste("Histogram"), breaks=breaks, ...)
            plot(s, obsp, main=paste("Cumulative distribution"), xlab="Data", 
            xlim=c(h$breaks[1], h$breaks[length(h$breaks)]), ylab="CDF", ...)
        }
        else {
            # plot for discrete data
            if (breaks!="default") 
            warning("Breaks are not taken into account for discrete data")
            # plot of empirical distribution
            t <- table(data)
            xval <- as.numeric(names(t))
            xvalfin <- seq(min(xval), max(xval))
            ydobs <- as.vector(t)/n
            ydmax <- max(ydobs)
            plot(xval, ydobs, type='h', xlim=xlim, ylim=c(0, ydmax), 
            main="Empirical distribution", xlab="Data", ylab="Density", ...)
            # plot of the cumulative probability distributions
            ycdfobs <- ecdf(data)(xvalfin)
            plot(xvalfin, ycdfobs, type='h', xlim=xlim, ylim=c(0, 1), 
            main="Empirical CDFs", xlab="Data", ylab="CDF", ...)
        }
    } #end of if (missing(distr))
    else {
        if (!is.character(distr)) distname <- substring(as.character(match.call()$distr), 2)
            else distname <- distr
        if (!missing(discrete))
        warning("the argument discrete is not taken into account when distr is defined")
        if (!is.list(para)) 
        stop("'para' must be a named list")
        ddistname <- paste("d", distname, sep="")
        if (!exists(ddistname, mode="function"))
            stop(paste("The ", ddistname, " function must be defined"))
        pdistname <- paste("p", distname, sep="")
        if (!exists(pdistname, mode="function"))
            stop(paste("The ", pdistname, " function must be defined"))
        qdistname <- paste("q", distname, sep="")
        if (!exists(qdistname, mode="function"))
            stop(paste("The ", qdistname, " function must be defined"))
        densfun <- get(ddistname, mode="function")    
        nm <- names(para)
        f <- formals(densfun)
        args <- names(f)
        m <- match(nm, args)
        if (any(is.na(m))) 
            stop(paste("'para' specifies names which are not arguments to ", ddistname))

        n <- length(data) 
        if (is.element(distname, c("binom", "nbinom", "geom", "hyper", "pois"))) 
            discrete <- TRUE
        else
            discrete <- FALSE
        if (!discrete) {
        # plot of continuous data with theoretical distribution
            par(mfrow=c(2, 2))
            s <- sort(data)
            obsp <- ppoints(s)
            theop <- do.call(pdistname, c(list(q=s), as.list(para)))
            # plot of the histogram with theoretical density
            # computes densities in order to define limits for y-axis
            if (breaks=="default")
                h <- hist(data, plot=FALSE)
            else
                h <- hist(data, breaks=breaks, plot=FALSE, ...)           
            xhist <- seq(min(h$breaks), max(h$breaks), length=1000)
            yhist <- do.call(ddistname, c(list(x=xhist), as.list(para)))
            ymax <- ifelse(is.finite(max(yhist)), max(max(h$density), max(yhist)), max(h$density)) 
            # plot of histograms and theoretical density
            hist(data, freq=FALSE, xlab="Data", ylim=c(0, ymax), breaks=h$breaks, 
                main=paste("Empirical and theoretical distr."), ...)
            lines(xhist, yhist,lty=1,col="red")
           
            # plot of the qqplot
            theoq <- do.call(qdistname, c(list(p=obsp), as.list(para)))
            plot(theoq, s, main=" Q-Q plot", xlab="Theoretical quantiles", 
            ylab="Empirical quantiles", ...)
            abline(0, 1)
            # plot of the cumulative probability distributions
            xmin <- h$breaks[1]
            xmax <- h$breaks[length(h$breaks)]
            plot(s, obsp, main=paste("Empirical and theoretical CDFs"), xlab="Data", 
            ylab="CDF", xlim=c(xmin, xmax), ...)
            sfin <- seq(xmin, xmax, by=(xmax-xmin)/100)
            theopfin <- do.call(pdistname, c(list(q=sfin), as.list(para)))
            lines(sfin, theopfin, lty=1,col="red")
            
            # plot of the ppplot
            plot(theop, obsp, main="P-P plot", xlab="Theoretical probabilities", 
            ylab="Empirical probabilities", ...)
            abline(0, 1)
        }
        else {
        # plot of discrete data with theoretical distribution
            par(mfrow=c(1, 2))
            if (breaks!="default") 
            warning("Breaks are not taken into account for discrete distributions")
            # plot of empirical and theoretical distributions
            t <- table(data)
            xval <- as.numeric(names(t))
            xvalfin <- seq(min(xval), max(xval))
            xlinesdec <- min((max(xval)-min(xval))/30, 0.4)
            yd <- do.call(ddistname, c(list(x=xvalfin), as.list(para)))
            ydobs <- as.vector(t)/n
            ydmax <- max(yd, ydobs)
            plot(xvalfin+xlinesdec, yd, type='h', xlim=c(min(xval), max(xval)+xlinesdec), 
                ylim=c(0, ydmax), lty=3, col="red",
                main="Emp. and theo. distr.", xlab="Data", 
                ylab="Density", ...)
            points(xval, ydobs, type='h', lty=1, col="black",...)
            legend("topright", lty=c(1, 3), col=c("black","red"),
                legend=c("empirical", paste("theoretical")), 
                bty="o", bg="white",cex=0.6,...)
            
            # plot of the cumulative probability distributions
            ycdfobs <- ecdf(data)(xvalfin)
            ycdf <- do.call(pdistname, c(list(q=xvalfin), as.list(para)))
            plot(xvalfin+xlinesdec, ycdf, type='h', xlim=c(min(xval), max(xval)+xlinesdec), 
                ylim=c(0, 1), lty=3, col="red", 
                main="Emp. and theo. CDFs", xlab="Data", 
                ylab="CDF", ...)
            points(xvalfin, ycdfobs, type='h', lty=1, col="black",...)
            legend("bottomright", lty=c(1, 3), col=c("black","red"), legend=c("empirical", paste("theoretical")), 
             bty="o", bg ="white",cex=0.6,...)
        }
    }
    par(def.par)    
}
