% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aux_hvanalysis.R
\name{aux_hvanalysis}
\alias{aux_hvanalysis}
\title{Perform H-V-spectral ratio analysis of a seismic data set}
\usage{
aux_hvanalysis(
  data,
  time,
  window,
  overlap = 0,
  dt,
  method = "periodogram",
  kernel,
  order = "xyz",
  plot = FALSE,
  ...
)
}
\arguments{
\item{data}{\code{List}, \code{data frame} or \code{matrix}, seismic
componenents to be processed. If \code{data} is a matrix, the components 
must be organised as columns. Also, \code{data} can be a list of 
\code{eseis} objects.}

\item{time}{\code{POSIXct} vector with time values. If omitted, an 
synthetic time vector will be created, based on \code{dt}.}

\item{window}{\code{Numeric} scalar, time window length in seconds used to 
calculate individual spectral ratios. Set to 10 percent of the time  
series length by default.}

\item{overlap}{\code{Numeric} value, fraction of window overlap.}

\item{dt}{\code{Numeric} value, sampling period.}

\item{method}{\code{Character} value, method for calculating the spectra. 
One out of \code{"periodogram"} , \code{"autoregressive"} and 
\code{"multitaper"}, default is \code{"periodogram"}.}

\item{kernel}{\code{Numeric} value, window size (defined by number of 
samples) of the moving window used for smoothing the spectra. By default 
no smoothing is performed.}

\item{order}{\code{Character} value, order of the seismic components. 
Describtion must contain the letters \code{"x"},\code{"y"} and
\code{"z"} in the order according to the input data set. Default is 
\code{"xyz"} (NW-SE-vertical).}

\item{plot}{\code{Logical} value, toggle plot output. Default is
\code{FALSE}.}

\item{\dots}{Additional arguments passed to the plot function.}
}
\value{
A \code{matrix} with the h-v-frequency ratios for each time slice.
}
\description{
This function cuts a three component seismic data set into time windows
that may or may not overlap and calculates the spectral ratio for each of 
these windows. It returns a matrix with the ratios for each time slice. 
Thus, it is a wrapper for the function \code{signal_hvratio}. For 
further information about the technique and function arguments see the 
description of \code{signal_hvratio}.
}
\examples{
## load example data set
data(earthquake)

## ATTENTION, THIS EXAMPLE DATA SET IS FAR FROM IDEAL FOR THIS PURPOSE

## detrend data
s <- signal_detrend(data = s)

## calculate the HV ratios straightforward
HV <- aux_hvanalysis(data = s,
                     dt = 1 / 200,
                     kernel = 100)

## calculate the HV ratios with plot output, userdefined palette
plot_col <- colorRampPalette(colors = c("grey", "darkblue", "blue", "orange"))
HV <- aux_hvanalysis(data = s,
                     dt = 1 / 200,
                     kernel = 100,
                     plot = TRUE,
                     col = plot_col(100))

## calculate the HV ratios with optimised method settings
HV <- aux_hvanalysis(data = s, 
                     time = t,
                     dt = 1 / 200, 
                     window = 10, 
                     overlap = 0.9, 
                     method = "autoregressive",
                     plot = TRUE,
                     col = plot_col(100),
                     xlab = "Time (UTC)",
                     ylab = "f (Hz)")
                     
## calculate and plot stack (mean and sd) of all spectral ratios
HV_mean <- apply(X = HV, MARGIN = 1, FUN = mean)
HV_sd <- apply(X = HV, MARGIN = 1, FUN = sd)
HV_f <- as.numeric(rownames(HV))

plot(x = HV_f, y = HV_mean, type = "l", ylim = c(0, 50))
lines(x = HV_f, y = HV_mean + HV_sd, col = 2)
lines(x = HV_f, y = HV_mean - HV_sd, col = 2)              

}
\author{
Michael Dietze
}
\keyword{eseis}
