\name{plot.editmatrix}
\alias{plot.editarray}
\alias{plot.editmatrix}
\alias{plot.editset}
\title{Graphical representation of edits}
\usage{
  \method{plot}{editmatrix} (x, nodetype = "all",
    rules = editnames(x), vars = getVars(x),
    violated = logical(nedits(x)),
    adapt = logical(length(getVars(x))), nabbreviate = 5,
    layout = layout.fruchterman.reingold,
    edgecolor = "steelblue", rulecolor = "khaki1",
    varcolor = "lightblue1", violatedcolor = "sienna1",
    adaptcolor = "sienna1", ...)

  \method{plot}{editarray} (x, nodetype = "all",
    rules = editnames(x), vars = getVars(x),
    violated = logical(nedits(x)),
    adapt = logical(length(getVars(x))), nabbreviate = 5,
    layout = layout.fruchterman.reingold,
    edgecolor = "steelblue", rulecolor = "khaki1",
    varcolor = "lightblue1", violatedcolor = "sienna1",
    adaptcolor = "sienna1", ...)

  \method{plot}{editset} (x, nodetype = "all",
    rules = editnames(x), vars = getVars(x),
    violated = logical(nedits(x)),
    adapt = logical(length(getVars(x))), nabbreviate = 5,
    layout = layout.fruchterman.reingold,
    edgecolor = "steelblue", rulecolor = "khaki1",
    varcolor = "lightblue1", violatedcolor = "sienna1",
    adaptcolor = "sienna1", ...)
}
\arguments{
  \item{x}{object of class \code{\link{editmatrix}}}

  \item{nodetype}{\code{'rules'}, \code{'vars'} or
  \code{'all'}.}

  \item{rules}{selection of edits}

  \item{vars}{selection of variables}

  \item{violated}{A named \code{logical} vector of
  \code{nrow(E)}. Ingnored when \code{nodetype='vars'}}

  \item{adapt}{A named \code{logical} vector of
  \code{length(getVars(E))}. Ignored when
  \code{nodetype='rules'}}

  \item{nabbreviate}{\code{integer} To how many characters
  should variable and edit names be abbreviated?}

  \item{layout}{an \code{igraph} layout function. See
  \code{?igraph::layout}}

  \item{edgecolor}{Color of edges and node frames}

  \item{rulecolor}{Color of rule nodes (ignored when
  \code{nodetype='vars'})}

  \item{varcolor}{Color of variable nodes (ignored when
  \code{nodetype='rules'})}

  \item{violatedcolor}{Color of nodes corresponding to
  violated edits (ignored when \code{nodetype='vars'})}

  \item{adaptcolor}{Color of nodes corresponding to
  variables to adapt (ignored when
  \code{nodetype='rules'})}

  \item{...}{further arguments to be passed to plot.}
}
\description{
  Plots a graph, showing which variables occur in what
  edits. By default, squares represent edits, circles
  represent variables and an edge connecing a variable with
  an edit indicates that the edit contains the variable.

  plot method for editarray

  plot method for editset
}
\details{
  Depending on the chosen \code{nodetype}, this function
  can plot three types of graphs based on an edit set.

  \itemize{ \item{If \code{nodetype="all"} (default), the
  full bipartite graph is plotted. Each variable is
  represented by a square node while each edit is
  represented by a circular node. An edge is drawn when a
  variable occurs in an edit.}

  \item{If \code{nodetype="vars"} the variable graph is
  drawn. Each node represents a variable, and an edge is
  drawn between two nodes if the variables occur together
  in at least one edit. The edge width relates to the
  number of edits connecting two variables.}

  \item{If \code{nodetype="rules"} the rule graph is drawn.
  Each node represents an edit rule and an edge is drawn
  between two nodes if they share at least one variable.
  The edge width relates to the number of edits connecting
  the two edit rules.} }

  The boolean vectors \code{violated} and \code{adapt} can
  be used to color violated edits or variables which have
  to be adapted. The vectors must have named elements, so
  variables and edit names can be matched.

  The function works by coercing an editmatrix to an
  \code{igraph} object, and therefore relies on the
  plotting capabilities of the igraph package. For more
  finetuning, use \code{\link{as.igraph}} and see
  \code{?igraph.plotting}.

  The default layout generated by the Fruchterman-Reingold
  algorithm. The resulting layout is one of several optimal
  layouts, generated randomly (using a attration-repulsion
  model between the nodes). To reproduce layouts, use fix a
  randseed before calling the plot function.
}
\examples{

## Examples with linear (in)equality edits

# load predefined edits from package
data(edits)
edits

# convert to editmatrix
E <- editmatrix(edits)

## Not run:
# (Note to reader: the Not run directive only prevents the examle commands from running when package is built)

# Total edit graph
plot(E)

# Graph with dependent edits
plot(E, nodetype="rules")

# Graph with dependent variables
plot(E, nodetype="vars")

# Total edit graph, but with curved lines (option from igraph package)
plot(E, edge.curved=TRUE)


# graph, plotting just the connections caused by variable 't'
plot(E,vars='t')

## End(Not run) 

# here's an example with a broken record.
r <- c(ct = 100, ch = 30, cp = 70, p=30,t=130 ) 
violatedEdits(E,r)
errorLocalizer(E,r)$searchBest()$adapt

# we color the violated edits and the variables that have to be adapted

## Not run
set.seed(1) # (for reprodicibility)
plot(E,
     adapt=errorLocalizer(E,r)$searchBest()$adapt,   
     violated=violatedEdits(E,r))
## End(Not run) 



# extract total graph (as igraph object)
as.igraph(E)

# extract graph with edges related to variable 't' and 'ch'
as.igraph(E,vars=c('t','ch'))

# extract total adjacency matrix
adjacency(E)

# extract adjacency matrix related to variables t and 'ch'
adjacency(E,vars=c('t','ch'))

## Examples with categorical edits

# generate an editarray:
E <- editarray(c(
    "age \\\%in\\\% c('<15','16-65','>65')",
    "employment \\\%in\\\% c('unemployed','employed','retired')",
    "salary \\\%in\\\% c('none','low','medium','high')",
    "if (age == '<15') employment=='unemployed'",
    "if (salary != 'none') employment != 'unemployed'",
    "if (employment == 'unemployed') salary == 'none'"))


## Not run:
# plot total edit graph
plot(E)

# plot with a different layout
plot(E,layout=layout.circle)

# plot edit graph, just the connections caused by 'salary'
plot(E,vars='salary')

## End(Not run)

# extract edit graph
as.igraph(E)

# extract edit graph, just the connections caused by 'salary'
as.igraph(E,vars='salary')

# extract adjacency matrix
adjacency(E)

# extract adjacency matrix, only caused by 'employment'
adjacency(E,vars='employment')





}
\references{
  Csardi G, Nepusz T: The igraph software package for
  complex network research, InterJournal, Complex Systems
  1695. 2006. http://igraph.sf.net
}
\seealso{
  \code{\link{as.igraph}}, \code{\link{adjacency}},
  \code{igraph.plotting}
}

