COPS <- c("==","<","<=",">",">=")

retrieveCoef <- function(e, co=1){
   #stopifnot(is.language(e))
   if (length(e) == 1){
     if (is.numeric(e)){
        l <- co*e   #the resulting matrix is augmented so b has a -
        names(l) <- getOption("editrules.CONSTANT", "CONSTANT")
     }
     else {
        l <- co
        names(l) <- as.character(e)
     }
      return(l)
   }
   if (length(e) == 2){
     op <- deparse(e[[1]])
      rhs <- e[[2]]
     if (op == "("){
	    return(retrieveCoef(rhs, co))
	  } 
     else if (op == "-"){
        return(retrieveCoef(rhs, -1*co))
     }
	  else { 
		stop("Operator ", op, " not implemented", "Invalid expression:", e)
	  }
   }
   if (length(e) == 3){
     op <- deparse(e[[1]])
      lhs <- e[[2]]
      rhs <- e[[3]]
      lsign <- rsign <- co
     if ( op %in% c(COPS, "-")){
	    rsign <- -1 * co
	  } 
	  else if (op == "+"){
	  }
	  else if (op == "*"){
       co <- retrieveCoef(lhs, co)
       if (!is.numeric(co)){
                stop(paste("Expression contains nonconstant coefficient", paste(lhs,collapse="")))
       }
       return(retrieveCoef(rhs, co))
	  }
	  else { 
		stop("Operator ", op, " not implemented", "Invalid expression:", e)
	  }
	  return(c( retrieveCoef(lhs, lsign)
		      , retrieveCoef(rhs, rsign)
		  )
		)
   }
   stop("Invalid expression:", e)
}

parseGuard <- function(g){
  op <- deparse(g[[1]])
  if (op %in% c( COPS
               , "||"
               , "&&"
               ,"%in%"
               )
     ){
  }
  else {
     stop("Invalid condition syntax: ", g)
  }
}

makeEditRow <- function(edt){
  op <- as.character(edt[[1]])
  if (op == "if"){
     stop("Conditional edit rules are not (yet) supported.", edt)
     guard <- edt[[2]]
     print(eval(guard[[3]]))
     parseGuard(guard)
     edt <- edt[[3]]
     op <- as.character(edt[[1]])
  }
  else if (!(op %in% COPS)){
     stop(paste("Invalid edit rule:", edt))
  }
  wgt <- retrieveCoef(edt)
  # simplify the coefficients by summing them
  wgt <- tapply(wgt, names(wgt), sum)
  return(wgt)  
}

#' Transforms a list of R (in)equalities into an edit matrix.
#'
#' Transforms a list of R (in)equalities into an edit matrix with coefficients (\code{A}) for each variable, and a constant (\code{b})
#' and operator (\code{ops}) for each edit rule.
#'
#' Each row in the resulting editmatrix represents an linear (in) equality.
#' Each column in the resulting editmatrix represents a variable.
#'
#' There are two forms of creating an editmatrix:
#' \enumerate{ 
#'    \item a \code{character} vector with (in)equalities written in R syntax
#'    \item a \code{data.frame} with three columns:
#'       \itemize{
#'            \item name = a \code{character} with the name of each rule
#'            \item edit = a \code{character} with (in)equalities written in R syntax
#'            \item description = a \code{character} describing the intention of the rule
#'       }
#'      Typically these rules are stored in a external csv file (or database). 
#' }
#'
#' The second form is the prefered form, because it allows the documentation of constraints. This
#' may be very useful when the incorrect observations are analyzed.
#' If the first form is used, \code{editmatrix} internally creates the second form. This information
#' can be retrieved by using \code{\link{editrules}}
#'
#' The matrix is created by retrieving the coefficients of the variables in the equalities.
#' i.e. \code{x == y}   results in  \code{c(x=1, y=-1)}
#' and \code{x == y + w} results in \code{c(x=1, y=-1, w=-1)}
#'
#' By default the editmatrix is created using the comparison operators (\code{==,<=,>=,<,>}) in the edits. If option \code{normalize=TRUE} is used all 
#' edits are transformed into an A == b, A < b or A <= b form, so that in the specification of the edit rules all inequalities can be mixed, 
#' but the resulting matrix has similar sign.
#' @title Create an editmatrix
#' @seealso \code{\link{editrules}} \code{\link{as.editmatrix}}
#' @export
#' @example examples/editmatrix.R
#'
#' @param editrules \code{data.frame} with (in)equalities written in R syntax, see details for description or alternatively 
#'        a \code{character} with (in)equalities written in R syntax
#' @param normalize \code{logical} specifying if all edits should be transformed (see description)
#'
#' @return an object of class "editmatrix" which is a \code{matrix} with extra attributes
editmatrix <- function( editrules
                      , normalize = TRUE
					       ){   
   if (is.character(editrules)){
      edit <- editrules
      name <- NULL
      description <- NULL
      editrules <- NULL
    }
    else if (is.data.frame(editrules)){
      name <- editrules$name
      edit <- editrules$edit
      description <- editrules$description

      if (is.null(edit)){
         stop("The supplied data.frame misses the column 'edit'.\nSee ?editmatrix for a valid input specification")
      }            
    }
   else {
      stop("Invalid input, please use a character vector or a data.frame.\n See ?editmatrix for a valid input specification")
   }

   edts <- tryCatch(parse(text=edit), error=function(e){
         stop(paste("The edits could not be parsed. Parser returned\n",e$message))})   

   stopifnot(is.language(edts))
   
   edit <- sapply(edts, deparse, width.cutoff=500)
   edit <- gsub(" * ","*", fixed=TRUE, edit)
   
	if (is.null(name)){
	   name <- paste("e", seq_along(edit),sep="")
	}
   
    rowedts <- lapply(edts, function(edt){makeEditRow(edt)})
    ops <- sapply(edts, function(e){deparse(e[[1]])})
   
    vars <- unique(names(unlist(rowedts)))
    vars <- c(vars[vars!="CONSTANT"], "CONSTANT")

    A <- matrix( 0
                 , ncol=length(vars)
                 , nrow=length(rowedts)
                 , dimnames = list( rules = name
                                  , var=vars
                                  )
                 )
                 
    for (i in 1:length(rowedts)){
       A[i,names(rowedts[[i]])] <- rowedts[[i]]
    }
    A[,ncol(A)] <- -A[,ncol(A)]
   
   if (normalize){
      geq <- ops == ">="
      gt <- ops == ">"
      A[geq | gt,] <- -A[geq | gt,]
      ops[geq] <- "<="
      ops[gt] <- "<"      
   }
   
   names(ops) <- name
   E <- neweditmatrix(A, ops=ops, normalized=all(ops %in% c("==","<","<=")))
   attr(E, "description") <- description
   E
}

#' Create an \code{editmatrix} object from its constituing attributes. 
#'
#' This function is for internal purposes, please use \code{\link{editmatrix}} for creating an editmatrix object.
#' @param A An augmented \code{matrix} of the form \code{A|b}
#' @param ops a character vector with the comparison operator of every edit.
#' @param normalized \code{logical} TRUE or FALSE
#' @param ... optional attributes
#' @return an S3 object of class \code{editmatrix} 
neweditmatrix <- function(A, ops, normalized=FALSE,...){
   structure( A
            , class="editmatrix"
            , ops = ops
            , normalized = normalized
            , ...
            )
}


# NOTE: cannot export subscript function directly because Roxygen removes backticks in export
# directive. Added explicitly in build.bash

#' Row index operator for \code{editmatrix}.
#'
#' Use this operator to select edits from an editmatrix object.
#'
#' @usage `[.editmatrix`(x,i,j,...)
#' @param x an object of class \code{\link{editmatrix}}
#' @param i the row index in the edit matrix
#' @param j the column index in the edit matrix
#' @param ... arguments to be passed to other methods. Currently ignored.
#' @rdname editmatrix-subscript
`[.editmatrix` <- function(x, i, j, ...){
    neweditmatrix(
        A = as.matrix(x)[i, j, drop=FALSE],
        ops = getOps(x)[i]
        )
}



#' Check if object is an editmatrix
#' 
#' @seealso \code{\link{editmatrix}}
#' @export
#' @param x object to be checked
#' @return TRUE if \code{x} is an \code{editmatrix}
is.editmatrix <- function(x){
   return(inherits(x, "editmatrix"))
}

#' Coerce to an edit matrix. This method will derive editrules from a matrix.
#'
#' \code{as.editmatrix} interpretes the matrix as an editmatrix and derives readable edit rules. 
#' The columns of the matrix
#' are the variables and the rows are the edit rules (contraints).
#' 
#' If only argument \code{x} is given (the default), the resulting editmatrix is of the form \eqn{Ax=0}. 
#' This can be influenced by using the parameters \code{b} and \code{ops}.
#'
#' @export
#' @seealso \code{\link{editmatrix}}
#'
#' @param A object to be transformed into an \code{\link{editmatrix}}. \code{A} will be coerced to a matrix.
#' @param b Constant, a \code{numeric} of \code{length(nrow(x))}, defaults to 0
#' @param ops Operators, \code{character} of \code{length(nrow(x))} with the equality operators, defaults to "=="
#' @param ... further parameters will be given to \code{editmatrix}
#'
#' @return an object of class \code{editmatrix}.
as.editmatrix <- function( A
                         , b = numeric(nrow(A))
                         , ops = rep("==", nrow(A))
                         , ...
                         ){
    if (is.editmatrix(A)){
        return(A)
    }    
    vars <- colnames(A)
    if (is.null(vars)){
       colnames(A) <- make.names(paste("x", 1:ncol(A), sep=""), unique=TRUE)
    }
    rn <- rownames(A)
    if ( is.null(rn) || length(unique(rn)) != length(rn) ){
       rownames(A) <- paste("e", 1:nrow(A), sep="")
    }
    A <- cbind(as.matrix(A), CONSTANT=b)
    neweditmatrix(A=A, ops=ops)    
}

#' Coerce an editmatrix to a normal matrix
#' 
#' An \code{editmatrix} is a matrix and can be used as such, but it has extra attributes.
#' In some cases it is preferable to convert the editmatrix to a normal matrix.
#
#' Please note that coercion returns the augmented matrix \code{A|b} and not the \code{ops} part.
#'
#' @export
#' @method as.matrix editmatrix
#'
#' @param x editmatrix object
#' @param ... further arguments passed to or from other methods.
#'
#' @return augmented matrix of editmatrix
as.matrix.editmatrix <- function(x, ...){
   array(x, dim=dim(x), dimnames=dimnames(x))
}

#' Coerce an editmatrix to a \code{data.frame}
#'
#' Coerces an editmatrix to a \code{data.frame}. Useful for storing/viewing the matrix representation of editrules.
#' @export 
#' @method as.data.frame editmatrix
#' @param x editmatrix object
#' @param ... further arguments passed to or from other methods.
#'
#' @return data.frame with the coefficient matrix representation of \code{x}, an operator column and CONSTANT column.
as.data.frame.editmatrix <- function(x, ...){
   dat <- as.data.frame(getA(x))
   nms <- make.names( c(names(dat), "Ops", "CONSTANT")
                    , unique=TRUE
                    )
   n <- length(nms)
   dat[[nms[n-1]]] <- getOps(x)
   dat[[nms[n]]] <- getb(x)
   dat
}


#' Get character representation of editmatrix
#'
#' @export
#' @method as.character editmatrix
#'
#' @param x editmatrix object to be printed
#' @param ... further arguments passed to or from other methods.
as.character.editmatrix <- function(x, ...){
   A <- getA(x)
   b <- getb(x)
   vars <- getVars(x)
   ops <- getOps(x)

   n <- ncol(A)

   nC <- ncol(A) + 1
   er <- character(nrow(A))

   left <- right <- character(nrow(A)) 
   for ( i in seq_along(rownames(A)) ){
     r <- A[i,]
     lhs <- r > 0
     rhs <- r < 0
     left[i] <- if(any(lhs)) { paste(r[lhs], "*", vars[lhs],sep="",collapse=" + ") } else ""
     right[i] <-if(any(rhs))  { paste(-r[rhs], "*",vars[rhs],sep="",collapse=" + ") } else ""
   }
   left <- gsub("1\\*","",left)
   right <- gsub("1\\*","",right)

   right <- ifelse( right==""
                  , b
                  , ifelse(b == 0, right, paste(right,b,sep=" + "))
                  )
   left <- ifelse(left=="", "0", left)
   txt <- paste(left,ops,right)    
   names(txt) <- rownames(x)
   txt
}

#' Get expression representation of editmatrix
#'
#' @export
#' @method as.expression editmatrix
#'
#' @param x editmatrix object to be parsed
#' @param ... further arguments passed to or from other methods.
as.expression.editmatrix <- function(x, ...){
  return(
     tryCatch(parse(text=as.character(x)), 
         error=function(e){
             stop(paste("Not all edits can be parsed, parser returned", e$message,sep="\n"))
         }
     )
 )
}

#' print edit matrix
#'
#' @export
#' @method print editmatrix
#'
#' @param x editmatrix object to be printed
#' @param ... further arguments passed to or from other methods.
print.editmatrix <- function(x, ...){
   cat("Edit matrix:\n")
   print(as.data.frame(x), ...)
   cat("\nEdit rules:\n")
   info <- editrules(x)
   desc <- paste("[",info$description,"]")
   desc <- ifelse(info$description=="","", desc)
   cat( paste( info$name,":", info$edit, desc, collapse="\n")
      , "\n"
      )
}


#' \code{\link{str}} method for editmatrix object
#'
#'
#' @param object \code{\link{editmatrix}} object
#' @param ... methods to pass to other methods
#' @export
str.editmatrix <- function(object,...){
    ivar <- rowSums(abs(object)) > 0
    vars <- paste(getVars(object)[ivar],collapse=", ")
    if (nchar(vars) > 20 ) vars <-  paste(strtrim(vars,16),"...") 
    cat(paste("editmatrix with", ncol(object), "edits containing variables",vars,"\n"))
}




