% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ddhazard.R
\name{ddhazard}
\alias{ddhazard}
\title{Fitting Dynamic Hazard Models}
\usage{
ddhazard(
  formula,
  data,
  model = "logit",
  by,
  max_T,
  id,
  a_0,
  Q_0,
  Q = Q_0,
  order = 1,
  weights,
  control = ddhazard_control(),
  verbose = F
)
}
\arguments{
\item{formula}{\code{\link[survival]{coxph}} like formula with \code{\link[survival]{Surv}(tstart, tstop, event)} on the left hand site of \code{~}.}

\item{data}{\code{data.frame} or environment containing the outcome and covariates.}

\item{model}{\code{"logit"}, \code{"cloglog"}, or \code{"exponential"} for
respectively the logistic link function with discrete outcomes,
the inverse cloglog link function with discrete outcomes,
or for the continuous time model with piecewise constant exponentially
distributed arrival times.}

\item{by}{interval length of the bins in which parameters are fixed.}

\item{max_T}{end of the last interval interval.}

\item{id}{vector of ids for each row of the in the design matrix.}

\item{a_0}{vector \eqn{a_0} for the initial coefficient vector for the first iteration (optional). Default is estimates from static model (see \code{\link{static_glm}}).}

\item{Q_0}{covariance matrix for the prior distribution.}

\item{Q}{initial covariance matrix for the state equation.}

\item{order}{order of the random walk.}

\item{weights}{weights to use if e.g. a skewed sample is used.}

\item{control}{list of control variables from
\code{\link{ddhazard_control}}.}

\item{verbose}{\code{TRUE} if you want status messages during execution.}
}
\value{
A list with class \code{ddhazard}. The list contains
\item{formula}{the passed formula.}
\item{call}{the matched call.}
\item{state_vecs}{2D matrix with the estimated state vectors (regression parameters) in each bin.}
\item{state_vars}{3D array with smoothed variance estimates for each state vector.}
\item{lag_one_cov}{3D array with lagged correlation matrix for each for each change in the state vector. Only present when the model is logit and the method is EKF.}
\item{n_risk}{the number of observations in each interval.}
\item{times}{the interval borders.}
\item{risk_set}{the object from \code{\link{get_risk_obj}} if saved.}
\item{data}{the \code{data} argument if saved.}
\item{weights}{\code{weights} used in estimation if saved.}
\item{id}{ids used to match rows in \code{data} to individuals.}
\item{order}{order of the random walk.}
\item{F_}{matrix which map from one state vector to the next.}
\item{method}{method used in the E-step.}
\item{est_Q_0}{\code{TRUE} if \code{Q_0} was estimated in the EM-algorithm.}
\item{family}{Rcpp \code{\link{Module}} with C++ functions used for estimation given the \code{model} argument.}
\item{discrete_hazard_func}{the hazard function corresponding to the \code{model} argument.}
\item{terms}{the \code{\link{terms}} object used.}
\item{has_fixed_intercept}{\code{TRUE} if the model has a time-invariant intercept.}
\item{xlev}{a record of the levels of the factors used in fitting.}
}
\description{
Function to fit dynamic hazard models using state space models.
}
\details{
This function can be used to estimate survival models where the regression parameters follows a given order random walk. The order is specified by the \code{order} argument. 1. and 2. order random walks is implemented. The regression parameters are updated at time \code{by}, 2\code{by}, ..., \code{max_T}. See the \code{vignette("ddhazard", "dynamichazard")} for details.

All filter methods needs a state covariance matrix \code{Q_0} and state vector \code{a_0}. An estimate from a time-invariant model is used for \code{a_0} if it is not supplied (the same model you would get from \code{\link{static_glm}}). A diagonal matrix with large entries is recommended for \code{Q_0}. What is large dependents on the data set and \code{model}. Further, a covariance matrix for the first iteration \code{Q} is needed. The \code{Q} and \code{a_0} are estimated with an EM-algorithm.

The model is specified through the \code{model} argument. The discrete
outcome models are where outcomes are binned into the intervals.
Be aware that there can be "loss" of information due to binning if outcomes
are not discrete to start with. It is key for these models that the
\code{id} argument is provided if individuals in the data set have
time-varying covariates. The the exponential model use a piecewise constant
exponential distribution
for the arrival times where there is no "loss" information due to binning.
Though, one of the assumptions of the model is not satisfied if outcomes
are only observed in discrete time intervals.

It is recommended to see the Shiny app demo for this function by calling \code{\link{ddhazard_app}()}.
}
\examples{
# example with first order model
library(dynamichazard)
fit <- ddhazard(
 Surv(time, status == 2) ~ log(bili), pbc, id = pbc$id, max_T = 3600,
 Q_0 = diag(1, 2), Q = diag(1e-4, 2), by = 50,
 control = ddhazard_control(method = "GMA"))
plot(fit)

# example with second order model
fit <- ddhazard(
 Surv(time, status == 2) ~ log(bili), pbc, id = pbc$id, max_T = 3600,
 Q_0 = diag(1, 4), Q = diag(1e-4, 2), by = 50,
 control = ddhazard_control(method = "GMA"),
 order = 2)
plot(fit)

}
\references{
Fahrmeir, Ludwig. \emph{Dynamic modelling and penalized likelihood estimation for discrete time survival data}. Biometrika 81.2 (1994): 317-330.

Durbin, James, and Siem Jan Koopman. \emph{Time series analysis by state space methods}. No. 38. Oxford University Press, 2012.
}
\seealso{
\code{\link[=plot.ddhazard]{plot}}, \code{\link[=residuals.ddhazard]{residuals}}, \code{\link[=predict.ddhazard]{predict}}, \code{\link{static_glm}}, \code{\link{ddhazard_app}}, \code{\link{ddhazard_boot}}
}
