#' Add a Marker Axis with Chromosome Names to a Plot of Polarized Genotypes
#'
#' This function adds a marker axis with chromosome names to an existing plot of polarized
#' genotypes. It requires that the plot is already created using \link{plotPolarized}.
#'
#' @param includedSites character. Path to a file with columns \code{CHROM} and \code{POS}.
#' @inheritParams diem
#' @param tickDist numeric. Indicates the spacing of physical tick marks along a chromosome.
#' @param axisInfo list with user-defined tick positions and labels for marker axis. See
#'   Details.
#' @details The \code{includedSites} file should ideally be generated by
#' \link{vcf2diem} to ensure congruence between the plotted genotypes and
#' the respective metadata.
#'
#' Tick mark distances within a chromosome are located at \code{tickDist} and formated to
#' multiples of millions.
#'
#' The optional \code{axisInfo} argument must have five named elements with the following
#' information:
#' * \code{CHROMbreaks}: Numeric vector with positions defining ticks separating 
#'   chromosomes. The metric for all positions is in the number of markers.
#' * \code{CHROMnamesPos}: Numeric vector with positions to place the chromosome
#'   labels.
#' * \code{CHROMnames}: Character vector with the names of the chromosomes. Must be the
#'   same length as \code{CHROMnamesPos}.
#' * \code{ticksPos}: Numeric vector with positions of ticks within chromosomes.
#' * \code{ticksNames}: Character vector with the names to be displayed at \code{ticksPos}.
#'
#' When \code{axisInfo = NULL}, the function extracts the necessary information from the
#' \code{includedSites} file.
#' @return Returns an invisible \code{axisInfo} list with the tick positions and labels 
#' for the marker axis.
#' @importFrom utils modifyList
#' @importFrom grDevices dev.cur 
#' @importFrom graphics mtext
#' @export
#' @examples
#' \dontrun{
#' # Run this example in a working directory with write permissions
#' myo <- system.file("extdata", "myotis.vcf", package = "diemr")
#' vcf2diem(myo, "myo")
#' inds <- 1:14
#' fit <- diem("myo-001.txt", ChosenInds = inds, ploidy = FALSE)
#' gen <- importPolarized("myo-001.txt", fit$markerPolarity, inds)
#' h <- apply(gen, 1, function(x) pHetErrOnStateCount(sStateCount(x)))[1, ]
#' plotPolarized(gen, h, xlab = "")
#' plotMarkerAxis("myo-includedSites.txt", rep(TRUE, 11), tickDist = 100)
#' }
plotMarkerAxis <- function(includedSites, ChosenSites, tickDist = 1e+06, axisInfo = NULL, ...) {
  if (dev.cur() == 1) {
    stop("Plot polarized genotypes with plotPolarized first.")
  }
  if (is.null(axisInfo)) {
    axisInfo <- markerAxis(
      includedSites = includedSites,
      ChosenSites = ChosenSites, tickDist = tickDist
    )
  }

  userArgs <- list(...)
  plottingArgs <- utils::modifyList(list(
    # axis defaults
    side = 1,
    las = 1,
    tcl = -.5,
    cex = 1,
    line = 0
  ), userArgs)


  acceptedAxisArgs <- c(
    "side", "col.ticks", "labels", "las", "tick", "line", "tcl", "cex", "cex.axis",
    "pos", "outer", "font", "lty", "lwd", "lwd.ticks", "hadj", "padj", "gap.axis",
    "xpd"
  )
  axisArgs <- plottingArgs[names(plottingArgs) %in% acceptedAxisArgs]

  # ticks of physical distances
  do.call(axis, utils::modifyList(
    axisArgs,
    list(at = axisInfo$ticksPos, labels = axisInfo$ticksNames, cex = axisArgs$cex)
  ))
  # ticks for chromosome separators
  do.call(axis, utils::modifyList(
    axisArgs,
    list(at = axisInfo$CHROMbreaks, tcl = axisArgs$tcl * 2.2, labels = FALSE)
  ))
  # chromosome names
  do.call(mtext, utils::modifyList(
    axisArgs,
    list(
      at = axisInfo$CHROMnamesPos,
      text = axisInfo$CHROMnames,
      line = axisArgs$line + 2.5,
      cex = axisArgs$cex * 1.1
    )
  ))
  invisible(return(axisInfo))
}
