#' deducorrect object
#'
#'
#' All \code{correct*} functions of the deducorrect package return an S3 object of class \code{deducorrect}.
#' The "public slots" are
#' \itemize{
#'  \item{\code{corrected} A copy of the input \code{data.frame}, with records corrected where possible.}
#'  \item{\code{corrections} A data.frame with the following colums:
#'  \itemize{    
#'      \item{\code{row} Row number where correction was applied}
#'      \item{\code{variable} Variable name where correction was applied} 
#'      \item{\code{old} Old value of adapted variable}
#'      \item{\code{new} New value of adapted variable}
#'  }}   
#'  \item{\code{status} A \code{data.frame} with the same number of rows as \code{corrected}. It has at least
#'  a column called \code{\link{status}}. Further columns might be present, depending on the used correction function.}
#'  \item{\code{generatedby} The name of the function that called \code{\link{newdeducorrect}} to construct the object.}
#'  \item{\code{timestamp} The time and date when the object was generated, as returned by \code{\link{date}}.}
#' } 
#' 
#' 
#' @title deducorrect object 
#' 
#' 
#' @name deducorrect-object
#' @rdname deducorrect-object
{}





#' Generate an S3 \code{deducorrect} object
#' @aliases print.deducorrect
#' @param corrected The corrected data.frame
#' @param corrections A \code{data.frame} listing old and new values for every row and variable where corrections were applied
#' @param status A \code{data.frame} with at least one \code{\link{status}} column.
#' @return an S3 object of class \code{deducorrect}
#' 
#' @seealso \code{\link{deducorrect-object}}
newdeducorrect <- function(corrected, corrections, status){  
    structure(
        list(
            corrected   = corrected, 
            corrections = corrections, 
            status      = status,
            timestamp   = date(),
            generatedby = sys.call(-1)[[1]]
        ),
        class = c("deducorrect","list"),
        statsummary = addmargins(table(status$status, useNA="ifany")),
        corrsummary = addmargins(table(corrections$variable, useNA="no"))
    )
}

#' @nord
#' @export
print.deducorrect <- function(x, ...){

    cat(paste(" deducorrect object generated by '", 
        x$generatedby, "'", " on ",x$timestamp, sep=""))
    cat(paste("\n slots: ",paste(paste("$",names(x),sep=""), collapse=", ")))
    
    cat("\n\n Record status:")  
    print(attr(x,"statsummary"))
    cat("\n Variables corrected:")
    print(attr(x,"corrsummary"))
}

