#' Collapse a distance matrix by amalgamating populations with pairwise fixed difference count 
#' less that a threshold
#'
#' This script takes a the file generated by gl.fixed.diff and amalgamates populations with 
#' distance less than or equal to a specified threshold. The distance matrix is generated by 
#' gl.fixed.diff().
#' 
#' The script then applies the new population assignments to the genlight object and recalculates 
#' the distance and associated matricies.
#'
#' @param fd -- name of the list of matricies produced by gl.fixed.diff() [required]
#' @param tloc -- threshold defining a fixed difference (e.g. 0.05 implies 95:5 vs 5:95 is fixed) [0]
#' @param tpop -- threshold number of fixed differences above which populatons will not be 
#' amalgamated [0]
#' @param pb -- if TRUE, show a progress bar on time consuming loops [FALSE]
#' @param verbose -- verbosity: 0, silent or fatal errors; 1, begin and end; 2, progress log ; 
#' 3, progress and results summary; 5, full report [default 2 or as specified using gl.set.verbosity]
#' @return A list containing the gl object x and the following square matricies
#'         [[1]] $gl -- the new genlight object with populations collapsed;
#'         [[2]] $fd -- raw fixed differences;
#'         [[3]] $pcfd -- percent fixed differences;
#'         [[4]] $nobs -- mean no. of individuals used in each comparison;
#'         [[5]] $nloc -- total number of loci used in each comparison;
#'         [[6]] $expfpos -- NA's, populated by gl.fixed.diff [by simulation]
#'         [[7]] $expfpos -- NA's, populated by gl.fixed.diff [by simulation]
#'         [[8]] $prob -- NA's, populated by gl.fixed.diff [by simulation]
#' @importFrom methods show       
#' @export
#' @author Arthur Georges (Post to \url{https://groups.google.com/d/forum/dartr})
#' @examples
#' fd <- gl.fixed.diff(testset.gl,tloc=0.05)
#' fd
#' fd2 <- gl.collapse(fd,tpop=1)
#' fd2
#' fd3 <- gl.collapse(fd2,tpop=1)
#' fd3

gl.collapse <- function(fd, 
                        tpop=0, 
                        tloc=0, 
                        pb=FALSE,
                        verbose=NULL) {
  
# TRAP COMMAND, SET VERSION
  
  funname <- match.call()[[1]]
  build <- "Jacob"
  
# SET VERBOSITY
  
  if (is.null(verbose)){ 
      verbose <- 2
  } 
  
  if (verbose < 0 | verbose > 5){
    cat(paste("  Warning: Parameter 'verbose' must be an integer between 0 [silent] and 5 [full report], set to 2\n"))
    verbose <- 2
  }
  
# FLAG SCRIPT START
  
  if (verbose >= 1){
    if(verbose==5){
      cat("Starting",funname,"[ Build =",build,"]\n")
    } else {
      cat("Starting",funname,"\n")
    }
  }
  
# STANDARD ERROR CHECKING
  
  if (class(fd) != "fd") {
    stop("Fatal Error: fd must be a list of class 'fd' produced by gl.fixed.diff or gl.collapse\n")
  }
  
# FUNCTION SPECIFIC ERROR CHECKING
  
  if (tloc > 0.5 || tloc < 0 ) {
    stop("Fatal Error: Parameter tloc should be positive in the range 0 to 0.5\n")
  } 
  
  if (tpop < 0 ) {
    stop("Fatal Error: Parameter tpop should be a positive integer\n")
  }  
  
# DO THE JOB
  
  if ( verbose >= 2){
    if (tloc > 0) {cat("  Comparing populations for fixed differences with tolerance",tloc,"\n")}
    if (tloc == 0) {cat("  Comparing populations for absolute fixed differences\n")}
    if (tpop == 1) {cat("  Amalgamating populations with corrobrated fixed differences, tpop =",tpop,"\n")}
    if (tpop > 1) {cat("  Amalgamating populations with fixed differences <= ",tpop,"\n")}
    if (tpop == 0) {cat("  Amalgamating populations with zero fixed differences\n")}
  }

# Store the number of populations in the matrix
  npops <- dim(fd$fd)[1]
  
# Extract the column names
  pops <- variable.names(fd$fd)
  
# Initialize a list to hold the populations that differ by <= tpop
  zero.list <- list()

  # For each pair of populations
  for(i in 1:npops){
    zero.list[[i]] <- c(rownames(fd$fd)[i])
    for (j in 1:npops) {
      if (fd$fd[i,j] <= tpop) {
        zero.list[[i]] <- c(zero.list[[i]],rownames(fd$fd)[i],rownames(fd$fd)[j])
        zero.list[[i]] <- unique(zero.list[[i]])
      }
    }
    zero.list[[i]] <- sort(zero.list[[i]])
  }
  
# Pull out the unique aggregations  
  zero.list <- unique(zero.list)
  
# Amalgamate populations
  if (length(zero.list) >= 2) {
    for (i in 1:(length(zero.list)-1)) {
      for (j in 2:length(zero.list)) {
        if (length(intersect(zero.list[[i]],zero.list[[j]])) > 0 ) {
          zero.list[[i]] <- union(zero.list[[i]],zero.list[[j]])
          zero.list[[j]] <- union(zero.list[[i]],zero.list[[j]])
        }
      }
    }
    for (i in 1:length(zero.list)) {
      zero.list <- unique(zero.list)
    }
  }  
  zero.list.hold <- zero.list
  
# Print out the results of the aggregations 
  if(verbose >= 3) {
    cat("\nInitial Populations\n",pops,"\n")
    cat("\nNew population groups\n")
  }
  
  x <- fd$gl
  for (i in 1:length(zero.list)) {
    # Create a group label
      if (length(zero.list[[i]])==1) {
        replacement <- zero.list[[i]][1]
      } else {
        replacement <- paste0(zero.list[[i]][1],"+")
      }
      if(verbose >= 3) {
        
        if(length(zero.list[[i]]) > 1){
          cat(paste0("Group:",replacement,"\n"))
          print(as.character(zero.list[[i]]))
          cat("\n")
        }
      }
    # Amalgamate the populations
      x <- gl.merge.pop(x,old=zero.list[[i]],new=replacement,verbose=0)
  }
    
# Recalculate matricies
    fd2 <- gl.fixed.diff(x,tloc=tloc,pb=pb,verbose=2)

  if(setequal(nPop(x),nPop(fd$gl))) { 
    if (verbose >= 2) {
      cat(paste("\nNo further amalgamation of populations at fd <=", tpop,"\n"))
      cat("  Analysis complete\n\n")
    }
    l <- list(gl=fd2$gl,fd=fd2$fd,pcfd=fd2$pcfd,nobs=fd2$nobs,nloc=fd2$nloc,expfpos=fd2$expfpos,sdfpos=fd$sdfpos,pval=fd2$pval)
    class(l) <- "fd"
    } else {
    # Display the fd matrix
    if (verbose >= 4) {
      cat("\n\nRaw Fixed Difference Matrix\n")
      print(fd2$fd)
      cat("\n")
    }
    if (verbose >= 3) {
      cat("Sample sizes")
      print(table(pop(x)))
      cat("\n")
    }

# Create the list for output
    l <- list(gl=fd2$gl,fd=fd2$fd,pcfd=fd2$pcfd,nobs=fd2$nobs,nloc=fd2$nloc,expfpos=fd2$expfpos,sdfpos=fd2$sdfpos,pval=fd2$pval)
  }
  class(l) <- "fd"
  
  # Explanatory bumpf
    if (verbose >= 4) {
      if(pb){cat("\n")}
      cat("Returning a list of class 'fd' containing the new genlight object and square matricies, as follows:\n",
          "         [[1]] $gl -- input genlight object;\n",
          "         [[2]] $fd -- raw fixed differences;\n",
          "         [[3]] $pcfd -- percent fixed differences;\n",
          "         [[4]] $nobs -- mean no. of individuals used in each comparison;\n",
          "         [[5]] $nloc -- total number of loci used in each comparison;\n",
          "         [[6]] $expfpos -- NAs, populated by gl.fixed.diff [by simulation];\n",
          "         [[7]] $sdfpos -- NAs, populated by gl.fixed.diff [by simulation];\n",
          "         [[8]] $prob -- NAs, populated by gl.fixed.diff [by simulation].\n")
    }  
    
# FLAG SCRIPT END
  
  if (verbose > 0) {
    cat("Completed:",funname,"\n")
  }
  
  return(l)
}
