% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calcConvCt.r
\name{calcConvCt}
\alias{calcConvCt}
\title{Computes Ct-metric scores for putatively convergent tips (or groups of tips) given a set of user provided phenotypic characters and a time calibrated phylogeny.}
\usage{
calcConvCt(
  phy,
  traits,
  focaltaxa,
  groups = NULL,
  conservative = FALSE,
  VERBOSE = FALSE
)
}
\arguments{
\item{phy}{The time calibrated phylogeny of interest in phylo format}

\item{traits}{A matrix of numeric phenotypic traits with rownames matching tip labels of phy}

\item{focaltaxa}{A vector of tip labels for the putatively convergent taxa to be compared}

\item{groups}{An optional vector of groups with names matching focaltaxa. Indicates the group identity of all putatively convergent taxa and limits Ct measures to intergroup comparisons only}

\item{conservative}{Logical value indicating whether measurement of Dmax.t should be restricted to before the origin of the oldest lineage in each pairwise comparison of the focaltaxa. The origin of each convergent lineages is taken as the most recent common ancestor of tips in each user defined group. Where groups include a single tip, the parent node of the tip's subtending branch is used. Requires group object to be provided by user.}

\item{VERBOSE}{Logical value indicating whether model information should be printed during computation}
}
\value{
A list of the following components:

mean A named vector of Ct-metrics averaged from all pairwise comparisons of focaltaxa. If user provided groups, this is based only on comparisons between taxa belonging to different groups.

Cmat A matrix of Ct-metrics for each pairwise comparison.

path_df A list of dataframes, one per pairwise comparison of the focal taxa, each containing information from all timepoint measurements of the two putatively convergent lineages. These provide the nodes at which comparisons were drawn, the evolutionary path along which that node fell (i.e., leading to one of two tips), the node height, reconstructed ancestral states at that node for each phenotypic trait, reconstructed ancestral values for each trait along the opposite path, and the phenotypic distance between the two lineages at that point.

grp.mean A matrix of Ct-metrics summarized for inter-group comparisons, returned only if user defined groups were specified. Provides overall results matching those reported in "mean", results for each unique inter-group comparison, and results averaged with equal weight given to each unique inter-group comparison (i.e., ignoring differences in the number of tips in each group).
}
\description{
calcConvCt Computes Ct-metric scores for putatively convergent tips (or groups of tips) given a set of user provided phenotypic characters and a time calibrated phylogeny.
}
\details{
Function incorporates the optimizations introduced by Zelditch et al. (2017), which significantly improve runtimes

Reconstructions part way along branches are obtained using equation [2] of Felsenstein (1985), following code modified from the phytools (Revell, 2012) function contMap
}
\examples{
\donttest{# create time calibrated tree
mytree<-rtree(100)
mycalibration <- makeChronosCalib(mytree, node="root", age.max=50)
phy <- chronos(mytree, calibration = mycalibration, control = chronos.control() )
class(phy)<-"phylo"

# create three normally distributed phenotypic traits
traits <- cbind(rnorm(Ntip(phy)),rnorm(Ntip(phy)),rnorm(Ntip(phy)))
rownames(traits) <- phy$tip.label
focaltaxa <- sample(phy$tip.label, 5)

system.time(run <- calcConvCt(phy, traits, focaltaxa))}
}
\references{
Grossnickle DM, Brightly WH, Weaver LN, Stanchak KE, Roston RA, Pevsner SK, Stayton CT, Polly PD, Law CJ. 2022. A cautionary note on quantitative measures of phenotypic convergence. in revision
Zelditch ML, Ye J, Mitchell JS, Swiderski DL. 2017. Rare ecomorphological convergence on a complex adaptive landscape: Body size and diet mediate evolution of jaw shape in squirrels (Sciuridae). Evolution 71: 633-649
Stayton CT. 2015. The definition, recognition, and interpretation of convergent evolution and two new measures for quantifying and assessing the significance of convergence. Evolution 69(8): 2140-2153.
Revell, L. J. (2012) phytools: An R package for phylogenetic comparative biology (and other things). Methods Ecol. Evol., 3, 217-223.
Felsenstein, J. 1985. Phylogenies and the comparative method. American Naturalist, 125, 1-15.
}
