% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/valQ.R
\name{valQ}
\alias{valQ}
\title{Empirical Q-matrix validation}
\usage{
valQ(
  fit,
  index = "PVAF",
  iterative = "test.att",
  emptyatt = TRUE,
  maxitr = 100,
  CDMconv = 1e-04,
  verbose = TRUE
)
}
\arguments{
\item{fit}{A G-DINA model fit object from the \code{GDINA} package (Ma & de la Torre, 2020).}

\item{index}{What index to use. It includes \code{"PVAF"} or \code{"R2"}. The default is \code{"PVAF"}.}

\item{iterative}{(Iterative) implementation procedure. It includes \code{"none"} (for non-iterative), \code{"test"} (for test-level iterations), \code{"test.att"} (for attribute-test-level), and \code{"item"} (for item-level iterations). The default is \code{"test.att"}.}

\item{emptyatt}{Is it possible for the suggested Q-matrix to have an empty attribute (i.e., an attribute not measured by any item)? Although rarely, it is possible for iterative procedures to provide a suggested Q-matrix in which one or more attributes are empty. This might indicate that the original Q-matrix had more attributes than necessary. If \code{FALSE}, then at least one item (i.e., the one that is most likely) will measure each attribute in the suggested Q-matrix. The default is \code{TRUE}.}

\item{maxitr}{Maximum number of iterations if an iterative procedure has been selected. The default is 100.}

\item{CDMconv}{Convergence criteria for the CDM estimations between iterations (only if an iterative procedure has been selected). The default is 0.0001.}

\item{verbose}{Print information after each iteration if an iterative procedure is used. The default is \code{TRUE}.}
}
\value{
\code{valQ} returns an object of class \code{valQ}.
\describe{
\item{\code{sug.Q}}{Suggested Q-matrix (\code{matrix}).}
\item{\code{Q}}{Original Q-matrix (\code{matrix}).}
\item{\code{sugQ.fit}}{Several fit indices from the model obtained with the suggested Q-matrix (\code{vector}).}
\item{\code{index}}{PVAF or pseudo R-squared (depending on which one was used) for each item (\code{matrix}).}
\item{\code{iter.Q}}{Q-matrices used in each iteration (\code{list}). Provided only if an iterative procedure has been used.}
\item{\code{iter.index}}{PVAF or pseudo R-squared (depending on which one was used) for each item in each iteration (\code{list}). Provided only if an iterative procedure has been used.}
\item{\code{n.iter}}{Number of iterations used (\code{double}). Provided only if an iterative procedure has been used.}
\item{\code{convergence}}{Convergence information (\code{double}). It can be 1 (convergence), 2 (lack of convergence: maximum number of iterations achieved), 3 (lack of convergence: empty attribute obtained), and 4 (lack of convergence: loop Q-matrices). Provided only if an iterative procedure has been used.}
\item{\code{time}}{Initial and finish time (\code{vector}).}
\item{\code{time.used}}{Total computation time (\code{difftime}).}
\item{\code{specifications}}{Function call specifications (\code{list}).}
}
}
\description{
Empirical Q-matrix validation using the \emph{Hull} method (Nájera, Sorrel, de la Torre, & Abad, 2020a).
The procedure can be used either with the PVAF (de la Torre & Chiu, 2016) or McFadden's pseudo R-squared (McFadden, 1974).
The PVAF is recommended (Nájera, Sorrel, de la Torre, & Abad, 2020a).
Note that the pseudo R-squared might not be computationally feasible for highly dimensional Q-matrices, say more than 10 attributes.
Different iterative implementations are available, such as the test-level implementation (see Terzi & de la Torre, 2018), attribute-test-level implementation (Nájera, Sorrel, de la Torre, & Abad, 2020a), and item-level implementation (Nájera, Sorrel, de la Torre, & Abad, 2020b).
If an iterative implementation is used, the \code{GDINA} R package (Ma & de la Torre, 2020) is used for the calibration of the CDMs.
}
\examples{
library(GDINA)
dat <- sim30GDINA$simdat
Q <- sim30GDINA$simQ # Generating Q-matrix
miss.Q <- missQ(Q = Q, qjk = .30, retainJ = 5, seed = 123)$miss.Q # Misspecified Q-matrix
fit <- GDINA(dat, miss.Q) # GDINA object
sug.Q <- valQ(fit = fit, verbose = TRUE) # Hull method for Q-matrix validation
mean(sug.Q$sug.Q == Q) # Check similarity with the generating Q-matrix
}
\references{
de la Torre, J., & Chiu, C.-Y. (2016). A general method of empirical Q-matrix validation. \emph{Psychometrika}, \emph{81}, 253-273. https://doi.org/10.1007/s11336-015-9467-8

Ma, W., & de la Torre, J. (2020). GDINA: An R package for cognitive diagnosis modeling. \emph{Journal of Statistical Software}, \emph{93}(14). https://doi.org/10.18637/jss.v093.i14

McFadden, D. (1974). Conditional logit analysis of qualitative choice behavior. In P. Zarembka (Ed.), \emph{Frontiers in Economics} (pp. 105-142). Academic Press.

Nájera, P., Sorrel, M. A., de la Torre, J., & Abad, F. J. (2020a). Balancing fit and parsimony to improve Q-matrix validation. \emph{British Journal of Mathematical and Statistical Psychology}. https://doi.org/10.1111/bmsp.12228

Nájera, P., Sorrel, M. A., de la Torre, J., & Abad, F. J. (2020b). Improving robustness in Q-matrix validation using an iterative and dynamic procedure. \emph{Applied Psychological Measurement}, \emph{46}, 431-446. https://doi.org/10.1177/0146621620909904

Terzi, R., & de la Torre, J. (2018). An iterative method for empirically-based Q-matrix validation. \emph{International Journal of Assessment Tools in Education}, \emph{5}, 248-262. https://doi.org/10.21449/ijate.407193
}
\author{
{Pablo Nájera, Universidad Autónoma de Madrid \cr Miguel A. Sorrel, Universidad Autónoma de Madrid \cr Francisco J. Abad, Universidad Autónoma de Madrid}
}
