% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/combinationIndex.R
\name{estimateCombinationIndices}
\alias{estimateCombinationIndices}
\alias{estimateCombinationIndex}
\alias{estimateChouIndices}
\alias{estimateChouIndex}
\title{Combination Index}
\usage{
estimateCombinationIndices(
  concs,
  act,
  level,
  weights = NULL,
  limits = c(NA, NA)
)

estimateCombinationIndex(dr1, dr2, drc, ratio, level, limits = c(NA, NA))

estimateChouIndices(concs, act, level, range, excess = "clip")

estimateChouIndex(dr1, dr2, drc, ratio, level, range, excess = "clip")
}
\arguments{
\item{concs}{A width-two array of concentrations representing all measured
dose pairs}

\item{act}{A vector of measured activity or effect values}

\item{level}{A numeric vector of one or more effect levels at which to
estimate the combination index}

\item{weights}{A vector of weights (between 0 and 1) the same length as
the data which determines the weight with which each measurement
will impact the the sum of squared errors.  Weights will be multiplied by
errors \emph{before} squaring.  If \code{NULL} (the default) all weights will be set
to 1. Can be a numeric vector, or the name of a column in \code{data} if \code{formula}
is a symbolic formula}

\item{limits}{The fixed values of the minimal and maximal effects of the
drugs and the combination. By default, both values are set to \code{NA}; any
value set to \code{NA} will fit from the data.}

\item{dr1}{A data frame with two columns, \code{conc} and \code{act} reflecting the
dose response behavior of the first drug alone}

\item{dr2}{A data frame with two columns, \code{conc} and \code{act} reflecting the
dose response behavior of the second drug alone}

\item{drc}{A data frame with two columns, \code{conc} and \code{act} reflecting the
dose response behavior of a constant ratio combination; \code{conc} should be the
combined concentrations of the two drugs}

\item{ratio}{The ratio of the two drugs in the constant ratio combination
(dose A to dose B)}

\item{range}{The range of effects assumed by the median effect model; a
two-element numeric vector containing the minimal effect and the maximal
effect, in that order.}

\item{excess}{For \code{estimateChouIndices} and \code{estimateChouIndex}, what should
be done with values outside the expected range.  If "clip" (the default),
values will be clipped to 0.1\% and 99.9\% of the expected range; if "drop",
values outside the range will be dropped}
}
\value{
For \code{estimateCombinationIndices} and \code{estimateChouIndices}, a data
frame with the following columns:
\itemize{
\item \code{ratio}: The ratio of doses (dose A to dose B) along which the index
was estimated
\item \code{level}: The effect level at which the index was estimated
\item \code{ci}: The estimated combination index at that dose ratio and effect level
}

Combination index estimates will be included for all provided effect levels
and all dose ratios present. For \code{estimateCombinationIndex} and
\code{estimateChouIndex}, a vector of estimated combinatoin indices the same
length as \code{level}.
}
\description{
Estimates the combination index using the median effet method of Chou and
Talalay (1984) or the more robust method of non-linear optimization.
}
\details{
The combination index is a peculiar method, as it does not produce values
corresponding to each measurement (as the deviation methods do), nor does it
produce a value shared by the entire surface (as response surface methods
do).  It instead, produces a value associated with a particular effect level
and a particular \emph{dose ratio}.  This makes implementing the method
consistently for a wide range of possible data sources quite tricky;
nevertheless, we have attempted to do so here.  In brief, the combination
index method involves fitting the dose response of both individual drugs and
and a constant ratio combination of the drugs (treated as a virtual third
drug).  It the compares the potency of the constant-ratio combination
(estimated a particular effect level) with the expected potency according to
Loewe additivity, and returns the degree to which the combination is \emph{more}
potent (with a combination index less than 1) or \emph{less} potent (with a
combination greater than one) than expected by additivity.  This method is
also in turns known as the sum of fractional inhibitory coefficients (FICs),
observed-over-expected, or originally as Berenbaum's interaction index.

Because the method was originally built for three distinct sets of dose
response measurements, we have included the \code{estimateCombinationIndex} and
\code{estimateChouIndex} functions which operate on three separate data frames.
However, in most cases, it will be easier to use the
\code{estimateCombinationIndices} and \code{estimateChouIndices} functions, which
operate on an array of concentrations and a vector of responses, just like
the numerous other functions in this package, and generate a set of
combination index values with level and ratio information  included.

The only difference between the \verb{estimateCombination*} and \verb{estimateChou*}
functions is the way in which they perform dose response fitting.
The \verb{estimateCombination*} functions use non-linear least squares
optimization (based on the package \code{basicdrm}) to estimate dose-response
parameters.  The \verb{estimateChou*} functions use the median-effect method
described by Chou and Talalay in their 1984 paper, which linearized all
measurements and performs linear regression.  We \emph{do not} recommend using
these methods, as they are much less reliable than the non-linear
optimization approach and extremely susceptible to noise at extreme values.
}
\examples{
surface <- synergisticExample
concs1 <- cbind(surface$concA, surface$concB)
act1 <- surface$measure

estimateCombinationIndices(concs1,act1,c(0.5))

dr1 <- surface[surface$concB==0, c("concA","measure")]
names(dr1) <- c("conc","act")
dr2 <- surface[surface$concA==0, c("concB","measure")]
names(dr2) <- c("conc","act")
drc <- surface[surface$concA==surface$concB,]
drc$conc <- drc$concA+drc$concB
drc <- drc[,c("conc","measure")]
names(drc) <- c("conc","act")

estimateChouIndex(dr1,dr2,drc,ratio=1,
                  level=c(0.5,0.9,0.99),
                  range=c(0,1))
}
\references{
Berenbaum, MC. 1978. “A Method for Testing for Synergy with Any Number of
Agents.” \emph{Journal of Infectious Diseases} \strong{137} (2): 122–30.

Berenbaum, MC. 1989. “What Is Synergy.” \emph{Pharmacol Rev} \strong{41}: 93–141.

Chou, Ting-Chao, and Paul Talalay. 1984. “Quantitative Analysis of
Dose-Effect Relationships: The Combined Effects of Multiple Drugs or Enzyme
Inhibitors.” \emph{Advances in Enzyme Regulation} \strong{22}: 27–55.
}
