% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/zicreg.R
\name{zic.reg}
\alias{zic.reg}
\title{Univariate zero-inflated Poisson and negative binomial regression models}
\usage{
zic.reg(
  fmla = NULL,
  data,
  dist = "pois",
  link.ct = "log",
  link.zi = "logit",
  optimizer = "nlm",
  starts = NULL,
  subset,
  na.action,
  weights = rep(1, length(y)),
  X = NULL,
  z = NULL,
  y = NULL,
  offset.ct = NULL,
  offset.zi = NULL,
  warn.parent = T,
  keep = F,
  ...
)
}
\arguments{
\item{fmla}{A \code{\link[stats]{formula}} of the form \verb{y ~ x_1 + x_2 + ... + x_n + offset(count_var) | z_1  + ... z_n + offset(zi_var)}, where the \code{x}
values are covariates in the count portion of the model, and \code{z} are in the
zero-inflation portion. The \code{z} and \code{x} variables can be the same. If \code{NULL},
design matrices, the response vector, and offsets can be entered directly; see
\code{X}, \code{z}, \code{y}, \code{offset.ct}, and \code{offset.zi} below.}

\item{data}{A \code{\link[base]{data.frame}} containing all variables
appearing in \code{fmla}, including offsets. If not specified, variables are
searched for in parent environment.}

\item{dist}{The distribution used for the count portion of the zero-inflated
mixture. One of \code{c("pois", "nbinom")}, partial matching supported.}

\item{link.ct}{String specifying the link function used for the count portion
of the mixture distribution. One of \code{c("log", "identity", "sqrt")}.
See \code{\link[stats]{family}}.}

\item{link.zi}{Character string specifying the link function used for the
zero-inflation portion of the mixture distribution. One of \code{c("logit", "probit", "cauchit", "log", "cloglog")}. See \code{\link[stats]{family}}.}

\item{optimizer}{String specifying the optimizer to be used for fitting, one
of \code{c("nlm", "optim")}. If \code{"optim"}, defaults to \code{method="BFGS"}.}

\item{starts}{Optional vector of starting values used for the numerical
optimization procedure. Should have count parameters first (with intercept
first, if applicable), followed by zero-inflated parameters (with intercept
first, if applicable), and the inverse dispersion parameter last (if
applicable).}

\item{subset}{Vector indicating the subset of observations on which to
estimate the model}

\item{na.action}{A function which indicates what should happen when the data
contain NAs. Default is \code{\link[stats]{na.omit}}.}

\item{weights}{An optional numeric vector of weights for each observation.}

\item{X, z}{If \code{fmla = NULL}, these are the design matrices of covariates for
the count and zero-inflation portions, respectively. Both require no
missingness. Similar in spirit to \code{\link[stats]{glm.fit}} in that it
can be faster for larger datasets because it bypasses model matrix
creation.}

\item{y}{If \code{fmla = NULL}, a vector containing the response variable.}

\item{offset.ct, offset.zi}{If \code{fmla = NULL}, vectors containing the
(constant) offset for the count and zero-inflated portions, respectively.
Must be equal in length to \code{y}, and row-dim of \code{X}, \code{z}. If left \code{NULL},
defaults to \code{rep(0, length(y))}.}

\item{warn.parent}{Logical indicating whether to warn about \code{data} not
being supplied.}

\item{keep}{Logical indicating whether to keep the model matrices in the
returned model object. Must be \code{TRUE} to use \code{\link[=DHARMa]{DHARMa}}
and \code{\link[texreg]{texreg}} with the model object, e.g., via
\code{\link{simulate.zicreg}} and \code{\link{extract.zicreg}}, as well as
base generics like \code{\link[stats]{fitted}} and
\code{\link[stats]{predict}}.}

\item{...}{Additional arguments to pass on to the chosen optimizer, either
\code{\link[stats]{nlm}} or \code{\link[stats]{optim}}. See 'Examples'.}
}
\value{
An S3 \code{\link{zicreg-class}} object, which is a list containing:
\itemize{
\item  \code{call} -- The original function call
\item  \code{obj} -- The class of the object
\item  \code{coef} -- Vector of coefficients, with count, then zi, then dispersion.
\item  \code{se} -- Vector of asymptotic standard errors
\item  \code{grad} -- Gradient vector at convergence
\item  \code{link.ct} -- Name of link used for count portion
\item  \code{link.zi} -- Name of link used for zero-inflated portion
\item  \code{dist} -- Name of distribution used for count portion
\item  \code{optimizer} -- Name of optimization package used in fitting
\item  \code{coefmat.ct} -- Coefficient matrix for count portion
\item  \code{coefmat.zi} -- Coefficient matrix for zero-inflated portion
\item  \code{convergence} -- Convergence code from optimization routine.
\item  \code{coefmat.all} -- Coefficient matrix for both parts of the model
\item  \code{theta} -- Coefficient matrix for dispersion, if applicable.
\item  \code{covmat} -- Asymptotic covariance matrix
\item  \code{nobs} -- Number of observations
\item  \code{aic} -- Akaike information
\item  \code{bic} -- Bayes information
\item  \code{loglik} -- Log-likelihood at convergence
\item  \code{model} -- List containing model matrices if \code{keep = TRUE}
}
}
\description{
This function from the \code{\link{bizicount}} package estimates
univariate zero-inflated Poisson and negative binomial regression models
via maximum likelihood using either the \code{\link[stats]{nlm}} or
\code{\link[stats]{optim}} optimization functions.  It's class has
associated \code{\link[stats]{simulate}} methods for post-estimation
diagnostics using the \code{\link[=DHARMa]{DHARMa}} package, as well as an
\code{\link[texreg]{extract}} method for printing professional tables using
\code{\link[texreg]{texreg}}. Visit the 'See Also' section for links to these
methods for \code{zicreg} objects.
}
\examples{
## ZIP example
# Simulate some zip data
n=1000
x = cbind(1, rnorm(n))
z = cbind(1, rbeta(n, 4, 8))
b = c(1, 2.2)
g = c(-1, 1.7)
lam = exp(x \%*\% b)
psi = plogis(z \%*\% g)

y = bizicount::rzip(n, lambda = lam, psi=psi)
dat = cbind.data.frame(x = x[,-1], z = z[,-1], y = y)

# estimate zip model using NLM, no data.frame

mod = zic.reg(y ~ x[,-1] | z[,-1])

# same model, with dataframe

mod = zic.reg(y ~ x | z, data = dat)


# estimate zip using NLM, adjust stepmax via ... param

mod = zic.reg(y ~ x[,-1] | z[,-1], stepmax = .5)


# estimate zip using optim

mod = zic.reg(y ~ x[,-1] | z[,-1], optimizer = "optim")


# pass different method, reltol to optim using ... param

mod = zic.reg(y ~ x[,-1] | z[,-1],
        optimizer = "optim",
        method = "Nelder-Mead",
        control = list(reltol = 1e-10)
        )

# No formula, specify design matrices and offsets.
zic.reg(y=y, X=x, z=z)



## ZINB example
# simulate zinb data

disp = .5
y = bizicount::rzinb(n, psi = psi, size = disp, mu=lam)


# zinb model, use keep = TRUE for post-estimation methods

mod = zic.reg(y ~ x[,-1] | z[,-1], dist = "n", keep = TRUE)

print(mod)
summary(mod)

}
\references{
Lambert, Diane. "Zero-inflated Poisson regression, with an
application to defects in manufacturing." Technometrics 34.1 (1992): 1-14.
}
\seealso{
\code{\link{simulate.zicreg}}, \code{\link{extract.zicreg}}
}
\author{
John Niehaus
}
