% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BIOMOD_presenceonly.R
\name{BIOMOD_presenceonly}
\alias{BIOMOD_presenceonly}
\title{evaluate models with presences only metrics}
\usage{
BIOMOD_presenceonly(
  modeling.output = NULL,
  EM.output = NULL,
  bg.env = NULL,
  perc = 0.9,
  save.output = T
)
}
\arguments{
\item{modeling.output}{"BIOMOD.models.out" object produced by a BIOMOD_Modeling run}

\item{EM.output}{a "BIOMOD.EnsembleModeling.out" returned by BIOMOD_EnsembleModeling}

\item{bg.env}{a data frame or matrix of environmental variables which was extracted from the background (might be used if presences should be compared to the background instead of Absences or Pseudo-Absences selected for modelling).}

\item{perc}{Percentage of correctly classified presences for MPA (Default 90\%).}

\item{save.output}{logical. If TRUE (Default) the output is saved to the ".BIOMOD_DATA" folder}
}
\value{
data.frame containing evaluation scores for the evaluation metrics used for 
the BIOMOD_Modeling function and additional Boyce index and MPA
}
\description{
This function enables to evaluate BIOMOD.models.out and 
BIOMOD.EnsembleModeling.out object with presence-only evaluation methods 
(Boyce index and Minimal Predicted Area MPA)
}
\details{
'em.by' of 'BIOMOD.EnsembleModeling' must be 'PA_dataset+repet' to have an 
ensemble for each RUN of the 'NbRunEval' argument (BIOMOD_Modeling funtion)
for evaluation.
The Boyce index returns NA values for 'SRE' models because it is not possible
to be calculated with binary predictions. This is also the reason why there 
are sometimes NA values for 'GLM' models if they don not converge.
}
\examples{
\dontrun{
requireNamesapce(PresenceAbsence, 'PresenceAbsence', quietly = TRUE)

# species occurrences
DataSpecies <- read.csv(system.file("external/species/mammals_table.csv",
                                    package="biomod2"), row.names = 1)
head(DataSpecies)

# the name of studied species
myRespName <- 'GuloGulo'

# the presence/absences data for our species 
myResp <- as.numeric(DataSpecies[,myRespName])

# the XY coordinates of species data
myRespXY <- DataSpecies[,c("X_WGS84","Y_WGS84")]


# Environmental variables extracted from BIOCLIM (bio_3, bio_4, bio_7, bio_11 & bio_12)
myExpl = stack( system.file( "external/bioclim/current/bio3.grd", 
                             package="biomod2"),
                system.file( "external/bioclim/current/bio4.grd", 
                             package="biomod2"), 
                system.file( "external/bioclim/current/bio7.grd", 
                             package="biomod2"),  
                system.file( "external/bioclim/current/bio11.grd", 
                             package="biomod2"), 
                system.file( "external/bioclim/current/bio12.grd", 
                             package="biomod2"))

# 1. Formatting Data
myBiomodData <- BIOMOD_FormatingData(resp.var = myResp,
                                     expl.var = myExpl,
                                     resp.xy = myRespXY,
                                     resp.name = myRespName)

# 2. Defining Models Options using default options.
myBiomodOption <- BIOMOD_ModelingOptions()

# 3. Doing Modelisation

myBiomodModelOut <- BIOMOD_Modeling( myBiomodData, 
                                     models = c('SRE','CTA','RF'), 
                                     models.options = myBiomodOption, 
                                     NbRunEval=1, 
                                     DataSplit=80, 
                                     Yweights=NULL, 
                                     VarImport=3, 
                                     models.eval.meth = c('TSS','ROC'),
                                     SaveObj = TRUE,
                                     rescal.all.models = FALSE,
                                     do.full.models = FALSE)

# 4. Doing Ensemble Modelling
myBiomodEM <- BIOMOD_EnsembleModeling( modeling.output = myBiomodModelOut,
                                       chosen.models = 'all',
                                       em.by = 'PA_dataset+repet',
                                       eval.metric = c('TSS'),
                                       eval.metric.quality.threshold = c(0.7),
                                       models.eval.meth = c('TSS','ROC'),
                                       prob.mean = TRUE,
                                       prob.cv = FALSE,
                                       prob.ci = FALSE,
                                       prob.ci.alpha = 0.05,
                                       prob.median = FALSE,
                                       committee.averaging = FALSE,
                                       prob.mean.weight = TRUE,
                                       prob.mean.weight.decay = 'proportional' )   

# evaluate Biomod models with the Boyce index and MPA
pres.only.eval <- BIOMOD_presenceonly(myBiomodModelOut, myBiomodEM)
pres.only.eval$eval

# evaluate Biomod models with the Boyce index and MPA using Background data
bg.Values <- getValues(myExpl)

pres.only.eval <- BIOMOD_presenceonly(myBiomodModelOut, myBiomodEM, bg.env = bg.Values)
pres.only.eval$eval
}
}
\references{
Engler, R., Guisan, A., and Rechsteiner L. 2004. An improved approach for predicting the distribution of rare and endangered species from occurrence and pseudo-absence data. Journal of Applied Ecology, 41(2), 263-274.
Hirzel, A. H., Le Lay, G., Helfer, V., Randin, C., and Guisan, A. 2006. Evaluating the ability of habitat suitability models to predict species presences. Ecological Modelling, 199(2), 142-152.
}
\seealso{
\code{\link[ecospat]{ecospat.boyce}}, \code{\link[ecospat]{ecospat.mpa}}, \code{\link[biomod2]{BIOMOD_Modeling}}, \code{\link[biomod2]{BIOMOD_EnsembleModeling}}
}
\author{
Frank Breiner \email{frank.breiner@wsl.ch}
}
