% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/propCI.R
\name{propCI}
\alias{propCI}
\title{Confidence intervals for one proportion in group testing}
\usage{
propCI(
  x,
  m,
  n,
  pt.method = "mle",
  ci.method,
  conf.level = 0.95,
  alternative = "two.sided",
  maxiter = 100,
  tol = .Machine$double.eps^0.5
)
}
\arguments{
\item{x}{integer specifying the number of positive groups when groups are 
of equal size, or a vector specifying the number of positive groups among 
the \kbd{n} groups tested when group sizes differ. If the latter, this 
vector must be of the same length as the \kbd{m} and \kbd{n} arguments.}

\item{m}{integer specifying the common size of groups when groups are of 
equal size, or a vector specifying the group sizes when group sizes differ. 
If the latter, this vector must be of the same length as the \kbd{x} and 
\kbd{n} arguments.}

\item{n}{integer specifying the number of groups when these groups are of 
equal size, or a vector specifying the corresponding number of groups of 
the sizes \kbd{m} when group sizes differ. If the latter, this vector must 
be of the same length as the \kbd{x} and \kbd{m} arguments.}

\item{pt.method}{character string specifying the point 
estimate to compute. Options include \kbd{"Firth"} for the 
bias-preventative, \kbd{"Gart"} and \kbd{"bc-mle"} for the 
bias-corrected MLE (where the latter allows for backward compatibility), 
and \kbd{"mle"} for the MLE.}

\item{ci.method}{character string specifying the confidence 
interval to compute. Options include "AC" for the Agresti-Coull interval, 
"bc-skew-score" for the bias- and skewness-corrected interval, "Blaker" 
for the Blaker interval, "CP" for the Clopper-Pearson interval, "exact" 
for the exact interval as given by Hepworth (1996), "lrt" for the 
likelihood ratio test interval, "score" for the Wilson score interval, 
"skew-score" for the skewness-corrected interval, "soc" for the 
second-order corrected interval, and "Wald" for the Wald interval. Note 
that the Agresti-Coull, Blaker, Clopper-Pearson, and second-order corrected 
intervals can only be calculated when \kbd{x}, \kbd{m}, and \kbd{n} are 
given as integers (equal group size case).}

\item{conf.level}{confidence level of the interval.}

\item{alternative}{character string defining the alternative 
hypothesis, either \kbd{"two.sided"}, \kbd{"less"}, or \kbd{"greater"}.}

\item{maxiter}{the maximum number of steps in the iteration of 
confidence limits, for use only with the \kbd{"exact"} method when 
group sizes differ.}

\item{tol}{the accuracy required for iterations in internal functions, 
for use with asymptotic intervals when group sizes differ only.}
}
\value{
A list containing:
\item{conf.int}{a confidence interval for the proportion.}
\item{estimate}{the point estimator of the proportion.}
\item{pt.method}{the method used for point estimation.}
\item{ci.method}{the method used for confidence interval estimation.}
\item{conf.level}{the confidence level of the interval.}
\item{alternative}{the alternative specified by the user.}
\item{x}{the number of positive groups.}
\item{m}{the group sizes.}
\item{n}{the numbers of groups with corresponding group sizes \kbd{m}.}
}
\description{
Calculates point estimates and confidence intervals for a 
single proportion with group testing data. Methods are available for groups 
of equal or different sizes.
}
\details{
Confidence interval methods include the Agresti-Coull 
(\kbd{ci.method="AC"}), bias- and skewness-corrected 
(\kbd{ci.method="bc-skew-score"}), Blaker (\kbd{ci.method="Blaker"}), 
Clopper-Pearson (\kbd{ci.method="CP"}), exact (\kbd{ci.method="exact"}), 
likelihood ratio test (\kbd{ci.method="lrt"}), Wilson score 
(\kbd{ci.method="score"}), skewness-corrected 
(\kbd{ci.method="skew-score"}), second-order corrected 
(\kbd{ci.method="soc"}), and Wald 
(\kbd{ci.method="Wald"}) intervals. The Agresti-Coull, Blaker, 
Clopper-Pearson, and second-order corrected intervals are available 
only for the equal group size case.

Point estimates available include the MLE (\kbd{pt.method="mle"}), 
bias-corrected MLE (\kbd{pt.method="Gart"} or \kbd{pt.method="bc-mle"}), 
and bias-preventative (\kbd{pt.method="Firth"}). Only the MLE method 
is available when calculating the Clopper-Pearson, Blaker, Agresti-Coull, 
second-order corrected, or exact intervals.

\subsection{Equal group sizes}{
Computation of confidence intervals for group testing with equal group 
sizes are described in Tebbs & Bilder (2004) and Schaarschmidt (2007).} 

\subsection{Unequal group sizes}{
While the exact method is available when group sizes differ, 
the algorithm becomes computationally very expensive if the number of 
different groups, \kbd{n}, becomes larger than three. See Hepworth (1996) 
for additional details on the exact method and other methods for 
constructing confidence intervals in group testing situations. For 
computational details and simulation results of the remaining methods, 
see Biggerstaff (2008). See Hepworth & Biggerstaff (2017) for 
recommendations on the best point estimator methods.}
}
\examples{
# Example from Tebbs and Bilder (2004):
#   3 groups out of 24 test positively; 
#   each group has a size of 7.
# Clopper-Pearson interval:
propCI(x=3, m=7, n=24, ci.method="CP", 
       conf.level=0.95, alternative="two.sided")
      
# Blaker interval:
propCI(x=3, m=7, n=24, ci.method="Blaker", 
       conf.level=0.95, alternative="two.sided")
      
# Wilson score interval: 
propCI(x=3, m=7, n=24, ci.method="score", 
       conf.level=0.95, alternative="two.sided")
      
# One-sided Clopper-Pearson interval:
propCI(x=3, m=7, n=24, ci.method="CP", 
       conf.level=0.95, alternative="less")

# Calculate confidence intervals with a group size of 1. 
#   These match those found using the binom.confint() 
#   function from the binom package.
propCI(x = 4, m = 1, n = 10, pt.method = "mle", 
       ci.method = "AC")
propCI(x = 4, m = 1, n = 10, pt.method = "mle", 
       ci.method = "score")
propCI(x = 4, m = 1, n = 10, pt.method = "mle", 
       ci.method = "Wald")

# Example from Hepworth (1996, table 5):
#   1 group out of 2 tests positively with 
#   groups of size 5; also, 
#   2 groups out of 3 test positively with 
#   groups of size 2.
propCI(x=c(1,2), m=c(5,2), n=c(2,3), ci.method="exact") 

# Recalculate the example given in
#   Hepworth (1996), table 5:
propCI(x=c(0,0), m=c(5,2), n=c(2,3), ci.method="exact")
propCI(x=c(0,1), m=c(5,2), n=c(2,3), ci.method="exact")
propCI(x=c(0,2), m=c(5,2), n=c(2,3), ci.method="exact")
propCI(x=c(0,3), m=c(5,2), n=c(2,3), ci.method="exact")
propCI(x=c(1,0), m=c(5,2), n=c(2,3), ci.method="exact")
propCI(x=c(1,1), m=c(5,2), n=c(2,3), ci.method="exact")
propCI(x=c(1,2), m=c(5,2), n=c(2,3), ci.method="exact")
propCI(x=c(1,3), m=c(5,2), n=c(2,3), ci.method="exact")
propCI(x=c(2,0), m=c(5,2), n=c(2,3), ci.method="exact")
propCI(x=c(2,1), m=c(5,2), n=c(2,3), ci.method="exact")
propCI(x=c(2,2), m=c(5,2), n=c(2,3), ci.method="exact")
propCI(x=c(2,3), m=c(5,2), n=c(2,3), ci.method="exact")

# Example with multiple groups of various sizes: 
#   0 out of 5 groups test positively with 
#   groups of size 1 (individual testing);
#   0 out of 5 groups test positively with 
#   groups of size 5;
#   1 out of 5 groups test positively with 
#   groups of size 10; and
#   2 out of 5 groups test positively with 
#   groups of size 50.
x1 <- c(0,0,1,2)
m1 <- c(1,5,10,50)
n1 <- c(5,5,5,5)
propCI(x=x1, m=m1, n=n1, pt.method="Gart", 
       ci.method="skew-score")
propCI(x=x1, m=m1, n=n1, pt.method="Gart", 
       ci.method="score")

# Reproducing estimates from Table 1 in
#   Hepworth & Biggerstaff (2017):
propCI(x=c(1,2), m=c(20,5), n=c(8,8), 
       pt.method="Firth", ci.method="lrt")
propCI(x=c(7,8), m=c(20,5), n=c(8,8), 
       pt.method="Firth", ci.method="lrt")
}
\references{
\insertRef{Biggerstaff2008}{binGroup2}
  
\insertRef{Hepworth1996}{binGroup2}

\insertRef{Hepworth2017}{binGroup2}

\insertRef{Schaarschmidt2007}{binGroup2}

\insertRef{Tebbs2004}{binGroup2}
}
\seealso{
\code{\link{propDiffCI}} for confidence intervals for the 
difference of proportions in group testing, \code{\link{gtTest}} for 
hypothesis tests in group testing, \code{\link{gtPower}} for power 
calculations in group testing, and \code{\link{binom.test}} for an exact 
confidence interval and test.

Other estimation functions: 
\code{\link{designEst}()},
\code{\link{designPower}()},
\code{\link{gtPower}()},
\code{\link{gtTest}()},
\code{\link{gtWidth}()},
\code{\link{propDiffCI}()}
}
\author{
This function is a combination of \code{bgtCI} and \code{bgtvs} 
written by Frank Schaarschmidt and \code{pooledBin} written by Brad 
Biggerstaff for the \code{binGroup} package. Minor modifications have been 
made for inclusion of the functions in the \code{binGroup2} package.
}
\concept{estimation functions}
