% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/STCAR_INLA.R
\name{STCAR_INLA}
\alias{STCAR_INLA}
\title{Fit a (scalable) spatio-temporal Poisson mixed model to areal count data.}
\usage{
STCAR_INLA(
  carto = NULL,
  data = NULL,
  ID.area = NULL,
  ID.year = NULL,
  ID.group = NULL,
  O = NULL,
  E = NULL,
  X = NULL,
  W = NULL,
  spatial = "Leroux",
  temporal = "rw1",
  interaction = "TypeIV",
  model = "partition",
  k = 0,
  strategy = "simplified.laplace",
  scale.model = NULL,
  PCpriors = FALSE,
  merge.strategy = "original",
  compute.intercept = NULL,
  compute.DIC = TRUE,
  n.sample = 1000,
  compute.fitted.values = FALSE,
  save.models = FALSE,
  plan = "sequential",
  workers = NULL,
  inla.mode = "classic",
  num.threads = NULL
)
}
\arguments{
\item{carto}{object of class \code{SpatialPolygonsDataFrame} or \code{sf}. This object must contain at least the variable with the identifiers of the spatial areal units specified in the argument \code{ID.area}.}

\item{data}{object of class \code{data.frame} that must contain the target variables of interest specified in the arguments \code{ID.area}, \code{ID.year}, \code{O} and \code{E}.}

\item{ID.area}{character; name of the variable that contains the IDs of spatial areal units. The values of this variable must match those given in the \code{carto} and \code{data} variable.}

\item{ID.year}{character; name of the variable that contains the IDs of time points.}

\item{ID.group}{character; name of the variable that contains the IDs of the spatial partition (grouping variable). Only required if \code{model="partition"}.}

\item{O}{character; name of the variable that contains the observed number of disease cases for each areal and time point.}

\item{E}{character; name of the variable that contains either the expected number of disease cases or the population at risk for each areal unit and time point.}

\item{X}{a character vector containing the names of the covariates within the \code{data} object to be included in the model as fixed effects, or a matrix object playing the role of the fixed effects design matrix.
If \code{X=NULL} (default), only a global intercept is included in the model as fixed effect.}

\item{W}{optional argument with the binary adjacency matrix of the spatial areal units. If \code{NULL} (default), this object is computed from the \code{carto} argument (two areas are considered as neighbours if they share a common border).}

\item{spatial}{one of either \code{"Leroux"} (default), \code{"intrinsic"}, \code{"BYM"} or \code{"BYM2"}, which specifies the prior distribution considered for the spatial random effect.}

\item{temporal}{one of either \code{"rw1"} (default) or \code{"rw2"}, which specifies the prior distribution considered for the temporal random effect.}

\item{interaction}{one of either \code{"none"}, \code{"TypeI"}, \code{"TypeII"}, \code{"TypeIII"} or \code{"TypeIV"} (default), which specifies the prior distribution for the space-time interaction random effect.}

\item{model}{one of either \code{"global"} or \code{"partition"} (default), which specifies the \emph{Global model} or one of the scalable model proposal's (\emph{Disjoint model} and \emph{k-order neighbourhood model}, respectively).}

\item{k}{numeric value with the neighbourhood order used for the partition model. Usually k=2 or 3 is enough to get good results. If k=0 (default) the \emph{Disjoint model} is considered. Only required if \code{model="partition"}.}

\item{strategy}{one of either \code{"gaussian"}, \code{"simplified.laplace"} (default), \code{"laplace"} or \code{"adaptive"}, which specifies the approximation strategy considered in the \code{inla} function.}

\item{scale.model}{logical value (default \code{NULL}); if \code{TRUE} the structure matrices of the model are scaled so their generalized variances are equal to 1.
It only works if arguments \code{spatial="intrinsic"} or \code{spatial="BYM2"} are specified.}

\item{PCpriors}{logical value (default \code{FALSE}); if \code{TRUE} then penalised complexity (PC) priors are used for the precision parameter of the spatial random effect.
It only works if arguments \code{spatial="intrinsic"} or \code{spatial="BYM2"} are specified.}

\item{merge.strategy}{one of either \code{"mixture"} or \code{"original"} (default), which specifies the merging strategy to compute posterior marginal estimates of the linear predictor. See \code{\link{mergeINLA}} for further details.}

\item{compute.intercept}{CAUTION! This argument is deprecated from version 0.5.2.}

\item{compute.DIC}{logical value; if \code{TRUE} (default) then approximate values of the Deviance Information Criterion (DIC) and Watanabe-Akaike Information Criterion (WAIC) are computed.}

\item{n.sample}{numeric; number of samples to generate from the posterior marginal distribution of the linear predictor when computing approximate DIC/WAIC values. Default to 1000.}

\item{compute.fitted.values}{logical value (default \code{FALSE}); if \code{TRUE} transforms the posterior marginal distribution of the linear predictor to the exponential scale (risks or rates). CAUTION: This method might be time consuming.}

\item{save.models}{logical value (default \code{FALSE}); if \code{TRUE} then a list with all the \code{inla} submodels is saved in '/temp/' folder, which can be used as input argument for the \code{\link{mergeINLA}} function.}

\item{plan}{one of either \code{"sequential"} or \code{"cluster"}, which specifies the computation strategy used for model fitting using the 'future' package.
If \code{plan="sequential"} (default) the models are fitted sequentially and in the current R session (local machine). If \code{plan="cluster"} the models are fitted in parallel on external R sessions (local machine) or distributed in remote computing nodes.}

\item{workers}{character or vector (default \code{NULL}) containing the identifications of the local or remote workers where the models are going to be processed. Only required if \code{plan="cluster"}.}

\item{inla.mode}{one of either \code{"classic"} (default) or \code{"compact"}, which specifies the approximation method used by INLA. See \code{help(inla)} for further details.}

\item{num.threads}{maximum number of threads the inla-program will use. See \code{help(inla)} for further details.}
}
\value{
This function returns an object of class \code{inla}. See the \code{\link{mergeINLA}} function for details.
}
\description{
Fit a spatio-temporal Poisson mixed model to areal count data, where several CAR prior distributions for the spatial random effects, first and second order random walk priors for the temporal random effects, and different types of spatio-temporal interactions described in \insertCite{knorrheld2000;textual}{bigDM} can be specified.
The linear predictor is modelled as \deqn{\log{r_{it}}=\alpha + \mathbf{x_{it}}^{'}\mathbf{\beta} + \xi_i+\gamma_t+\delta_{it}, \quad \mbox{for} \quad i=1,\ldots,n; \quad t=1,\ldots,T}
where \eqn{\alpha} is a global intercept, \eqn{\mathbf{x_{it}}^{'}=(x_{it1},\ldots,x_{itp})} is a p-vector of standardized covariates in the i-th area and time period t, \eqn{\mathbf{\beta}=(\beta_1,\ldots,\beta_p)} is the p-vector of fixed effects coefficients,
\eqn{\xi_i} is a spatially structured random effect, \eqn{\gamma_t} is a temporally structured random effect, and \eqn{\delta_{it}} is the space-time interaction effect. If the interaction term is dropped, an additive model is obtained.
To ensure model identifiability, sum-to-zero constraints are imposed over the random effects of the model. Details on the derivation of these constraints can be found in \insertCite{goicoa2018spatio;textual}{bigDM}.
\cr\cr
As in the \code{\link{CAR_INLA}} function, three main modelling approaches can be considered:
\itemize{
\item the usual model with a global spatial random effect whose dependence structure is based on the whole neighbourhood graph of the areal units (\code{model="global"} argument)
\item a Disjoint model based on a partition of the whole spatial domain where independent spatial CAR models are simultaneously fitted in each partition (\code{model="partition"} and \code{k=0} arguments)
\item a modelling approach where \emph{k}-order neighbours are added to each partition to avoid border effects in the Disjoint model (\code{model="partition"} and \code{k>0} arguments).
}
For both the Disjoint and k-order neighbour models, parallel or distributed computation strategies can be performed to speed up computations by using the 'future' package \insertCite{bengtsson2020unifying}{bigDM}.

Inference is conducted in a fully Bayesian setting using the integrated nested Laplace approximation (INLA; \insertCite{rue2009approximate;textual}{bigDM}) technique through the R-INLA package (\url{https://www.r-inla.org/}).
For the scalable model proposals \insertCite{orozco2022}{bigDM}, approximate values of the Deviance Information Criterion (DIC) and Watanabe-Akaike Information Criterion (WAIC) can also be computed.

The function allows also to use the new hybrid approximate method that combines the Laplace method with a low-rank Variational Bayes correction to the posterior mean \insertCite{vanNiekerk2023}{bigDM} by including the \code{inla.mode="compact"} argument.
}
\details{
For a full model specification and further details see the vignettes accompanying this package.
}
\examples{
\dontrun{
if(require("INLA", quietly=TRUE)){

  ## Load the sf object that contains the spatial polygons of the municipalities of Spain ##
  data(Carto_SpainMUN)
  str(Carto_SpainMUN)

  ## Create province IDs ##
  Carto_SpainMUN$ID.prov <- substr(Carto_SpainMUN$ID,1,2)

  ## Load simulated data of lung cancer mortality data during the period 1991-2015 ##
  data("Data_LungCancer")
  str(Data_LungCancer)

  ## Disjoint model with a BYM2 spatial random effect, RW1 temporal random effect and      ##
  ## Type I interaction random effect using 4 local clusters to fit the models in parallel ##
  Disjoint <- STCAR_INLA(carto=Carto_SpainMUN, data=Data_LungCancer,
                         ID.area="ID", ID.year="year", O="obs", E="exp", ID.group="ID.prov",
                         spatial="BYM2", temporal="rw1", interaction="TypeI",
                         model="partition", k=0, strategy="gaussian",
                         plan="cluster", workers=rep("localhost",4))
  summary(Disjoint)

  ## 1st-order nb. model with a BYM2 spatial random effect, RW1 temporal random effect and ##
  ## Type I interaction random effect using 4 local clusters to fit the models in parallel ##
  order1 <- STCAR_INLA(carto=Carto_SpainMUN, data=Data_LungCancer,
                       ID.area="ID", ID.year="year", O="obs", E="exp", ID.group="ID.prov",
                       spatial="BYM2", temporal="rw1", interaction="TypeI",
                       model="partition", k=1, strategy="gaussian",
                       plan="cluster", workers=rep("localhost",4))
  summary(order1)
}
}

}
\references{
\insertRef{goicoa2018spatio}{bigDM}

\insertRef{knorrheld2000}{bigDM}

\insertRef{orozco2020}{bigDM}

\insertRef{orozco2022}{bigDM}

\insertRef{vanNiekerk2023}{bigDM}
}
