% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/autoimage.R
\name{autoimage}
\alias{autoimage}
\title{Create an image plot with a legend scale.}
\usage{
autoimage(x, y, z, col = heat.colors(12), legend = TRUE,
  horizontal = TRUE, common.legend = TRUE, size = c(1, 1), mratio = 5,
  project = FALSE, map.grid = FALSE, mmar, legend.mar, axis.args,
  project.args, grid.args, map.poly, poly.args, ...)
}
\arguments{
\item{x, y}{Locations of grid lines at which the values in \code{z} are measured.  The values must be finite and non-missing.  These arguments can be either vectors, in which case the values must be in (strictly) ascending order.  If an irregular grid is to be used, then \code{x} and \code{y} should be numeric matrices having the same number of rows and columns as \code{z}.  If these arguments are not provided, equally spaced values from 0 to 1 are used by default. If \code{x} is a list, its components \code{x$x} and \code{x$y} are used for \code{x} and \code{y}, respectively. If the list has component \code{x$z}, this is used for \code{z}.}

\item{z}{A numeric or logical matrix containing the values to be plotted (NAs are allowed). If multiple images are to be plotted, a numeric array can be provided instead.  The third dimension of the array indicates the number of images that should be plotted.  Note that \code{x} can be used instead of \code{z} for convenience.}

\item{col}{A list of colors such as that generated by \code{\link[grDevices]{rainbow}}, \code{\link[grDevices]{heat.colors}}, \code{\link[grDevices]{topo.colors}}, \code{\link[grDevices]{terrain.colors}} or similar functions.  The default is given by \code{heat.colors(12)}.}

\item{legend}{A logical value indicating whether a legend scale should be added to the plot.  Default is \code{TRUE}.}

\item{horizontal}{A logical value indicating whether the legend scale should be horizontal or vertical.  Default is \code{TRUE}, indicating the legend scale will be on the bottom of the plot and the colors will run from left to right.  If \code{FALSE}, the legend scale will be on the right side of the plot and the colors will run vertically.}

\item{common.legend}{A logical value indicating whether a common legend scale should be used for all images provided in the \code{z} array.  Default is \code{TRUE}.}

\item{size}{A vector of length two indicating the number of rows and columns that should be used for the series of image data in \code{z}.  Note that \code{prod(size)} must match the length of the third dimension of \code{z} (if it is an array), or \code{c(1, 1)} if \code{z} is a matrix.}

\item{mratio}{A numeric value indicating the ratio of the width of each image to the width of the legend scale.  Default is \code{mratio = 5}.}

\item{project}{A logical value indicating whether the \code{x} and \code{y} coordinates should be projected before plotting.  This only makes sense if the values in \code{x} and \code{y} are longitude and latitude coordinates.  Default is \code{FALSE}.  If \code{TRUE}, then the \code{\link[mapproj]{mapproject}} function is used to project the coordinates.  If \code{TRUE}, then additional arguments for \code{\link[mapproj]{mapproject}} should be provided in \code{project.args}.}

\item{map.grid}{A logical value indicating whether a map grid should be added to the current plot using the \code{\link[mapproj]{map.grid}} function.  This will only be sensible when \code{project = TRUE}.  See Details.  Additional arguments to \code{\link[mapproj]{map.grid}} can be provided as a list to \code{grid.args}.}

\item{mmar}{A vector of length four indicating the margins of each image in the plot.  This is passed internally to the \code{mar} argument of the \code{\link[graphics]{par}} function.  See the \code{\link[graphics]{par}} function for more details.  The default is the currently specified \code{mar} argument of the \code{par} function.}

\item{legend.mar}{A vector of length four indicating the margins of the legend scale.  This is passed internally to the \code{mar} argument of the \code{\link[graphics]{par}} function.  See the \code{\link[graphics]{par}} function for more details.  The default is sensible values based on the value of \code{mmar}.}

\item{axis.args}{A list with arguments matching the arguments of the \code{\link[graphics]{axis}} function that is used to customize the legend scale.  See \code{\link{legend.scale}} for examples.}

\item{project.args}{A list with arguments matching the non \code{x} and \code{y} arguments of the \code{\link[mapproj]{mapproject}} function.}

\item{grid.args}{A list with arguments matching the non \code{lim} arguments of the \code{\link[mapproj]{map.grid}} function.  This is used to customize the plotted grid when \code{map.grid = TRUE}.}

\item{map.poly}{A list with named elements \code{x} and \code{y} that are used to plot relevent polygons on each image using the \code{\link[graphics]{lines}} function.}

\item{poly.args}{A list with arguments matching those provided to the \code{\link[graphics]{lines}} function used to plot \code{map.poly}.  This would be used to customize the lines, e.g., with different thickness, type, or color.}

\item{...}{Additional arguments passed to the \code{\link[graphics]{image}} or \code{\link[fields]{poly.image}} functions.  e.g., \code{xlab}, \code{ylab}, \code{xlim}, \code{ylim}, \code{zlim}, etc.}
}
\description{
\code{autoimage} creates an image plot while also automatically plotting a legend that indicates the correspondence between the colors and the values of the \code{z} variable.  \code{autoimage} is intended to be backwards compatible with the \code{image} function, but no promises are made.  Additionally, one can plot multiple images in one graphic using this function, both with and without a common scale for the images.  Perhaps more importantly, the \code{\link[fields]{poly.image}} function from the \code{fields} package is used to provide image plots of data on an irregular grid (e.g., data measured at longitude/latitude coordinates).
}
\details{
When \code{project = TRUE}, the \code{\link[mapproj]{mapproject}} function is used to project the \code{x} and \code{y} coordinates.  In that case, the projected \code{x} and \code{y} coordinates of the plotted image are unlikely to be similar to the original values.  It is recommended that the user sets \code{axes = FALSE} since the x and y axis scales will not be interpretable.  However, the axes will still be useful for scaling purposes using \code{xlim} and \code{ylim}.  If reference axes are still desired, set \code{map.grid = TRUE}, in which case the \code{\link[mapproj]{map.grid}} function is used to draw correct longitude and latitude grid lines.

If multiple images are to be plotted (i.e., if \code{z} is an array), then the \code{main} argument can be a vector with length matching \code{dim(z)[3]}, and each successive element of the vector will be used to add a title to each successive image plotted.  See the Examples.

Note that the more images that are plotted simulataneously, the smaller one typically wants \code{mratio} to be.

The multiple plots are constructed using the \code{\link[graphics]{layout}} function, which is incompatible with the \code{mfrow} and \code{mfcol} arguments in the \code{\link[graphics]{par}} function and is also incompatible with the \code{\link[graphics]{split.screen}} function.
}
\examples{
# Example from image function documentation
x <- y <- seq(-4*pi, 4*pi, len = 27)
r <- sqrt(outer(x^2, y^2, "+"))
z <- cos(r^2)*exp(-r/6)
image(z, col  = gray((0:32)/32))
autoimage(z, col  = gray((0:32)/32), legend = FALSE)

# now with legend
autoimage(z, col  = gray((0:32)/32))
autoimage(z, col  = gray((0:32)/32), horizontal = FALSE)

# add some customization
autoimage(x, y, z, xlab = "x1", ylab = "y1", main = "Math is beautiful ...")

# now do some examples with multiple images
z2 <- cos(r^2/2)*exp(-r/3)
z3 <- cos(r^2/2)*exp(-r/6)
z4 <- cos(r^2/3)*exp(-r/5)
z5 <- cos(r^2/4)*exp(-r/2)
z6 <- cos(r^2/5)*exp(-r)
library(abind)
zarray <- abind(z, z2, z3, z4, z5, z6, along = 3)

# multiple images with common scale, separate titles
autoimage(x, y, zarray, main = letters[1:6], size = c(3, 2),
          mratio = 2, mmar = c(2.1, 4.1, 2.1, 2.1))
# change the orientation of the scale
autoimage(x, y, zarray, main = letters[1:6], size = c(3, 2),
          mratio = 3, mmar = c(4.1, 4.1, 2.1, 2.1),
          horizontal = FALSE)

# do some examples with an irregular grid
# load data from fields package
data("RCMexample", package = "fields")
# restructure data with 2, 4, and 6 images
RCMexample2 = RCMexample
RCMexample2$z = RCMexample2$z[,,1:2]
RCMexample4 = RCMexample
RCMexample4$z = RCMexample4$z[,,1:4]

# plot irregularly gridded images
autoimage(RCMexample2, col = fields::tim.colors(12), size = c(2, 1),
          mmar = c(2.1, 2.1, 1, 2.1), horizontal = FALSE)
autoimage(RCMexample2, col = fields::tim.colors(12), size = c(1, 2))

# Do the same plot, but with a projection.
# Notice that the axes scales seem off because of the projection
autoimage(RCMexample2, col = fields::tim.colors(12), size = c(1, 2),
          project = TRUE,
          project.args = list(projection = "albers", parameters = c(33, 45)))
# compare the axes for the projected coordinates to the correct references lines using map.grid.
autoimage(RCMexample2, col = fields::tim.colors(12), size = c(1, 2),
          project = TRUE, map.grid = TRUE,
          project.args = list(projection = "albers", parameters = c(33, 45)),
          grid.args = list(col = "yellow", nx = 5, ny = 5))

# more images in a plot
autoimage(RCMexample4, col = fields::tim.colors(12), size = c(2, 2), horizontal = FALSE, mratio = 3)
autoimage(RCMexample4, col = fields::tim.colors(12), size = c(2, 2), mratio = 4)
# adjusting the x axis spacing (without using a projection)
autoimage(RCMexample4, col = fields::tim.colors(12), size = c(2, 2),
          xaxp = c(-160, -40, 8), mratio = 4)

# add a nice polygon to the images
library(maps) # need to get world map
# get the polygon for the world from the maps package
worldpoly = map("world", plot = FALSE)
# project and plot two images, no axes,
# with polygon of national boundaries and map.grid.
# This plot is a bit busy
autoimage(RCMexample2, size = c(1, 2), project = TRUE, map.grid = TRUE,
          project.args = list(projection = "albers", parameters = c(33, 45)),
          map.poly = worldpoly, axes = FALSE,
          mmar = c(0.5, 0.5, 0.5, 0.5), legend.mar = c(2, 0.5, 0.5, 0.5))

# some plots without a common legend
autoimage(RCMexample2, size = c(1, 2), project = TRUE,
          map.grid = TRUE, axes = FALSE, common.legend = FALSE,
          horizontal = FALSE, legend.mar = c(5.1, 0, 4.1, 3.1),
          project.args = list(projection = "albers", parameters = c(33, 45)),
          map.poly = worldpoly, mratio = 4)
autoimage(RCMexample2, size = c(1, 2), project = TRUE, map.grid = TRUE,
          project.args = list(projection = "albers", parameters = c(33, 45)),
          map.poly = worldpoly, axes = FALSE, common.legend = FALSE,
          legend.mar = c(3.1, 4.1, 0, 2.1))
}
\references{
The code for this function is derived from the internals of the \code{\link[fields]{image.plot}} function written by Doug Nychka and from \code{image.scale.2} function written by Marc Taylor and discussed at \code{http://menugget.blogspot.com/2013/12/new-version-of-imagescale-function.html}.
}
\seealso{
\code{\link[graphics]{image}}, \code{\link[fields]{image.plot}}, \code{\link[graphics]{axis}}
}

