\encoding{UTF-8}
\name{Arnhofen}
\alias{Arnhofen}
\docType{data}
\title{
Point pattern of mining pits from the Neolithic chert mine at Arnhofen
}
\description{
The list object contains two data tables of coordinates, one representing the centers of round mining pits, the other holding the vertices of the observed polygonal area. Data was generated by G. Roth in 2006 (Roth 2008). For converting the data to a point pattern see \strong{Examples}. Spatial distance unit is meter.
}
\usage{data(Arnhofen)}
\format{
  A list with two entries. The first is a data frame, \code{points}, with 216 observations of \code{x} and \code{y} coordinates. Each line represents the center of a round mining pit. The second is a data frame, \code{window},  with 100 observations and 3 variables, \code{x}, \code{y}, and vertex \code{ID}. Each line represents a vertex of the polygonal observation area. 
  \describe{
    \item{\code{points$x}}{(num) x coordinates of pit centers in m}
	\item{\code{points$y}}{(num) x coordinates of pit centers in m}
    \item{\code{window$x}}{(num) x coordinates of observation window vertices in m}
	\item{\code{window$y}}{(num) x coordinates of observation window vertices in m}
	\item{\code{window$id}}{(num) id for observation window vertices}
  }
}
\details{
The coordinates in dataframe \code{points} represent the centres of 216 mining pits in the southeastern section of the 2001 excavation at the Neolithic chert mine of Abensberg-Arnhofen (Roth 2008). Direct dates for mining from the excavation place the site at 5300-4200 BC while use of mined material ends with the Bell Beaker Culture suggesting an end date for the mine of ca. 2200 BC. The regular pit pattern presented here dates to about 4200 BC, i.e. to the Münchshöfen Culture (4500-4000 BC). Arnhofen is the largest neolithic mine in Germany. The mining organization was analyzed by Roth (2008) using point pattern analysis (cf. Baddeley et al. 2016) which showed the neolithic mining to be conducted by farmers from surrounding villages (presumably on a seasonal basis). 

The pit centers were located manually in a GIS using excavation maps from between 3 to 4 m below present surface. Mining pits were similar to vertical tubes with an average diameter of 1 m and a rounded horizontal section. A point therefore represents the center of such a vertical tube in the horizontal plane. A few of the pits reached a depth of nearly 8 m below surface. The vertices of the observation area polygon circumscribe a slightly smaller region than the excavated area. The list contains the additional attributes: reference for the data, short data description (site) and geographical coordinates (Lat/Lon) of the excavation.
}
\source{
Roth, G. 2008. \emph{Geben und Nehmen, Eine Wirtschaftshistorische Studie zum Neolithischen Hornsteinbergbau von Abensberg-Arnhofen, Kr. Kelheim (Niederbayern)} [in 4 volumes]. online PhD-thesis, University of Cologne 2008. \url{http://kups.ub.uni-koeln.de/4176}.
}

\references{
Baddeley, A., E. Rubak and R. Turner. 2016. \emph{Spatial Point Patterns: Methodology and Applications with R}. CRC Press. Boca Raton.
}
\examples{
# data and package spatstat by A. Baddeley et al. 2016 for point pattern analysis
library(archdata)
data(Arnhofen)
# package spatstat is described and illustrated in Baddeley et al. (2016)
library(spatstat)
ap <- Arnhofen	      # to shorten the following code

# generate observation window object; note the polygonal outline.
arnwin <- owin(poly=ap$window[, 1:2])	

# generate point process pattern object from points and owin object
app <- ppp(ap$points$x, ap$points$y, arnwin) 
unitname(app) <- c("metre", "metres")   # optional, asign unitnames

# note that owin vertices traverse the polygon anticlockwise
plot(arnwin)
points(ap$window[, 1:2], pch=3, cex=.5)
text(ap$window[, 1], ap$window[, 2], ap$window[, 3], pos=3, cex=.7)

# visual inspection of the point process pattern 
plot(app)

# Computing the summary function "centered Besag's L" assuming
# homogeneous intensity. Centered Besag's L is just a conveniently 
# transformed Ripley's K. see references in ?Lest.

set.seed(1)	
Lcentrd <- envelope(app, Lest, nsim=99, nrank=1, global=TRUE, r=seq(0,7, 0.01),
     correction="translate", transform=expression(.-r))
# for the arguments see ?Kest and ?envelope. 

tm <- "Centered Besags's L for Arnhofen-Southeast"  # title

plot(Lcentrd, legendpos="bottomright", legendargs=list(bg="white"),
     main=tm, las=1)

# The deviations below envelopes suggest regular inter point distances
# at the 1% level - deviations above would have suggested clustering
# with r representing the radius of round clusters.

plot(Lcentrd, xlim=c(.5,2), legendpos="topright", legendargs=list(bg="white"),
     las=1, main=tm)
(inhibr <- Lcentrd$r[Lcentrd$obs<Lcentrd$lo])	

# significant inhibition between pits at distances of
# With an average diameter of 1 m pits were spaced at regular 
# distances up to about 0.7 m apart:

max(inhibr) - 1

citation("spatstat")      # don't forget to reference the method.
}
\keyword{datasets}
