% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/orsf_predict.R
\name{predict.orsf_fit}
\alias{predict.orsf_fit}
\title{Compute predictions using ORSF}
\usage{
\method{predict}{orsf_fit}(
  object,
  new_data,
  pred_horizon = NULL,
  pred_type = "risk",
  na_action = "fail",
  boundary_checks = TRUE,
  ...
)
}
\arguments{
\item{object}{(\emph{orsf_fit}) a trained oblique random survival forest
(see \link{orsf}).}

\item{new_data}{a \link{data.frame}, \link[tibble:tibble-package]{tibble}, or \link[data.table:data.table]{data.table} to compute predictions in.}

\item{pred_horizon}{(\emph{double}) a value or vector indicating the time(s)
that predictions will be calibrated to. E.g., if you were predicting
risk of incident heart failure within the next 10 years, then
\code{pred_horizon = 10}. \code{pred_horizon} can be \code{NULL} if \code{pred_type} is
\code{'mort'}, since mortality predictions are aggregated over all
event times}

\item{pred_type}{(\emph{character}) the type of predictions to compute. Valid
options are
\itemize{
\item 'risk' : probability of having an event at or before \code{pred_horizon}.
\item 'surv' : 1 - risk.
\item 'chf': cumulative hazard function
\item 'mort': mortality prediction
}}

\item{na_action}{(\emph{character}) what should happen when \code{new_data} contains missing values (i.e., \code{NA} values). Valid options are:
\itemize{
\item 'fail' : an error is thrown if \code{new_data} contains \code{NA} values
\item 'pass' : the output will have \code{NA} in all rows where \code{new_data} has 1 or more \code{NA} value for the predictors used by \code{object}
\item 'omit' : rows in \code{new_data} with incomplete data will be dropped
\item 'impute_meanmode' : missing values for continuous and categorical variables in \code{new_data} will be imputed using the mean and mode, respectively. To clarify,
the mean and mode used to impute missing values are from the
training data of \code{object}, not from \code{new_data}.
}}

\item{boundary_checks}{(\emph{logical}) if \code{TRUE}, \code{pred_horizon} will be
checked to make sure the requested values are less than the maximum
observed time in \code{object}'s training data. If \code{FALSE}, these checks
are skipped.}

\item{...}{Further arguments passed to or from other methods (not currently used).}
}
\value{
a \code{matrix} of predictions. Column \code{j} of the matrix corresponds
to value \code{j} in \code{pred_horizon}. Row \code{i} of the matrix corresponds to
row \code{i} in \code{new_data}.
}
\description{
Predicted risk, survival, hazard, or mortality from an ORSF model.
}
\details{
\code{new_data} must have the same columns with equivalent types as the data
used to train \code{object}. Also, factors in \code{new_data} must not have levels
that were not in the data used to train \code{object}.

\code{pred_horizon} values should not exceed the maximum follow-up time in
\code{object}'s training data, but if you truly want to do this, set
\code{boundary_checks = FALSE} and you can use a \code{pred_horizon} as large
as you want. Note that predictions beyond the maximum follow-up time
in the \code{object}'s training data are equal to predictions at the
maximum follow-up time, because \code{aorsf} does not estimate survival
beyond its maximum observed time.

If unspecified, \code{pred_horizon} may be automatically specified as the value
used for \code{oobag_pred_horizon} when \code{object} was created (see \link{orsf}).
}
\section{Examples}{
Begin by fitting an ORSF ensemble:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(aorsf)

set.seed(329730)

index_train <- sample(nrow(pbc_orsf), 150) 

pbc_orsf_train <- pbc_orsf[index_train, ]
pbc_orsf_test <- pbc_orsf[-index_train, ]

fit <- orsf(data = pbc_orsf_train, 
            formula = Surv(time, status) ~ . - id,
            oobag_pred_horizon = 365.25 * 5)
}\if{html}{\out{</div>}}

Predict risk, survival, or cumulative hazard at one or several times:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# predicted risk, the default
predict(fit, 
        new_data = pbc_orsf_test[1:5, ], 
        pred_type = 'risk', 
        pred_horizon = c(500, 1000, 1500))
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##             [,1]       [,2]       [,3]
## [1,] 0.502277249 0.78369254 0.92143996
## [2,] 0.035699097 0.07776627 0.14863153
## [3,] 0.110355739 0.26854128 0.40820574
## [4,] 0.011659607 0.02787088 0.07267587
## [5,] 0.006580104 0.01585194 0.04896800
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# predicted survival, i.e., 1 - risk
predict(fit, 
        new_data = pbc_orsf_test[1:5, ], 
        pred_type = 'surv',
        pred_horizon = c(500, 1000, 1500))
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##           [,1]      [,2]       [,3]
## [1,] 0.4977228 0.2163075 0.07856004
## [2,] 0.9643009 0.9222337 0.85136847
## [3,] 0.8896443 0.7314587 0.59179426
## [4,] 0.9883404 0.9721291 0.92732413
## [5,] 0.9934199 0.9841481 0.95103200
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# predicted cumulative hazard function
# (expected number of events for person i at time j)
predict(fit, 
        new_data = pbc_orsf_test[1:5, ], 
        pred_type = 'chf',
        pred_horizon = c(500, 1000, 1500))
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##             [,1]       [,2]       [,3]
## [1,] 0.705240501 1.37551172 1.79408459
## [2,] 0.037859536 0.09860152 0.20087409
## [3,] 0.131539617 0.38704960 0.68840843
## [4,] 0.011659607 0.02870233 0.08408614
## [5,] 0.006580104 0.01783446 0.05794846
}\if{html}{\out{</div>}}

Predict mortality, defined as the number of events in the forest’s
population if all observations had characteristics like the current
observation. This type of prediction does not require you to specify a
prediction horizon

\if{html}{\out{<div class="sourceCode r">}}\preformatted{predict(fit, 
        new_data = pbc_orsf_test[1:5, ], 
        pred_type = 'mort')
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##           [,1]
## [1,] 71.355153
## [2,] 10.811443
## [3,] 27.509084
## [4,]  5.930542
## [5,]  3.881523
}\if{html}{\out{</div>}}
}

