% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CreateGRiwrm.R
\name{CreateGRiwrm}
\alias{CreateGRiwrm}
\alias{GRiwrm}
\title{Generation of a network description containing all hydraulic nodes and the description
of their connections}
\usage{
CreateGRiwrm(
  db,
  cols = list(id = "id", down = "down", length = "length", model = "model", area =
    "area"),
  keep_all = FALSE
)
}
\arguments{
\item{db}{\link{data.frame} description of the network (See details)}

\item{cols}{\link{list} or \link{vector} columns of \code{db}. By default, mandatory column names are: \code{id}, \code{down}, \code{length}. Other names can be handled with a named list or vector containing items defined as \code{"required name" = "column name in db"}}

\item{keep_all}{\link{logical} indicating if all columns of \code{db} should be kept or if only columns defined in \code{cols} should be kept}
}
\value{
\link{data.frame} of class \code{GRiwrm} describing the airGR semi-distributed model network, with each line corresponding to a location on the river network and with the following columns:
\itemize{
\item \code{id} (\link{character}): node identifier
\item \code{down} (\link{character}): identifier of the node downstream of the current node (\link{NA} for the most downstream node)
\item \code{length} (\link{numeric}): hydraulic distance to the downstream node in km (\link{NA} for the most downstream node)
\item \code{area} (\link{numeric}): total area of the basin starting from the current node location in km2
\item \code{model} (\link{character}): hydrological model to use if necessary (\link{NA} for using observed flow instead of a runoff model output)
}
}
\description{
Generation of a network description containing all hydraulic nodes and the description
of their connections
}
\details{
\code{db} is a \link{data.frame} which at least contains in its columns:
\itemize{
\item a node identifier (column \code{id}),
\item the identifier and the hydraulic distance to the downstream node (\link{character} columns \code{down} and \link{numeric} columns \code{length} in km). The last downstream node should have fields \code{down} and \code{length} set to \code{NA},
\item the area of the basin (\link{numeric} column \code{area} in km2)
\item the hydrological model to use if necessary (\link{character} column \code{model}) (\link{NA} for using observed flow instead of a runoff model output)
}
}
\examples{
###################################################################
# Run the `airGR::RunModel_Lag` example in the GRiwrm fashion way #
# Simulation of a reservoir with a purpose of low-flow mitigation #
###################################################################

## ---- preparation of the InputsModel object

## loading package and catchment data
library(airGRiwrm)
data(L0123001)

## ---- specifications of the reservoir

## the reservoir withdraws 1 m3/s when it's possible considering the flow observed in the basin
Qupstream <- matrix(-sapply(BasinObs$Qls / 1000 - 1, function(x) {
  min(1, max(0, x, na.rm = TRUE))
}), ncol = 1)

## except between July and September when the reservoir releases 3 m3/s for low-flow mitigation
month <- as.numeric(format(BasinObs$DatesR, "\%m"))
Qupstream[month >= 7 & month <= 9] <- 3
Qupstream <- Qupstream * 86400 ## Conversion in m3/day

## the reservoir is not an upstream subcachment: its areas is NA
BasinAreas <- c(NA, BasinInfo$BasinArea)

## delay time between the reservoir and the catchment outlet is 2 days and the distance is 150 km
LengthHydro <- 150
## with a delay of 2 days for 150 km, the flow velocity is 75 km per day
Velocity <- (LengthHydro * 1e3 / 2) / (24 * 60 * 60) ## Conversion km/day -> m/s

# This example is a network of 2 nodes which can be describe like this:
db <- data.frame(id = c("Reservoir", "GaugingDown"),
                 length = c(LengthHydro, NA),
                 down = c("GaugingDown", NA),
                 area = c(NA, BasinInfo$BasinArea),
                 model = c(NA, "RunModel_GR4J"),
                 stringsAsFactors = FALSE)

# Create GRiwrm object from the data.frame
griwrm <- CreateGRiwrm(db)
str(griwrm)

# Formatting observations for the hydrological models
# Each input data should be a matrix or a data.frame with the good id in the name of the column
Precip <- matrix(BasinObs$P, ncol = 1)
colnames(Precip) <- "GaugingDown"
PotEvap <- matrix(BasinObs$E, ncol = 1)
colnames(PotEvap) <- "GaugingDown"

# Observed flows contain flows that are directly injected in the model
Qobs = matrix(Qupstream, ncol = 1)
colnames(Qobs) <- "Reservoir"

# Creation of the GRiwrmInputsModel object (= a named list of InputsModel objects)
InputsModels <- CreateInputsModel(griwrm,
                            DatesR = BasinObs$DatesR,
                            Precip = Precip,
                            PotEvap = PotEvap,
                            Qobs = Qobs)
str(InputsModels)

## run period selection
Ind_Run <- seq(which(format(BasinObs$DatesR, format = "\%Y-\%m-\%d")=="1990-01-01"),
               which(format(BasinObs$DatesR, format = "\%Y-\%m-\%d")=="1999-12-31"))

# Creation of the GriwmRunOptions object
RunOptions <- CreateRunOptions(InputsModels,
                                IndPeriod_Run = Ind_Run)
str(RunOptions)

# Parameters of the SD models should be encapsulated in a named list
ParamGR4J <- c(X1 = 257.238, X2 = 1.012, X3 = 88.235, X4 = 2.208)
Param <- list(`GaugingDown` = c(Velocity, ParamGR4J))

# RunModel for the whole network
OutputsModels <- RunModel(InputsModels,
                          RunOptions = RunOptions,
                          Param = Param)
str(OutputsModels)

# Compare Simulation with reservoir and observation of natural flow
plot(OutputsModels, data.frame(GaugingDown = BasinObs$Qmm[Ind_Run]))
}
