\name{modpartltraj}
\alias{modpartltraj}
\alias{print.modpartltraj}
\alias{partmod.ltraj}
\alias{print.partltraj}
\alias{bestpartmod}
\alias{plot.partltraj}
\title{Segmentation of a traject based on Markov models (STILL UNDER RESEARCH!!)}
\description{
  These functions partition a traject into several segments
  corresponding to different behaviours of the animal.\cr
  \code{modpartltraj} is used to generate the models to which the traject
  is compared.\cr
  \code{bestpartmod} is used to compute the optimal number of segments
  of the partition.\cr
  \code{partmod.ltraj} is used to partition the traject into
  \code{npart} segments.  \code{plot.partltraj} can be used to plot the
  results. 
}
\usage{
modpartltraj(tr, limod)
print.modpartltraj(x, \dots)

bestpartmod(mods, Km = 30, plotit = TRUE,
            correction = TRUE, nrep = 100)

partmod.ltraj(tr, npart, mods, na.manage = c("prop.move","locf"))
print.partltraj(x, \dots)
plot.partltraj(x, col, addpoints = TRUE, lwd = 2, \dots)
}
\arguments{
  \item{tr}{an object of class \code{ltraj} containing only one traject
    (one burst of relocation)}
  \item{limod}{a list of syntactically correct R expression giving the
    models for the traject, implying one or several elements in
    \code{tr} (see details and examples)}
  \item{x, mods}{an object of class \code{modpartltraj} (for
    \code{print.modpartltraj}), \code{partltraj} (for
    \code{print.partltraj} and \code{plot.partltraj}) returned
    respectively by the function \code{genmod.crw} and
    \code{partmod.ltraj}}
  \item{na.manage}{a character string indicating what should be done
    with the missing values located between two segments.  With
    \code{"locf"}, the missing values are added at the end of the first
    segment. With \code{"prop.move"}, the missing values are distributed
    at the end of the first and the beginning of the second segment.  The
    proportion of missing values added at the end of the first segment
    correspond the relative proportion of "internal" missing values
    found within the segments predicted by the model used to predict the
    first segment.}
  \item{npart}{the number of partitions of the traject}
  \item{Km}{the maximum number of partitions of the traject}
  \item{plotit}{logical. Whether the results should be plotted.}
  \item{correction}{logical. Whether the log-likelihood should be
    corrected (see details).}
  \item{nrep}{logical. The number of Monte Carlo simulations used to
    correct the log-likelihood for each number of segments.}
  \item{col}{the colors to be used for the models}
  \item{addpoints}{logical.  Whether the relocations should be added to
    the graph}
  \item{lwd}{the line width}
  \item{\dots}{additional arguments to be passed to other functions}
}
\details{
  A traject is made of successive steps traveled by an organism in the
  geographical space.  These steps (the line connecting two successive
  relocations) can be described by a certain number of descriptive
  parameters (relative angles between successive steps, length of the
  step, etc.).  One aim of the traject analysis is to identify 
  the structure of the traject, i.e.  the parts of the traject where the
  steps have homogeneous properties. Indeed, an animal may have a wide
  variety of behaviours (feeding, traveling, escape from a predator,
  etc.). As a result, partitioning a traject occupies a central place in
  traject analysis.\cr
  
  These functions are to be used to partition a traject based on Markov
  models of animal movements.  For example, one may suppose that a
  normal distribution generated the step lengths, with a different mean
  for each type of behaviour.  These models and the value of their
  parameters are supposed a priori by the analyst.  These functions
  allow, based on these a priori models, to find both the number and the
  limits of the segments building up the traject (see examples).  Any
  model can be supposed for any parameter of the steps (the distance,
  relative angles, etc.), provided that the model is Markovian.\cr 
  
  The rationale behind this algorithm is the following.  First, the user
  should propose a set of model describing the movements of the animals,
  in the different segments of the traject.  For example, the user may
  define two models of normal distribution for the step length, with
  means equal to 10 meters (i.e. a traject with relatively small steps)
  and 100 meters (i.e. a traject with longer step lengths).  For a
  given step of the traject, it is possible to compute the probability
  density that the step has been generated by each model of the set.
  The function \code{modpartltraj} computes the matrix containing the
  probability densities associated to each step (rows), under each model
  of the set (columns).  This matrix is of class \code{modpartltraj}.\cr

  Then, the user can estimate the optimal number of segments in the
  traject, given the set of a priori models, using the function
  \code{bestpartmod}, taking as argument the matrix of class
  \code{modpartltraj}.  If \code{correction = FALSE}, this function
  returns the log of the probability (log-likelihood) that the traject
  is actually made of \code{K} segments, with each one described by one
  model.  The resulting graph can be used to choose an optimal number of
  segment for the partition.  Note that Gueguen (2007) noted that this
  algorithm tends to overestimate the number of segments in a
  traject. He proposed to correct this estimation using Monte Carlo
  simulations of the independence of the steps within the traject.  At
  each step of the randomization process, the order of the rows of the
  matrix is randomized, and the curve of log-likelihood is computed for
  each number of segments, for the randomized traject.  Then, the
  observed log-likelihood is corrected by these simulations: for a given
  number of segments, the corrected log-likelihood is equal to the
  observed log-likelihood minus the simulated log-likelihood.  Because
  there is a large number of simulations of the independence, a
  distribution of corrected log-likelihoods is available for each number
  of segments.  The "best" number of segments is the one for which the
  median of the distribution of corrected log-likelihood is maximum.\cr

  Finally, once the optimal number of segments \code{npart} has been
  chosen, the function \code{partmod.ltraj} can be used to compute the
  partition.\cr
  
  The mathematical rationale underlying these two functions is the
  following: given an optimal k-partition of the traject, if the ith
  step of the traject belongs to the segment k predicted by the model d,
  then either the relocation (i-1) belongs to the same segment, in which
  case the segment containing (i-1) is predicted by d, or the relocation
  (i-1) belongs to another segment, and the other (k-1) segments
  together constitute an optimal (k-1) partition of the traject
  1-(i-1). These two probabilities are computed recursively by the
  functions from the matrix of class \code{partmodltraj}, observing that
  the probability of a 1-partition of the traject from 1 to i described
  by the model m (i.e. only one segment describing the traject) is
  simply the product of the probability densities of the steps from 1 to
  i under the model m.  Further details can be found in Calenge et
  al. (in prep), and in Gueguen (2001, 2007).
}
\value{
  \code{partmodltraj} returns a matrix of class \code{partmodltraj}
  containing the probability densities of the steps of the traject
  (rows) for each model (columns).\cr
  
  \code{bestpartmod} returns a list with two elements: (i) the element
  \code{mk} is a vector containing the values of the log-probabilities
  for each number of segments (varying from 1 to \code{Km}), and (ii)
  the element \code{correction} contains either \code{"none"} or a
  matrix containing the corrected log-likelihood for each number of
  segments (rows) and each simulation of the independence (column).\cr
  
  \code{partmod.ltraj} returns a list of class \code{partltraj} with the
  following components: \code{ltraj} is an object of class \code{ltraj}
  containing the segmented traject (one burst of relocations per segment
  of the partition); \code{stats} is a list containing the following
  elements:
  \item{locs}{The number ID of the relocations starting the segments
    (except the last one which ends the last segment)}
  \item{Mk}{The value of the cumulative log-probability for the Partition
    (i.e. the log-probability associated to a K-partition is equal to the
    log-probability associated to the (K-1)-partition plus the
    log-probability associated to the Kth segment)}
  \item{mod}{The number ID of the model chosen for each segment}
  \item{which.mod}{the name of the model chosen for each segment}
}
\references{
  Calenge, C., Gueguen, L., Royer, M. and Dray, S. (in prep.)
  Partitioning the traject of an animal with Markov models.
  
  Gueguen, L. (2001) Segmentation by maximal predictive partitioning
  according to composition biases. Pp 32--44 in: Gascuel, O. and Sagot,
  M.F. (Eds.), \emph{Computational Biology}, LNCS, 2066.
  
  Gueguen, L. (in prep.) Computing the probability of sequence
  segmentation under Markov models.
}
\author{Clément Calenge \email{calenge@biomserv.univ-lyon1.fr}}
\seealso{\code{\link{ltraj}}}
\examples{

\dontrun{
## Example on the porpoise
data(porpoise)

## Keep the first porpoise
gus <- porpoise[1]
plot(gus)

## First test the independence of the step length
indmove(gus)
## There is a lack of independence between successive distances

## plots the distance according to the date
plotltr(gus, "dist")

## One supposes that the distance has been generated
## by normal distribution, with different means for the
## different behaviours
## The means of the normal distribution range from 0 to
## 130000. We suppose a standard deviation equal to 5000:

tested.means <- round(seq(0, 130000, length = 10), 0)
(limod <- as.list(paste("dnorm(dist, mean =",
                  tested.means,
                  ", sd = 5000)")))

## Build the probability matrix
mod <- modpartltraj(gus, limod)

## computes the corrected log-likelihood for each
## number of segments
bestpartmod(mod)

## The best number of segments is 4. Compute the partition:
(pm <- partmod.ltraj(gus, 4, mod))
plot(pm)


## Shows the partition on the distances:
plotltr(gus, "dist")

lapply(1:length(pm$ltraj), function(i) {
   lines(pm$ltraj[[i]]$date, rep(tested.means[pm$stats$mod[i]],
         nrow(pm$ltraj[[i]])),
         col=c("red","green","blue")[as.numeric(factor(pm$stats$mod))[i]],
         lwd=2)
})


## Computes the residuals of the partition
res <- unlist(lapply(1:length(pm$ltraj), function(i) {
   pm$ltraj[[i]]$dist - rep(tested.means[pm$stats$mod[i]],
         nrow(pm$ltraj[[i]]))
}))

plot(res, ty = "l")

## Test of independence of the residuals of the partition:
wawotest(res)

}
}
\keyword{spatial}
