% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fsdea.R
\name{fsdea}
\alias{fsdea}
\title{Feature Selection in Data Envelopment Analysis with Mathematical Programming}
\usage{
fsdea(
  input,
  output,
  orientation = c("input", "output"),
  name = "",
  ninputs = ncol(input),
  noutputs = ncol(output),
  nvariables = ncol(input) + ncol(output),
  solver = "auto"
)
}
\arguments{
\item{input}{A matrix or a data frame containing the inputs of the units to be evaluated, with one row for each DMU and one column for each input.}

\item{output}{A matrix or a data frame containing the outputs of the units to be evaluated, with one row for each DMU and one column for each output.}

\item{orientation}{Use "input" for input orientation or "output" for output orientation in DEA model.
The default is "input".}

\item{name}{An optional descriptive name for the model.
The default is an empty string. 
This name will be displayed in printed and summarized results.}

\item{ninputs}{Number of input features (variables) to be selected.
Default is the number of input variables.}

\item{noutputs}{Number of output features (variables) to be selected.
Default is the number of output variables.}

\item{nvariables}{Number of total features (variables) to be selected, only applicable when both ninputs and noutputs are omitted.
Default is the number of input plus output variables.}

\item{solver}{The solver to be used by ROI to solve the DEA optimization problem.
The solver must be installed and capable of solving mixed integer linear programming problems.
Default is "auto."
Use `ROI_installed_solvers()` to list available solvers.}
}
\value{
This function return a fsdea class object with the following named members:
\itemize{
\item orientation: DEA model orientation.
\item name: A label of the model.
\item ninputs: Number of inputs to be selected.
\item noutputs: Number of outputs to be selected.
\item nvariables: Number of total variables to be selected.
\item inputnames: Names of input variables.
\item outputnames: Names of output variables.
\item eff: A vector with DMU scores.
\item ux: A set of weights for input variables.
\item vy: A set of weights for output variables.
\item obj: Optimal value of the objective function in the optimization problem.
\item iselected: A vector of zeros and ones indicating the selected input variables.
\item oselected: A vector of zeros and ones indicating the selected output variables.
\item niselected: Number of input selected variables.
\item noselected: Number of output selected variables.
\item nvselected: Number of selected variables.
\item vinput: Standardized virtual input divided by the sum of the weights, see [Costa2006] in \code{\link{adea-package}}.
\item voutput: Standardized virtual output divided by the sum of the weights, see [Costa2006] in \code{\link{adea-package}}.
\item solver: The solver used for the resolution of the optimization problem.
}
}
\description{
Data Envelopment Analysis (DEA) calculates a relative efficiency score for a set of Decision Making Units (DMUs) by comparing one unit with others.
}
\details{
Each DMU transforms inputs into outputs.
The set of inputs and outputs is the same for all the DMUs, but not their quantities.

One of the fundamental steps in the application of data envelopment analysis is the choice of variables to include in the model.
One of the methods proposed for this is what is known as the feature selection method.
This method constructs a linear programming problem to maximize some objective function related to the dmu efficiencies.
This function implements the feature selection method in the article [Benitez-Pena, S., Bogetoft, P. and Romero Morales, D. *Feature Selection in Data Envelopment Analysis: A Mathematical Optimization approach* Omega, Elsevier BV, **2020**, Vol. 96, pp. 102068](http://www.sciencedirect.com/science/article/pii/S0305048318312131)

This function, in the case of input orientation, maximize the sum of all efficiencies, while in the output orientation case, the goal is to minimize this sum. 
Once the relevant variables are selected, the function calculates the relative efficiency scores for each Decision Making Unit (DMU) and determines the weights for all input and output variables within the model.
}
\examples{
data('cardealers4')
input <- cardealers4[, c('Employees', 'Depreciation')]
output <- cardealers4[, c('CarsSold', 'WorkOrders')]

# Compute DEA model selecting at most 1 output
model1o <- fsdea(input, output, noutputs = 1)
model1o
#   Dealer A  Dealer B  Dealer C  Dealer D  Dealer E  Dealer F 
#  0.7875000 0.7500000 0.3000000 0.8653846 1.0000000 0.5400000 
#  Selected inputs : Depreciation
#  Selected outputs: CarsSold

# Compute DEA model selecting at most 1 input
model1i <- fsdea(input, output, ninputs = 1)
model1i
# Dealer A  Dealer B  Dealer C  Dealer D  Dealer E  Dealer F
# 0.9915929 1.0000000 0.8928571 0.8653846 1.0000000 0.6515044
# Selected inputs : Depreciation
# Selected outputs: CarsSold, WorkOrders

# Compute DEA model selecting at most 3 variables
model3v <- fsdea(input, output, nvariables = 3)
model3v
# Dealer A  Dealer B  Dealer C  Dealer D  Dealer E  Dealer F
# 0.9915929 1.0000000 0.8928571 0.8653846 1.0000000 0.6515044
# Selected inputs : Depreciation
# Selected outputs: CarsSold, WorkOrders

}
\seealso{
\code{\link{adea-package}}.
}
\concept{Data Envelopment Analysis}
\concept{Feature selection}
\concept{Variable selection}
