#' Quick sample
#' 
#' Faster than inbuilt sample because it avoids some checks
#' @param x vector to sample
#' @param len length of vector
#' @keywords internal
#' @export
SampleOne <- function (x, len = length(x)) x[sample.int(len, 1L, FALSE, NULL, FALSE)]

Assert <- function (statement) if (!statement) stop(deparse(statement), " is FALSE")

#' Edge list to edge matrix
#' @param edgeList tree edges in the format list(parent, child).
#' @return edges in the format expected by \code{tree$edge},
#'         where \code{tree} is a tree of class \code{phylo}.
#' @keywords internal
#' @export
ListToMatrix <- function (edgeList) matrix(c(edgeList[[1]], edgeList[[2]]), ncol=2)

#' Edge matrix to edge list
#' @param edge edges in the matrix format used by \code{tree$edge}, where \code{tree} is a tree of class \code{phylo}
#' @return tree edges in the format list(parent, child).
#' @export
MatrixToList <- function (edge) list(edge[, 1], edge[, 2])

#' Descendant Edges
#'
#' Quickly identifies edges that are 'descended' from a particular edge in a tree
#'
#' @param edge number of the edge whose child edges are required
#' @template treeParent
#' @template treeChild
#' @param nEdge number of edges (calculated from length(parent) if not supplied)
#' @return `DescendantEdges` returns a logical vector stating whether each edge in turn is a descendant of the specified edge
#'         (or the edge itself)
#' @export
DescendantEdges <- function (edge, parent, child, nEdge = length(parent)) {
  ret <- logical(nEdge)
  edgeSister <- match(parent[edge], parent[-edge])
  if (edgeSister >= edge) {
    # edgeSister is really 1 higher than you think, because we knocked out edge 'edge' in the match
    ret[edge:edgeSister] <- TRUE
    return(ret)
  } else {
    nextEdge <- edge
    revParent <- rev(parent)
    repeat {
      if (revDescendant <- match(child[nextEdge], revParent, nomatch=FALSE)) {
        nextEdge <- 1 + nEdge - revDescendant
      } else break;
    }
    ret[edge:nextEdge] <- TRUE 
    return(ret)
  }
}

#' All Descendant Edges
#'
#' @return `AllDescendantEdges` returns a matrix of class logical, with row N specifying whether each edge is a descendant of edge N
#'         (or the edge itself)
#' @describeIn DescendantEdges Quickly identifies edges that are 'descended' from each edge in a tree
#' @export
AllDescendantEdges <- function (parent, child, nEdge = length(parent)) {
  ret <- diag(nEdge) == 1
  blankLogical <- logical(nEdge)
  allEdges <- seq_len(nEdge)
  for (edge in rev(allEdges[child > parent[1]])) {
    ret[edge, ] <- apply(ret[parent == child[edge], ], 2, any)
    ret[edge, edge] <- TRUE
  }
  ret
}

#' Ancestral edge
#'
#' @param edge Number of an edge
#' @param parent parent nodes (given by phylo.object$edge[, 1])
#' @param child  child nodes (given by phylo.object$edge[, 2])
#' @return a logical vector identifying whether each edge is the edge that is ancestral to the given edge.
#' @keywords internal
#' @export
AncestorEdge <- function (edge, parent, child) child == parent[edge]

#' EdgeAncestry
#'
#' Descendant Edges
#'
#' Quickly identifies edges that are 'ancestral' to a particular edge in a tree
#'
#' @param edge number of the edge whose child edges are required
#' @template treeParent
#' @template treeChild
#' @param stopAt number of the edge at which the search should terminate; defaults to the root edges
#' @return a logical vector stating whether each edge in turn is a descendant of the specified edge
#'
#' @author Martin R. Smith
#' @export
EdgeAncestry <- function (edge, parent, child, stopAt = (parent==min(parent))) {
  ret <- edge <- AncestorEdge(edge, parent, child)
  repeat {
    if (any(ret[stopAt])) return(ret)
    ret[edge <- AncestorEdge(edge, parent, child)] <- TRUE    
  }
}

#' Non-duplicate root
#' 
#' Identify, for each edge, whether it is not a duplicate of the root edge
#' 
#' @template treeParent
#' @template treeChild
#' @template treeNEdgeOptional
#' 
#' @author Martin R. Smith
#' @export
#' 
#' @keywords internal
NonDuplicateRoot <- function (parent, child, nEdge = length(parent)) {
  notDuplicateRoot <- !logical(nEdge)
  rightSide <- DescendantEdges(1, parent, child, nEdge)
  nEdgeRight <- sum(rightSide)
  if (nEdgeRight == 1) {
    notDuplicateRoot[2] <- FALSE
  } else if (nEdgeRight == 3) {
    notDuplicateRoot[4] <- FALSE
  } else {
    notDuplicateRoot[1] <- FALSE
  }
  notDuplicateRoot
}
