% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tdm_cal.sfd.R
\name{tdm_cal.sfd}
\alias{tdm_cal.sfd}
\title{Calculate sap flux density}
\usage{
tdm_cal.sfd(
  input,
  genus,
  species,
  study,
  wood,
  calib,
  a,
  b,
  decimals,
  make.plot = TRUE,
  df = FALSE
)
}
\arguments{
\item{input}{An \code{\link{is.trex}}-compliant object (\code{zoo} vector,
\code{data.frame}) of \eqn{K} values containing a timestamp and a value column.}

\item{genus}{Optional, character vector specifying genus-specific calibration
data that should be used (e.g., \code{c("Picea", "Larix")}). See \code{\link{cal.data}}
for the specific labels (default = Use all).}

\item{species}{Optional, character vector of species specific calibration data that should be used,
e.g. \code{c("Picea abies")}. See \code{\link{cal.data}} for the specific labels (default = Use all).}

\item{study}{Optional character vector of study specific calibration data that
should be used (e.g., \code{c("Peters et al. 2018"}) ). See \code{\link{cal.data}}
for the specific labels (default= Use all).}

\item{wood}{Optional, character vector of wood type specific calibration
data that should be used (one of \code{c("Diffuse-porous", "Ring-porous", "Coniferous")}).
See \code{\link{cal.data}} for the specific labels (default= Use all).}

\item{calib}{Optional \code{data.frame} containing raw calibration experiment values.
Required columns include: \code{[ ,1]} \eqn{K = K} values measured with the probe (numeric),
and [,2] \eqn{SFD =} Gravimetrically measured sap flux density (\eqn{cm^3 cm^{-2} h^{-1}}{cm3 cm-2 h-1})) (numeric).
If not provided, literature values are used.}

\item{a}{Optional, numeric value for the calibration curve (\eqn{SFD = aK^b}).
No uncertainty can be calculated when this value is provided.}

\item{b}{Optional, numeric value for the calibration curve (\eqn{SFD = aK^b}).
No uncertainty can be calculated when this value is provided.}

\item{decimals}{Integer, the number of decimals of the output (default = 6).}

\item{make.plot}{Logical; if \code{TRUE}, a plot is generated showing
the calibration curve with \eqn{K vs sap flux density} (\eqn{cm^3 cm^{-2} h^{-1}}{cm3 cm-2 h-1})).}

\item{df}{Logical; If \code{TRUE}, output is provided in a \code{data.frame} format
with a timestamp and a value column. If \code{FALSE}, output
is provided as a \code{zoo} vector object (default = FALSE).}
}
\value{
A list containing either a \code{zoo} object or \code{data.frame} in the appropriate format
 for other functionalities (see \code{\link{tdm_dt.max}} output specifications), as well as
 all \eqn{SFD} values for each method are provided and added to the
 \code{\link{is.trex}}-compliant object (e.g., [['sfd.pd']], [['sfd.mw']])
 if this format was provided as an input, and,
 finally, a \code{data.frame} is provided with the mean and 95\% confidence
 interval of the applied calibration functions (see [['model.ens']]).
 If an individual time series is provided for input with \eqn{K} values an alternative output is provided:

 \describe{
   \item{input}{K values provided as input.}
   \item{sfd.input}{\eqn{SFD} values calculated for the input according to the mean of the calibration function.}
   \item{model.ens}{A \code{data.frame} providing the mean and 95\% confidence interval of the applied calibration function.}
   \item{out.param}{A \code{data.frame} with the coefficients of calibration function.}
 }
}
\description{
The acquired \eqn{K} values are calculated to sap flux density
 (\eqn{SFD} in \eqn{cm^3 cm^{-2} h^{-1}}{cm3 cm-2 h-1}). As many calibration curves exist
 (see Peters \emph{et al}. 2018; Flo \emph{et al.} 2019), the function provides the option to
 calculate \eqn{SFD} using calibration experiment data from the meta-analyses by
 Flo \emph{et al.} (2019; see \code{\link{cal.data}}). Additionally,
 raw calibration data can be provided or parameters \eqn{a} and \eqn{b}
 for a specific calibration function (\eqn{aK^b}) can be provided.
 The algorithm determines for each calibration experiment dataset
 the calibration curve (\eqn{SFD = aK^b}) and calculates \eqn{SFD} from
 either the mean of all curves and the 95\% confidence interval
 of either all curves, or bootstrapped resampled uncertainty around
 the raw calibration experiment data when one calibration dataset is selected.
}
\details{
The function fits a calibration curve (\eqn{SFD = aK^b}{SFD = aK^b})
through all selected raw calibration data. If multiple studies are provided,
multiple calibration curves are fitted. In case a single calibration dataset
is provided a bootstrap resampling is applied (n = 100) to determined the
mean and 95\% confidence interval of the fit. When multiple calibration curves
are requested the mean and 95\% confidence interval is determined on the fitted functions.
The mean and confidence interval are used to calculate \eqn{SFD} from \eqn{K}.
}
\examples{
#calculating sap flux density
\donttest{
raw   <-is.trex(example.data(type="doy"),
    tz="GMT",time.format="\%H:\%M",
    solar.time=TRUE,long.deg=7.7459,
    ref.add=FALSE)

input <-dt.steps(input=raw,start="2014-05-08 00:00",
end="2014-07-25 00:50",
     time.int=15,max.gap=60,decimals=10,df=FALSE)

input[which(input<0.2)]<-NA

input <-tdm_dt.max(input, methods=c("pd","mw","dr"),
     det.pd=TRUE,interpolate=FALSE,max.days=10,df=FALSE)

output.data<-tdm_cal.sfd(input,make.plot=TRUE,df=FALSE,
wood="Coniferous", decimals = 6)


str(output.data)
plot(output.data$sfd.pd$sfd,ylim=c(0,10))
lines(output.data$sfd.pd$q025,lty=1,col="grey")
lines(output.data$sfd.pd$q975,lty=1,col="grey")
lines(output.data$sfd.pd$sfd)

output.data$out.param
}

}
\references{
Peters RL, Fonti P, Frank DC, Poyatos R, Pappas C, Kahmen A, Carraro V, Prendin AL, Schneider L, Baltzer JL,
 Baron-Gafford GA, Dietrich L, Heinrich I, Minor RL, Sonnentag O, Matheny AM, Wightman MG, Steppe K. 2018.
 Quantification of uncertainties in conifer sap flow measured with the thermal dissipation method.
 New Phytologist 219:1283-1299 <doi: 10.1111/nph.15241>

 Flo V, Martinez-Vilalta J, Steppe K, Schuldt B, Poyatos, R. 2019.
 A synthesis of bias and uncertainty in sap flow methods.
 Agricultural and Forest Meteorology 271:362-374 \doi{10.1016/j.agrformet.2019.03.012}
}
