SesReport <- function(x, outdir=getwd(), name="SesReport", export=TRUE, report=TRUE){
  stopifnot(inherits(x, "SesIndex") || inherits(x, "SesClassif"))
  stopifnot(is.character(name))
  stopifnot(is.character(outdir))
  stopifnot(is.logical(export))
  stopifnot(is.logical(report))
  
  if(inherits(x, "SesClassif")) {
    x.SesIndex <- x$call$X
    x.export <- x$table
    title <- "Creation and classification of a socioeconomic index"
  }
  else {
    x.SesIndex <- x
    x.export <- x$step3$indices
    title <- "Creation of a socioeconomic index"
  }
  if (inherits(x.SesIndex$step2$analysis, "MFA")) {
    step2.method <- "Multiple Factor Analysis"
  }
  else {
    step2.method <- "Principal Component Analysis"
  }
  
  ##Creation of the outdir
  old.wd <- getwd()
  if(file.exists(outdir)){
    if(!file.info(outdir)$isdir){
      stop(sprintf("'%s' must be a directory.", outdir))
    }
    outdirContents = dir(outdir, all.files = TRUE)
    outdirContents = setdiff(outdirContents, c(".", ".."))
    
    if(length(outdirContents)>0){
      warning(sprintf("'%s' was not empty.", outdir))
    } 
  }
  else {
    dir.create(outdir, recursive=TRUE)
    message(sprintf("The directory '%s' has been created.", outdir))
  }
  setwd(outdir)
      
  if (report) {  
    # Open an HTML file
    con <- file(paste(name, ".html", sep=""), open="wt")
    writeLines(paste("<html><head><title>", title,
                     "</title></head><body style=\"font-family: ",
                     "helvetica,arial,sans-serif;\">", sep=""), con)
    titletext = sprintf("<h1><center>%s</h1></center>", title)
    writeLines(titletext, con) 
    
    writeLines('<center><span style="font-style:italic;">This report was automatically generated by the SesReport function of the R package SesIndexCreatoR .</span></center>', con)           
               
    # Index
    writeLines('<hr style="color:black; background-color:black; height:3px; text-align:center; width:800px;" /> <h2>Index</h2><table border = "0" cellspacing = 5 cellpadding = 2><UL>', con)
    writeLines('<tr><td><LI><b><a href="#Step1">Step 1: Reduction of the redundant groups</b></a>', con)
    writeLines('<tr><td><LI><b><a href="#Step2">Step 2: Selection of the variables</b></a>', con)
    writeLines('<tr><td><LI><b><a href="#Step3">Step 3: Creation of the index</b></a>', con)
    if (inherits(x, "SesClassif")) {
      writeLines('<tr><td><LI><b><a href="#Classif">Classification</b></a>', con)
    }
    writeLines("</td></tr></UL></table>", con)
    
    
    # Step 1
    writeLines('<hr style="color:black; background-color:black; height:3px; text-align:center; width:800px;" /> <h2><a name="Step1">Step 1: Reduction of the redundant groups</a></h2>', con)
    if (!is.null(x.SesIndex$step1)) {
      writeLines("A Principal Component Analysis is performed for each redundant group and only the variable with the highest correlation with the first component is kept.", con)
      writeLines("<br /> The results of the analysis of each group are :", con)
      
      writeLines('<ul style="list-style-type:disc;">', con)
      
      tmp_plotname=paste(name,"-step1", sep="")
      png(filename=paste(tmp_plotname,"-%02d.png", sep=""))
      plot(x.SesIndex, step=1, choice="var")
      dev.off()
      for (i in 1:length(x.SesIndex$step1$analysis)){
        group_title=paste('<hr style="color:black; text-align:center; width:300px;" /><li><h3>Group ', i,"</h3>")
        writeLines(group_title, con)
        
        tmp_grp <- paste("group",i, sep="")
        tmp_print <- round(cbind("coord"=x.SesIndex$step1$analysis[[tmp_grp]]$var$coord[,1], "cos2"=x.SesIndex$step1$analysis[[tmp_grp]]$var$cos2[,1], "contrib"=x.SesIndex$step1$analysis[[tmp_grp]]$var$contrib[,1]), 2)

        writeLines("The following results are obtained for the first component of the PCA :", con)
        writeLines('<center><table frame="above" cellspacing="0" cellpadding="3px" style="margin-top:20px; margin-bottom:20px; border-top:solid 1px black; border-collapse:collapse;" >', con)
        writeLines('<thead><tr style="background-color:lightgrey; border-bottom:solid 1px black;"><th style="text-align:left;">Variable</th><th style="text-align:center;width:100px;">Coord</th><th style="text-align:center;width:100px;">Cos2</th><th style="text-align:center;width:100px;">Contrib</th></tr></thead>', con)
        writeLines('<tbody>', con)

        for (j in 1:nrow(tmp_print)){
          writeLines('<tr>', con)
          writeLines(paste('<th style="text-align:left;font-style:italic;">', rownames(tmp_print)[j], '</th>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print[j,1], '</td>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print[j,2], '</td>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print[j,3], '</td>', sep=""), con)
          writeLines('</tr>', con)
        }        
        writeLines('</tbody></table></center>', con)
        
        writeLines('<center><table frame=box cellspacing="0" cellpadding ="2" style="margin-top:20px; margin-bottom:20px; border-collapse:collapse;"><tr><td>', con)
        writeLines(paste('<img src="',tmp_plotname,'-0',i,'.png" alt="',tmp_plotname,'-0',i,'" />', sep=""), con) 
        writeLines('</td></tr>', con) 
        writeLines('<tr style="background-color:lightgrey;"><td><center><b>Correlation circle</b></center>', con) 
        writeLines('</td></tr></table></center>', con) 
        
        tmp_selection <- paste("<br />The variable selected for group ", i, ' is "<span style="font-style:italic;">', x.SesIndex$step1$selection[i], '</span>"</li>', sep="")
        writeLines(tmp_selection, con)
              
      }
      writeLines("</ul>", con)
    }
    else {
      writeLines("Step 1 was skipped. No redondant groups considered.")
    }
    writeLines("<br />", con)
    
    
    # Step 2
    writeLines('<hr style="color:black; background-color:black; height:3px; text-align:center; width:800px;" /> <h2><a name="Step2">Step 2: Selection of the variables</a></h2>', con)
    if (!is.null(x.SesIndex$step2)) {      
      writeLines(paste("A", step2.method, "is performed on the variables kept after Step 1. Only variables with a correlation higher than the average correlation will be kept after Step 2."), con)
      writeLines("<br /> The results of the analysis are :", con)
      
      tmp_step2 <- x.SesIndex$step2$analysis
      tmp_print2_eig <- round(tmp_step2$eig[1:10,], 2)
      if (inherits(x.SesIndex$step2$analysis, "MFA")) {
        tmp_print2_var <- round(cbind("coord 1"=tmp_step2$quanti.var$coord[,1], "cos2 1"=tmp_step2$quanti.var$cos2[,1], "contrib 1"=tmp_step2$quanti.var$contrib[,1],"coord 2"=tmp_step2$quanti.var$coord[,2], "cos2 2"=tmp_step2$quanti.var$cos2[,2], "contrib 2"=tmp_step2$quanti.var$contrib[,2]),2)
      }
      else {
        tmp_print2_var <- round(cbind("coord 1"=tmp_step2$var$coord[,1], "cos2 1"=tmp_step2$var$cos2[,1], "contrib 1"=tmp_step2$var$contrib[,1],"coord 2"=tmp_step2$var$coord[,2], "cos2 2"=tmp_step2$var$cos2[,2], "contrib 2"=tmp_step2$var$contrib[,2]),2)
      }
     
      writeLines('<center><table frame="above" cellspacing="0" cellpadding="3px" style="margin-top:20px; margin-bottom:20px; border-top:solid 1px black; border-collapse:collapse;"><caption>10 first eigenvalues</caption>', con)
      writeLines('<thead><tr style="background-color:lightgrey;border-bottom:solid 1px black;text-align:center;"><th></th><th style="width:100px;">Eigenvalue</th><th style="width:100px;">Percentage of variance</th><th style="width:100px;">Cumulative percentage of variance</th></tr></thead>', con)
      writeLines('<tbody>', con)

      for (i in 1:nrow(tmp_print2_eig)){
        writeLines('<tr>', con)
        writeLines(paste('<th style="text-align:left;font-style:italic;">', rownames(tmp_print2_eig)[i], '</th>', sep=""), con)
        writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print2_eig[i,1], '</td>', sep=""), con)
        writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print2_eig[i,2], '</td>', sep=""), con)
        writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print2_eig[i,3], '</td>', sep=""), con)
        writeLines('</tr>', con)   
      }      
      writeLines('</tbody></table></center>', con)
      
      writeLines('<center><table frame="above" cellspacing="0" cellpadding="3px" style="margin-top:20px; margin-bottom:20px;border-top:solid 1px black; border-collapse:collapse;"><caption>Summary of the two first components</caption>', con)
      writeLines('<thead><tr style="background-color:lightgrey; border-bottom:solid 1px black;"><th style="text-align:left;">Variable</th><th style="text-align:center;width:100px;">Coord 1</th><th style="text-align:center;width:100px;">Cos2 1</th><th style="text-align:center;width:100px;">Contrib 1</th><th style="text-align:center;width:100px;">Coord 2</th><th style="text-align:center;width:100px;">Cos2 2</th><th style="text-align:center;width:100px;">Contrib 2</th></tr></thead>', con)
      writeLines('<tbody>', con)
      
      for (i in 1:nrow(tmp_print2_var)){
        if (rownames(tmp_print2_var)[i] %in% x.SesIndex$step2$selection) {
          writeLines('<tr>', con)
          writeLines(paste('<th style="text-align:left;font-style:italic;color:red">', rownames(tmp_print2_var)[i], '</th>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;color:red">', tmp_print2_var[i,1], '</td>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;color:red">', tmp_print2_var[i,2], '</td>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;color:red">', tmp_print2_var[i,3], '</td>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;color:red">', tmp_print2_var[i,4], '</td>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;color:red">', tmp_print2_var[i,5], '</td>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;color:red">', tmp_print2_var[i,6], '</td>', sep=""), con)
          writeLines('</tr>', con)
        }
        else {
          writeLines('<tr>', con)
          writeLines(paste('<th style="text-align:left;font-style:italic;">', rownames(tmp_print2_var)[i], '</th>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print2_var[i,1], '</td>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print2_var[i,2], '</td>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print2_var[i,3], '</td>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print2_var[i,4], '</td>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print2_var[i,5], '</td>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print2_var[i,6], '</td>', sep=""), con)
          writeLines('</tr>', con)
        }       
      }
      writeLines('<tr style="background-color:lightgrey; text-align:center; border-top:solid 1px black;"><td colspan="7">In red: variables selected in the second step.</td></tr>', con)
      writeLines('</tbody></table></center>', con)
      
      tmp_plotname2_var=paste(name,"-step2-var.png", sep="")
      png(filename=tmp_plotname2_var)
      plot(x.SesIndex, step=2, choice="var")
      dev.off()
      tmp_plotname2_ind=paste(name,"-step2-ind.png", sep="")
      png(filename=tmp_plotname2_ind)
      if (inherits(x.SesIndex$step2$analysis, "MFA")) {
        plot(x.SesIndex, step=2, choice="ind", lab.ind=F, habillage="group")
      }
      else {
        plot(x.SesIndex, step=2, choice="ind", label="none")
      }      
      dev.off()
      
      writeLines('<center><table frame=box cellspacing="0" cellpadding ="2" style="margin-top:20px; margin-bottom:20px; border-collapse:collapse;"><tr><td>', con)
      writeLines(paste('<img src="',tmp_plotname2_var,'" alt="',tmp_plotname2_var,'" />', sep=""), con) 
      writeLines('</td><td>', con) 
      writeLines(paste('<img src="',tmp_plotname2_ind,'" alt="',tmp_plotname2_ind,'" />', sep=""), con) 
      writeLines('</td></tr>', con)
      writeLines('<tr style="background-color:lightgrey;"><td><center><b>Correlation circle</b></center></td><td><center><b>Projections of the units</b></center>', con) 
      writeLines('</td></tr></table></center></li>', con) 
      
      tmp_liste_var <- paste(x.SesIndex$step2$selection, collapse=", ")
      writeLines('<br />The variables selected in the step 2 are :<br /> <span style="font-style:italic;">', con)
      writeLines(tmp_liste_var, con)
      writeLines('</span>', con)   
    }
    else {
      writeLines("Step 2 was skipped. All the variables in the dataframe were used (except the ones which were not selected in Step 1).")  
    }
    writeLines("<br /><br />", con)

    
    # Step 3
    if (!is.null(x.SesIndex$step3)) {
      writeLines('<hr style="color:black; background-color:black; height:3px; text-align:center; width:800px;" /> <h2><a name="Step3">Step 3: Creation of the index</a></h2>', con)
      writeLines('The final step of the creation perform a Principal Component Analysis on the variables selected in Step 2. The socio-economic index is therefore the first component of the final PCA, <span style="font-style:italic;">if this component can be interpreted as a socio-economic component.</span>', con)
      writeLines("<br /> The results of the final analysis are :", con)
      
      tmp_step3 <- x.SesIndex$step3$analysis
      tmp_print3_eig <- round(tmp_step3$eig[1:10,], 2)
      tmp_print3_var <- round(cbind("coord 1"=tmp_step3$var$coord[,1], "cos2 1"=tmp_step3$var$cos2[,1], "contrib 1"=tmp_step3$var$contrib[,1],"coord 2"=tmp_step3$var$coord[,2], "cos2 2"=tmp_step3$var$cos2[,2], "contrib 2"=tmp_step3$var$contrib[,2]),2)
      
      writeLines('<center><table frame="above" cellspacing="0" cellpadding="3px" style="margin-top:20px; margin-bottom:20px; border-top:solid 1px black; border-collapse:collapse;"><caption>10 first eigenvalues</caption>', con)
      writeLines('<thead><tr style="background-color:lightgrey;border-bottom:solid 1px black;text-align:center;"><th></th><th style="width:100px;">Eigenvalue</th><th style="width:100px;">Percentage of variance</th><th style="width:100px;">Cumulative percentage of variance</th></tr></thead>', con)
      writeLines('<tbody>', con)
      
      for (i in 1:nrow(tmp_print3_eig)){
        writeLines('<tr>', con)
        writeLines(paste('<th style="text-align:left;font-style:italic;">', rownames(tmp_print3_eig)[i], '</th>', sep=""), con)
        writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print3_eig[i,1], '</td>', sep=""), con)
        writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print3_eig[i,2], '</td>', sep=""), con)
        writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print3_eig[i,3], '</td>', sep=""), con)
        writeLines('</tr>', con)   
      }      
      writeLines('</tbody></table></center>', con)
      
      writeLines('<center><table frame="above" cellspacing="0" cellpadding="3px" style="margin-top:20px; margin-bottom:20px; border-top:solid 1px black; border-collapse:collapse;"><caption>Summary of the two first components</caption>', con)
      writeLines('<thead><tr style="background-color:lightgrey; border-bottom:solid 1px black;"><th style="text-align:left;">Variable</th><th style="text-align:center;width:100px;">Coord 1</th><th style="text-align:center;width:100px;">Cos2 1</th><th style="text-align:center;width:100px;">Contrib 1</th><th style="text-align:center;width:100px;">Coord 2</th><th style="text-align:center;width:100px;">Cos2 2</th><th style="text-align:center;width:100px;">Contrib 2</th></tr></thead>', con)
      writeLines('<tbody>', con)
      
      for (i in 1:nrow(tmp_print3_var)){
        writeLines('<tr>', con)
        writeLines(paste('<th style="text-align:left;font-style:italic;">', rownames(tmp_print3_var)[i], '</th>', sep=""), con)
        writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print3_var[i,1], '</td>', sep=""), con)
        writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print3_var[i,2], '</td>', sep=""), con)
        writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print3_var[i,3], '</td>', sep=""), con)
        writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print3_var[i,4], '</td>', sep=""), con)
        writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print3_var[i,5], '</td>', sep=""), con)
        writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_print3_var[i,6], '</td>', sep=""), con)
        writeLines('</tr>', con)
      }
      writeLines('</tbody></table></center>', con)
      
      tmp_plotname3_var=paste(name,"-step3-var.png", sep="")
      png(filename=tmp_plotname3_var)
      plot(x.SesIndex, step=3, choice="var")
      dev.off()
      tmp_plotname3_ind=paste(name,"-step3-ind.png", sep="")
      png(filename=tmp_plotname3_ind)
      plot(x.SesIndex, step=3, choice="ind", label="none")
      dev.off()
      
      writeLines('<center><table frame=box cellspacing="0" cellpadding ="2" style="margin-top:20px; margin-bottom:20px; border-collapse:collapse;"><tr><td>', con)
      writeLines(paste('<img src="',tmp_plotname3_var,'" alt="',tmp_plotname3_var,'" />', sep=""), con) 
      writeLines('</td><td>', con) 
      writeLines(paste('<img src="',tmp_plotname3_ind,'" alt="',tmp_plotname3_ind,'" />', sep=""), con) 
      writeLines('</td></tr>', con)
      writeLines('<tr style="background-color:lightgrey;"><td><center><b>Correlation circle</b></center></td><td><center><b>Projections of the units</b></center>', con) 
      writeLines('</td></tr></table></center></li>', con)
      
    }
    writeLines("<br /><br />", con)
    
    
    # Classification
    if (inherits(x, "SesClassif")) {
      writeLines('<hr style="color:black; background-color:black; height:3px; text-align:center; width:800px;" /><h2><a name="Classif">Classification</a></h2>', con)
      writeLines(paste('After the creation of the socioeconomic index, a classification is performed using ', x$call$method,'. ',length(table(x$table[,"Classes"])), ' classes are created using this technic.', sep=""), con)
      
      writeLines('<br />', con)
      tmp_plotname4_ind=paste(name,"-classif-ind.png", sep="")
      if (x$call$method %in% c("quantiles", "interval")){
        png(filename=tmp_plotname4_ind)
        plot(x, label="none")
        dev.off()
        
        writeLines('<center><table frame=box cellspacing="0" cellpadding ="2" style="margin-top:20px; margin-bottom:20px;border-collapse:collapse;"><tr><td>', con)
        writeLines(paste('<img src="',tmp_plotname4_ind,'" alt="',tmp_plotname4_ind,'" />', sep=""), con) 
        writeLines('</td></tr>', con)
        writeLines('<tr style="background-color:lightgrey;"><td><center><b>Projections of the units</b></center>', con) 
        writeLines('</td></tr></table></center></li>', con)   
      }      
      if (x$call$method %in% c("HC", "HC+knn")){
        tmp_plotname4_dendro=paste(name,"-classif-dendro.png", sep="")
        png(filename=tmp_plotname4_dendro)
        plot(x$analysis, choice="tree")
        dev.off()
        png(filename=tmp_plotname4_ind)
        plot(x$analysis, choice="map", draw.tree=F, label="none")
        dev.off() 
        
        writeLines('<center><table frame=box cellspacing="0" cellpadding ="2" style="margin-top:20px; margin-bottom:20px; border-collapse:collapse;"><tr><td>', con)
        writeLines(paste('<img src="',tmp_plotname4_dendro,'" alt="',tmp_plotname4_dendro,'" />', sep=""), con) 
        writeLines('</td><td>', con) 
        writeLines(paste('<img src="',tmp_plotname4_ind,'" alt="',tmp_plotname4_ind,'" />', sep=""), con) 
        writeLines('</td></tr>', con)
        writeLines('<tr style="background-color:lightgrey;"><td><center><b>Tree</b></center></td><td><center><b>Projections of the units</b></center>', con) 
        writeLines('</td></tr></table></center></li>', con)
      }
      tmp_select <- colnames(x$call$X$step3$analysis$call$X)
      tmp_x <- x$table[,c(tmp_select, "Classes")]
      ##tmp_x <- transform(tmp_x, Classes=as.factor(Classes))
      tmp_x$Classes <- as.factor(tmp_x$Classes)
      tmp_desc <- catdes(tmp_x, ncol(tmp_x))
      tmp_desc <- tmp_desc$quanti
      
      writeLines('The following tables contain descriptions of each classe by the variables selected in the socioeconomic index (only variables with a significant difference between the class and the whole sample are shown):', con)
 
      for (i in 1:length(tmp_desc)){
        writeLines('<center><table frame="above" cellspacing="0" cellpadding="3px" style="margin-top:20px; margin-bottom:20px; border-top:solid 1px black; border-collapse:collapse;">', con)
        writeLines(paste('<caption>Description by the variables of the class ',names(tmp_desc)[i], '</caption>', sep=""), con)
        writeLines('<thead><tr style="background-color:lightgrey; border-bottom:solid 1px black;"><th style="text-align:left;">Variable</th><th style="text-align:center;width:100px;">Mean in category</th><th style="text-align:center;width:100px;">Overall mean</th><th style="text-align:center;width:100px;">Sd in category</th><th style="text-align:center;width:100px;">Overall Sd</th></tr></thead>', con)
        writeLines('<tbody>', con)         
        
        tmp_class <- round(tmp_desc[[i]], 2)
        for (j in 1:nrow(tmp_class)){          
          writeLines('<tr>', con)
          writeLines(paste('<th style="text-align:left;font-style:italic;">', rownames(tmp_class)[j], '</th>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_class[j,2], '</td>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_class[j,3], '</td>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_class[j,4], '</td>', sep=""), con)
          writeLines(paste('<td style="text-align:center; font-family:Courier New;">', tmp_class[j,5], '</td>', sep=""), con)
          writeLines('</tr>', con)
        }
      }
      writeLines('</tbody></table></center>', con)
      
    }
    writeLines("<br />", con)     
    
    writeLines("</body></html>", con)
    close(con)
  }  
  
  if (export) {
  write.table(x.export, paste(name, ".csv", sep=""), sep=";",dec=".", row.names=FALSE, na="")
  }
  setwd(old.wd)
}
