\name{snpgdsLDMat}
\alias{snpgdsLDMat}
\title{
	Linkage Disequilibrium (LD) analysis
}
\description{
	Return a LD matrix for SNP pairs.
}
\usage{
snpgdsLDMat(gdsobj, sample.id=NULL, snp.id=NULL, slide=250,
	method=c("composite", "r", "dprime", "corr"), num.thread=1, verbose=TRUE)
}
\arguments{
	\item{gdsobj}{a GDS file object (\code{\link[gdsfmt]{gds.class}})}
	\item{sample.id}{a vector of sample id specifying selected samples; if NULL, all samples are used}
	\item{snp.id}{a vector of snp id specifying selected SNPs; if NULL, all SNPs are used}
	\item{slide}{# of SNPs, the size of sliding window, see details}
	\item{method}{"composite", "r", "dprime", "corr", see details}
	\item{num.thread}{the number of CPU cores used}
	\item{verbose}{if TRUE, show information}
}
\details{
	Four methods can be used to calculate linkage disequilibrium values:
"composite" for LD composite measure, "r" for R coefficient (by EM algorithm assuming
HWE, it could be negative), "dprime" for D', and "corr" for correlation coefficient.
The method "corr" is equivalent to "composite", when SNP genotypes are coded as:
0 -- BB, 1 -- AB, 2 -- AA.

	if \code{slide} <= 0, output a n-by-n LD matrix; otherwise, output a m-by-n LD matrix,
where n is the total number of SNPs, m is the size of sliding window.
	For n-by-n matrix, the value of i row and j column is LD of i and j SNPs.
	For m-by-n matrix, the value of i row and j column is LD of j and j+i SNPs.
}
\value{
	Return a list:
	\item{sample.id}{the sample ids used in the analysis}
	\item{snp.id}{the SNP ids used in the analysis}
	\item{LD}{a matrix of LD values}
	\item{slide}{the size of sliding window}
}
\references{
	Weir B: Inferences about linkage disequilibrium. Biometrics 1979; 35: 235-254.

	Weir B: Genetic Data Analysis II. Sunderland, MA: Sinauer Associates, 1996.

	Weir BS, Cockerham CC: Complete characterization of disequilibrium at two loci;
		in Feldman MW (ed): Mathematical Evolutionary Theory.
		Princeton, NJ: Princeton University Press, 1989.
}
\author{Xiuwen Zheng}
\seealso{
	\code{\link{snpgdsLDpair}}, \code{\link{snpgdsLDpruning}}
}

\examples{
library(lattice)

# open an example dataset (HapMap)
genofile <- openfn.gds(snpgdsExampleFileName())

# chromosome 15
snpset <- read.gdsn(index.gdsn(genofile, "snp.id"))[
	read.gdsn(index.gdsn(genofile, "snp.chromosome")) == 15]
length(snpset)

# LD matrix
ld.slide <- snpgdsLDMat(genofile, snp.id=snpset, method="composite")

# plot
levelplot(t(ld.slide$LD^2), col.regions = terrain.colors)


# LD matrix
ld.noslide <- snpgdsLDMat(genofile, snp.id=snpset, slide=-1, method="composite")

# plot
levelplot(t(ld.noslide$LD^2), col.regions = terrain.colors)


# close the genotype file
closefn.gds(genofile)
}

\keyword{gds}
\keyword{GWAS}
\keyword{LD}
