#' Removes temporary files created by Rsagacmd
#'
#' For convenience, functions in the Rsagacmd package create temporary files if
#' any outputs for a SAGA-GIS tool are not specified as arguments. Temporary
#' files in R are automatically removed at the end of each session. However,
#' when dealing with raster data, these temporary files potentially can consume
#' large amounts of disk space. These temporary files can be observed during a
#' session by using the saga_show_tmpfiles function, and can be removed using
#' the saga_remove_tmpfiles function. Note that this function also removes any
#' accompanying files, i.e. the '.prj' and '.shx' files that may be written as
#' part of writing a ESRI Shapefile '.shp' format
#'
#' @param h Remove temporary files that are older than h (in number of hours).
#'
#' @return Nothing is returned.
#' @export
#'
#' @examples
#' \dontrun{
#' # Remove all temporary files generated by Rsagacmd
#' saga_remove_tmpfiles(h = 0)
#' }
saga_remove_tmpfiles <- function(h = 0) {
  message(paste0("Removing Rsagacmd temporary files h=", h))
  for (f in pkg.env$sagaTmpFiles) {
    if (file.exists(f) == TRUE) {
      tdelay <- difftime(Sys.time(), file.mtime(f), units = "hours")
      if (tdelay > h) {
        message(f)
        assoc_files <- list.files(
          path = dirname(f),
          pattern = utils::glob2rx(
            paste0(tools::file_path_sans_ext(basename(f)), ".*")
          ),
          full.names = T
        )
        file.remove(assoc_files)
        pkg.env$sagaTmpFiles <- pkg.env$sagaTmpFiles[pkg.env$sagaTmpFiles != f]
      }
    } else {
      pkg.env$sagaTmpFiles <- pkg.env$sagaTmpFiles[pkg.env$sagaTmpFiles != f]
    }
  }
}


#' List temporary files created by Rsagacmd
#'
#' For convenience, functions in the Rsagacmd package create temporary files if
#' any outputs for a SAGA-GIS tool are not specified as arguments. Temporary
#' files in R are automatically removed at the end of each session. However,
#' when dealing with raster data, these temporary files potentially can consume
#' large amounts of disk space. These temporary files can be observed during a
#' session by using the saga_show_tmpfiles function, and can be removed using
#' the saga_remove_tmpfiles function.
#'
#' @return returns the file names of the files in the temp directory that have
#'   been generated by Rsagacmd. Note this list of files only includes the
#'   primary file extension, i.e. '.shp' for a shapefile without the accessory
#'   files (e.g. .prj, .shx etc.).
#' @export
#' @examples
#' \dontrun{
#' # Show all temporary files generated by Rsagacmd
#' saga_remove_tmpfiles(h = 0)
#' }
saga_show_tmpfiles <- function() {
  message("Rsagacmd temporary files:")
  for (f in pkg.env$sagaTmpFiles) {
    message(f)
  }
  return(pkg.env$sagaTmpFiles)
}


# local environment to store vector of tempfiles in package namespace
pkg.env <- new.env(parent = emptyenv())
pkg.env$sagaTmpFiles <- c()
