#' Compare effect sizes of a gene across all datasets in meta-analysis
#' 
#' @description A forest plot can be used to compare the expression values of a gene across different datasets. The size of the blue boxes is proportional to the number of samples in the study and light blue lines indicate the standard error of the effect sizes for each study (95\% confidence interval). The summary effect size for all studies is indicated as yellow diamond below and the width of the diamond indicates the summary standard error.
#' @usage forestPlot(metaObject, geneName, boxColor = "blue", whiskerColor = "lightblue", 
#' zeroLineColor = "black", summaryColor = "orange", textColor = "red")
#' @param metaObject a filtered metaObject, i.e. it needs to include a \code{filterObject} generated by the function \code{filterGenes()} 
#' @param geneName name of the gene for which the forest plot should be generated
#' @param boxColor desired color for the box (default: "blue")
#' @param whiskerColor desired color for the whiskers (default: "lightblue")
#' @param zeroLineColor desired color for the line indicating 0 (default: "black")
#' @param summaryColor desired color for the diamond representing the summary effect size (default: "orange")
#' @param textColor desired color for the text of the dataset names (default: "red")
#' @return Plot to compare effect sizes of a gene across datasets
#' @author Winston A. Haynes, Jiaying Toh
#' @seealso	\code{\link{filterGenes}},  \code{\link{runMetaAnalysis}},  \code{\link{violinPlot}}
#' @examples
#' # compare effect sizes of the Gene1 for all discovery datasets in tinyMetaObject 
#' forestPlot(tinyMetaObject, geneName="Gene1")
#' @keywords 
#' hplot 
#' graphs
#' @export
forestPlot<- function (metaObject, geneName, boxColor = "blue", whiskerColor = "lightblue", zeroLineColor = "black", summaryColor = "orange", textColor = "red") { 
  if(! checkDataObject(object = metaObject, objectType="Meta", objectStage="Pre-Filter")) {
		stop("metaObject that was passed to forestPlot was not appropriately formatted as a Meta object")
	} 
	
	#If there are results for our disease and gene, generate the plot
	if(geneName %in% rownames(metaObject$metaAnalysis$datasetEffectSizes)) {
		
		#Get the gene effect sizes
		studyData <- data.frame(means=metaObject$metaAnalysis$datasetEffectSizes[geneName,])
		
		#Get the standard errors
		studyData$SEs <- metaObject$metaAnalysis$datasetEffectSizeStandardErrors[geneName,]
		
		getFormattedName <- function(uglyName) {
			if(uglyName %in% names(metaObject$originalData)) {
				return(metaObject$originalData[uglyName][[1]]$formattedName)
			}
			return("")
		}
		
		#Get the study names
		studyData$names <- sapply(rownames(studyData),getFormattedName)
		
		print(class(studyData$names))
		
		#Order the results alphabetically
		studyData<- studyData[order(studyData$names),]
		
		#Pull objects out of the query results
		studyMeans <-studyData$means
		studySEs <- studyData$SEs
		studyNames <-studyData$names
		names(studyMeans) <- studyNames
		
		#Get the pooled results for our forest plot
		pooledMean<- metaObject$metaAnalysis$pooledResults[geneName,"effectSize"]
		pooledSE<- metaObject$metaAnalysis$pooledResults[geneName, "effectSizeStandardError"]
		
		rmeta::metaplot( studyMeans, studySEs, labels=studyNames,
		                 xlab="Standardized Mean Difference (log2 scale)", ylab="",
		                 colors=rmeta::meta.colors(box = boxColor, lines = whiskerColor, 
		                                    zero = zeroLineColor, summary = summaryColor, text = textColor),
		                 summn=pooledMean, sumse=pooledSE, sumnn=1/pooledSE^2, main=geneName)
	} else {
		stop(paste("No gene named \'", geneName, "\' in this metaObject. No plot generated", sep=""))
	}
}
