% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/meshes.R
\name{Mesh}
\alias{Mesh}
\title{Make a 3D mesh}
\usage{
Mesh(
  vertices,
  faces,
  mesh = NULL,
  triangulate = FALSE,
  clean = FALSE,
  normals = FALSE,
  numbersType = "double"
)
}
\arguments{
\item{vertices}{a numeric matrix with three columns, or a \code{bigq}
matrix with three columns if \code{numbersType="gmp"}}

\item{faces}{either an integer matrix (each row provides the vertex indices
of the corresponding face) or a list of integer vectors, each one
providing the vertex indices of the corresponding face}

\item{mesh}{if not \code{NULL}, this argument takes precedence over \code{vertices}
and \code{faces}, and must be either a list containing the fields \code{vertices}
and \code{faces} (objects as described above), otherwise a \strong{rgl} mesh
(i.e. a \code{mesh3d} object)}

\item{triangulate}{Boolean, whether to triangulate the faces; if \code{TRUE},
it is highly recommended to use an exact type of numbers, i.e. 
\code{numbersType="lazyExact"} or \code{numbersType="gmp"}}

\item{clean}{Boolean, whether to clean the mesh (merging duplicated
vertices, duplicated faces, removed isolated vertices)}

\item{normals}{Boolean, whether to compute the normals}

\item{numbersType}{the type of the numbers used in C++ for the
computations; must be one of \code{"double"}, \code{"lazyExact"}
(a type provided by CGAL for exact computations), or \code{"gmp"}
(exact computations with rational numbers); using exact computations can
improve the detection of the exterior edges}
}
\value{
A list giving the vertices, the edges, the faces of the mesh, the
  exterior edges, the exterior vertices and optionally the normals. This
  list has two additional components \code{edges0} and \code{normals0} if
  \code{triangulate=TRUE}, giving the edges and the normals before the
  triangulation, unless the mesh is already triangulated, in which case
  the \code{triangulate} option is ignored.
}
\description{
Make a 3D mesh from given vertices and faces; the returned
  faces are coherently oriented, normals are computed if desired, and
  triangulation is performed if desired.
}
\examples{
library(MeshesOperations)
library(rgl)

# a tetrahedron with ill-oriented faces ####
vertices <- rbind(
  c(-1, -1, -1),
  c(1, 1, -1),
  c(1, -1, 1),
  c(-1, 1, 1)
)
faces <- rbind(
  c(1, 2, 3),
  c(3, 4, 2),
  c(4, 2, 1),
  c(4, 3, 1)
)

# plot the tetrahedron, hiding the back of the faces
# then some faces do not appear, as their orientation is not correct
tmesh1 <- tmesh3d(
  vertices = t(vertices),
  indices = t(faces),
  homogeneous = FALSE
)
open3d(windowRect = c(50, 50, 562, 562))
shade3d(tmesh1, color = "green", back = "cull")

# now run the `Mesh` function
mesh2 <- Mesh(vertices, faces, normals = FALSE)
# plot the tetrahedron, hiding the back of the faces
# then all faces appear now
tmesh2 <- toRGL(mesh2)
open3d(windowRect = c(50, 50, 562, 562))
shade3d(tmesh2, color = "blue", back = "cull")

# illustration of the `clean` option ####
# we construct a mesh with a lot of duplicated vertices
library(misc3d) # to compute a mesh of an isosurface
a <- 0.94; mu <- 0.56; c <- 0.34 # cyclide parameters
f <- function(x, y, z, a, c, mu){ # implicit equation of the cyclide
  b <- sqrt(a^2 - c^2)
  (x^2 + y^2 + z^2 - mu^2 + b^2)^2 - 4*(a*x - c*mu)^2 - 4*b^2*y^2
}
x <- seq(-c - mu - a, abs(mu - c) + a, length.out = 45)
y <- seq(-mu - a, mu + a, length.out = 45)
z <- seq(-mu - c, mu + c, length.out = 30)
g <- expand.grid(x = x, y = y, z = z)
voxel <- array(with(g, f(x, y, z, a, c, mu)), c(45, 45, 30))
cont <- computeContour3d(voxel, level = 0, x = x, y = y, z = z)
ids <- matrix(1:nrow(cont), ncol = 3, byrow = TRUE)
# run the `Mesh` function with `clean=TRUE`
mesh <- Mesh(cont, ids, clean = TRUE, normals = TRUE)
# plot the cyclide
tmesh <- toRGL(mesh)
open3d(windowRect = c(50, 50, 562, 562), zoom = 0.9)
shade3d(tmesh, color = "green")

# illustration of the `triangulate` option ####
# the faces of the truncated icosahedron are hexagonal or pentagonal:
truncatedIcosahedron[["faces"]]
# so we triangulate them:
mesh <- Mesh(
  mesh = truncatedIcosahedron,
  triangulate = TRUE, normals = FALSE,
  numbersType = "lazyExact"
)
# now we can plot the truncated icosahedron
tmesh <- toRGL(mesh)
open3d(windowRect = c(50, 50, 562, 562), zoom = 0.9)
shade3d(tmesh, color = "orange")
}
