% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MODIStsp_extract.R
\name{MODIStsp_extract}
\alias{MODIStsp_extract}
\title{Extract data from MODIStsp time series}
\usage{
MODIStsp_extract(
  in_rts,
  sf_object,
  start_date = NULL,
  end_date = NULL,
  id_field = NULL,
  FUN = "mean",
  out_format = "xts",
  small = TRUE,
  small_method = "centroids",
  na.rm = TRUE,
  verbose = FALSE
)
}
\arguments{
\item{in_rts}{A \code{RasterStack} bject created by MODIStsp
(it MUST contain acquisition dates in the "Z" attribute)}

\item{sf_object}{"sf" object OR name of an GDAL-readable vector file specifying the
"area" from which data has to be extracted.
\itemize{
\item If \code{sf_object} represents lines, the output object contains one column for
each line, containing values obtained applying the function specified
as the FUN argument over all pixels touched by the line, and one line for
each date.
\item If \code{sf_object} represents points, the output object contains one column
for each point, containing values of the cells corresponding to the point,
and one line for each date.
\item If \code{sf_object} represents polygons, the output object contains one column
for each polygon, containing values obtained applying the function
specified as the FUN argument over all pixels belonging to the polygon,
and one line for each date
}}

\item{start_date}{object of class \code{Date}, \code{POSIXct} or \code{POSIXlt} OR \code{character}
coercible to Date class (format = "yyyy-mm-dd")Starting date of the period
to be considered for data extraction . If not provided, the first date of
the RasterStack is used.}

\item{end_date}{object of class \code{Date}, \code{POSIXct} or \code{POSIXlt} OR \code{character}
coercible to Date class (format = "yyyy-mm-dd"). Ending date of the period
to be considered for data extraction . If not provided, the last date of
the RasterStack is used.}

\item{id_field}{\code{character} name of the column of the input sp object or
shapefile to be used in the data extraction. Values contained in the column
MUST be unique. The names of the columns of the output are taken from this
column. If not provided, or an invalid value is provided, then the names
of the columns of the output reflect the number of the feature in
\code{sf_object}.}

\item{FUN}{function to summarize the values (e.g. mean) on polygon data frames.
The function should take a single numeric vector as argument and return a
single value (e.g. mean, min or max), and accept a na.rm argument. Thus,
standard R functions not including an na.rm argument must  be wrapped as in
this example: fun=function(x,...)length(x). Defaults to "mean"}

\item{out_format}{\code{character ["xts" | "dframe"]} If dframe, the output is a
data frame with dates in the first column and extracted data in the others,
otherwise it is a \code{xts} object, Default: "xts"}

\item{small}{\code{logical} If TRUE, and input is polygons, then values are
returned also for polygons not covering at least one raster cell. "Included"
cells in this case depend on the values of the "small_method" parameter.}

\item{small_method}{\code{character ["centroids" | "full"]} If small == TRUE and
input is polygons, controls which cells are "extracted" for small polygons.
If set to "centroids" (default), then only the cells corresponding to polygon
centroid are considered (faster, may have problems on strangely shaped
polygons). If set to "full", then all cells intersected by the small polygon
are extracted and used in calculations, Default: "centroids"}

\item{na.rm}{\code{logical} If TRUE, and sf_object is a polygon, then na.rm = TRUE
is used when applying FUN to the different pixels of the polygon, Default = TRUE.}

\item{verbose}{\code{logical} If TRUE, messages on processing status are sent
to the console. Default = TRUE.}
}
\value{
data.frame or xts object. Each column of data corresponds to one
point or one polygon, each row to a date.
}
\description{
function used to extract time series data from rts files created by MODIStsp
on spatial locations provided in the form of "R" spatial objects (SpatialPoints,
SpatialPolygons, etc.)
}
\details{
The function takes as input a RasterStack object containing time information
in the "z" attribute (set by \code{raster::setZ}), a starting and ending date
and a standard "R" spatial object, and returns the time series for the spatial locations
specified in the spatial object in the form of a "R" xts object OR a plain data.frame
with a "date" column in first position.
If the input spatial object is a "point" or "line" one, the  output object
contains one column for each specified point, or for each cell intersecting
the line, and one line for each date. If the input spatial object is a "polygon"
one, the output object contains one column for each polygon, containing values
obtained applying the function specified as the FUN argument over all pixels
belonging to the polygon, and one line for each date.
}
\note{
License: GPL 3.0
}
\examples{
\dontrun{
# Extract average and standard deviation values from a rts object created by
# MODIStsp for each polygon of a shapefile, for each date in the period
# between 2001-01-01 and 2014-12-31

# The example uses tif files in testdata/VI_16Days_500m_v61 to build
# a MODIStsp rasterStack corresponding to the 2016 time series of the NDVI index
# over the Como Lake (Italy). It then extracts data on polygons corresponding
# to different land cover classes saved in testdata/extract_polys.shp

# First, prepare the test dataset.
# __NOTE__ To avoid redownloading, here we copy some test data from MODIStsp
# installation folder to tempdir and use it to create a test time series.

test_zip <-  system.file("testdata/VI_16Days_500m_v6/NDVI.zip",
                         package = "MODIStsp")
dir.create(file.path(tempdir(), "MODIStsp/VI_16Days_500m_v61"),
           showWarnings = FALSE, recursive = TRUE)
utils::unzip(test_zip,
             exdir = file.path(tempdir(), "MODIStsp/VI_16Days_500m_v61"))

opts_file <- system.file("testdata/test_extract.json", package = "MODIStsp")
MODIStsp(opts_file = opts_file, gui = FALSE, verbose = FALSE)

# Now load the MODIStsp stack: This is a MODIS NDVI time series ranging between
# 2016-01-01 and 2016-12-18
# __NOTE__: MODIStsp rasterStack files are always saved in the "Time_Series\/RData"
# subfolder of your main output folder - see
# "https://docs.ropensci.org/MODIStsp/articles/output.html")

# Specify the filename of the RData RasterStack of interest
stack_file  <- file.path(tempdir(),
 "MODIStsp/VI_16Days_500m_v61/Time_Series/RData/Terra/NDVI",
  "MOD13A1_NDVI_1_2016_353_2016_RData.RData")
basename(stack_file)

ts_data <- get(load(stack_file))
ts_data

# Now load a shapefile containing polygons from which we want to extract data

polygons <- sf::st_read(system.file("testdata/extract_polys.shp",
                           package = "MODIStsp"), quiet = TRUE)
polygons

# Finally, extract the average values for each polygon and date and plot the
# results

out_dataavg <- suppressMessages(MODIStsp_extract(ts_data, polygons, id_field = "lc_type",
                             small = FALSE))
head(out_dataavg)

plot(out_dataavg, legend.loc = "topleft")

# use a different summarization function

out_datasd <- MODIStsp_extract(ts_data, polygons, id_field = "lc_type",
                              FUN = "sd", small = FALSE)
head(out_datasd)

# (See also https://docs.ropensci.org/MODIStsp/articles/Analyze.html for a
# worked-out example)
}
}
\author{
Lorenzo Busetto, phD (2015 - 2017)
email: busetto.l@irea.cnr.it
}
