% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/0_LambertW-toolkit.R, R/create_LambertW_input.R, R/create_LambertW_output.R
\name{LambertW-toolkit}
\alias{LambertW-toolkit}
\alias{create_LambertW_input}
\alias{create_LambertW_output}
\title{Do-it-yourself toolkit for Lambert W\eqn{ \times} F distribution}
\usage{
create_LambertW_input(distname = NULL, beta, input.u = list(beta2tau = NULL,
  d = NULL, p = NULL, r = NULL, q = NULL, distname = "MyFavoriteDistribution"))

create_LambertW_output(LambertW.input = NULL, theta = NULL,
  distname = LambertW.input$distname)
}
\arguments{
\item{distname}{string; name of input distribution; see \code{\link{get_distnames}}.}

\item{beta}{vector; parameter \eqn{\boldsymbol \beta} of the input distribution.
See \code{\link{check_beta}} on how to specify \code{beta} for each distribution.}

\item{input.u}{optional; users can make their own 'Lambert W\eqn{    imes} F' distribution by
supplying the necessary functions. See Description for details.}

\item{LambertW.input}{an object of class \code{LambertW_input}}

\item{theta}{list; a (possibly incomplete) list of parameters \code{alpha}, \code{beta}, \code{gamma},
\code{delta}. \code{\link{complete_theta}} fills in default values for missing entries.}
}
\value{
\code{create_LambertW_output} returns a list of class \code{LambertW_output}
with values that are (for the most part) functions themselves (see Examples):

\item{d}{ pdf of Y \eqn{\sim} Lambert W \eqn{\times} 'MyFavoriteDistribution',}
\item{p}{ cdf of Y,}
\item{q}{ quantile function for Y,}
\item{r}{ random number generator for Y,}
\item{distname}{ character string with the name of the new distribution.
Format: "Lambert W x 'MyFavoriteDistribution'",}
\item{beta, theta}{see Arguments,}
\item{distname.with.beta}{name of the new distribution
including the parameter \code{beta}. Format: "Lambert W x 'MyFavoriteDistribution'(beta)".}
}
\description{
This do-it-yourself Lambert W\eqn{ \times} F toolkit implements the flexible input/output
framework of Lambert W \eqn{\times} F random variables (see References).
Using a modular approach, it allows users to create their own Lambert W \eqn{\times}
'MyFavoriteDistribution' RVs. See Details below.

If the distribution you inted to use is not already implemented
(\code{\link{get_distnames}}), then you can create it:
\describe{
\item{create input:}{use \code{\link{create_LambertW_input}} with your
favorite distribution,}
\item{create output:}{pass it as an input argument to \code{\link{create_LambertW_output}},}
\item{use output:}{use Rs standard functionality for distributions
such as random number generation (\code{rY}), pdf (\code{dY}) and cdf
(\code{pY}), quantile function (\code{qY}), etc. for this newly generated
Lambert W \eqn{\times} 'MyFavoriteDistribution'.}
}

\strong{IMPORTANT:} This toolkit functionality is still under active development;
function names, arguments, return values, etc. may be subject to change.

\code{create_LambertW_output} converts the input \code{LambertW_input} representing
random variable \eqn{X \sim F_X} to the Lambert W \eqn{\times} \eqn{F_X} output.
}
\details{
\code{\link{create_LambertW_output}} takes an object of class \code{LambertW_input} and
creates a class \code{LambertW_output} for standard distributions as well as
the user-defined distribution. This \code{LambertW_output} represents the RV
Y \eqn{\sim} Lambert W \eqn{\times} 'MyFavoriteDistribution' with all its
properties and R functionality, such as random number generation
(\code{rY}), pdf (\code{dY}) and cdf (\code{pY}), etc.

\code{create_LambertW_input} allows users to define their own Lambert W\eqn{\times} F distribution by supplying the necessary
functions about the input random variable \eqn{U} and \eqn{\boldsymbol \beta}.
Here \eqn{U} is the zero mean and/or unit variance version of
\eqn{X \sim  F_X(x \mid \boldsymbol \beta)} (see References).

The argument \code{input.u} must be a list containing all of the following:
\describe{
\item{\code{beta2tau}}{ R function of \code{(beta)}: converts \eqn{\boldsymbol \beta} to \eqn{\tau} for the
user defined distribution }
\item{\code{distname}}{ optional; users can specify the name
of their input distribution. By default it's called \code{"MyFavoriteDistribution"}.
The distribution name will be used in plots and summaries of the Lambert W\eqn{\times} F
input (and output) object.}
\item{\code{d}}{ R function of \code{(u, beta)}: probability density function (pdf) of U,}
\item{\code{p}}{ R function of \code{(u, beta)}: cumulative distribution function (cdf) of U,}
\item{\code{q}}{ R function of \code{(p, beta)}: quantile function of U,}
\item{\code{r}}{ R function \code{(n, beta)}: random number generator for U,}
}
}
\examples{
# create a Gaussian N(1,2) input
Gauss.input <- create_LambertW_input("normal", beta = c(1, 2))

# create a heavy-tailed version of a normal
# gamma = 0, alpha = 1 are set by default; beta comes from input
params <- list(delta = c(0.3))
LW.Gauss <- create_LambertW_output(LambertW.input = Gauss.input,
                                   theta = params)
LW.Gauss

op <- par(no.readonly=TRUE)
par(mfrow = c(2, 1), mar = c(3, 3, 2, 1))
curve(LW.Gauss$d(x, params), -7, 10, col = "red")
# parameter will get detected automatically from the input
curve(LW.Gauss$d(x), -7, 10, col="blue") # same in blue;

# compare to the input case (i.e. set delta = 0)
params.0 <- params
params.0$delta <- 0

# to evaluate the RV at a different parameter value,
# it is necessary to pass the new parameter
curve(LW.Gauss$d(x, params.0), -7, 10, add=TRUE, col=1) #' par(op)

curve(LW.Gauss$p(x, params), -7, 10, col="red")
curve(LW.Gauss$p(x, params.0), -7, 10, add=TRUE, col=1)

test_normality(LW.Gauss$r(n=100), add.legend = FALSE)

## generate a positively skewed version of a shifted, scaled t_3
t.input <- create_LambertW_input("t", beta = c(2, 1, 3))
t.input
params <- list(gamma = 0.05) # skew it
LW.t <- create_LambertW_output(LambertW.input = t.input, theta = params)
LW.t

plot(t.input$d, -7, 11, col = 1)
plot(LW.t$d, -7, 11, col = 2, add = TRUE)
abline(v = t.input$beta["location"], lty = 2)

# draw samples from the skewed t_3
yy <- LW.t$r(n=100)
test_normality(yy)

### create a skewed exponential distribution
exp.input <- create_LambertW_input("exp", beta = 1)
plot(exp.input)
params <- list(gamma = 0.2)
LW.exp <- create_LambertW_output(exp.input, theta = params)
plot(LW.exp)

# create a heavy-tail exponential distribution
params <- list(delta = 0.2)
LW.exp <- create_LambertW_output(exp.input, theta = params)
plot(LW.exp)

# create a skewed chi-square distribution with 5 df
chi.input <- create_LambertW_input("chisq", beta = 5)
plot(chi.input)
params <- list(gamma = sqrt(2)*0.2)
LW.chi <- create_LambertW_output(chi.input, theta = params)
plot(LW.chi)
# a demo on how a user-defined U input needs to look like
user.tmp <- list(d = function(u, beta) dnorm(u),
                 r = function(n, beta) rnorm(n),
                 p = function(u, beta) pnorm(u),
                 q = function(p, beta) qnorm(p),
                 beta2tau = function(beta) {
                   c(mu_x = beta[1], sigma_x = beta[2],
                     gamma = 0, alpha = 1, delta = 0)
                   },
                 distname = "MyNormal")
my.input <- create_LambertW_input(input.u = user.tmp, beta = c(0, 1))
my.input
plot(my.input)
}
\author{
Georg M. Goerg
}
\keyword{datagen}
\keyword{distribution}
\keyword{models}
\keyword{univar}

