% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/anova.R
\name{anova.lmm}
\alias{anova.lmm}
\title{Multivariate Tests For Linear Mixed Model}
\usage{
\method{anova}{lmm}(
  object,
  effects = NULL,
  robust = FALSE,
  rhs = NULL,
  df = !is.null(object$df),
  ci = TRUE,
  transform.sigma = NULL,
  transform.k = NULL,
  transform.rho = NULL,
  transform.names = TRUE,
  ...
)
}
\arguments{
\item{object}{a \code{lmm} object. Only relevant for the anova function.}

\item{effects}{[character or numeric matrix] Should the Wald test be computed for all variables (\code{"all"}),
or only variables relative to the mean (\code{"mean"} or \code{"fixed"}),
or only variables relative to the variance structure (\code{"variance"}),
or only variables relative to the correlation structure (\code{"correlation"}).
Can also be use to specify linear combinations of coefficients or a contrast matrix, similarly to the \code{linfct} argument of the \code{multcomp::glht} function.}

\item{robust}{[logical] Should robust standard errors (aka sandwich estimator) be output instead of the model-based standard errors.}

\item{rhs}{[numeric vector] the right hand side of the hypothesis. Only used when the argument effects is a matrix.}

\item{df}{[logical] Should a F-distribution be used to model the distribution of the Wald statistic. Otherwise a chi-squared distribution is used.}

\item{ci}{[logical] Should an estimate, standard error, confidence interval, and p-value be output for each hypothesis?}

\item{transform.sigma, transform.k, transform.rho, transform.names}{are passed to the \code{vcov} method. See details section in \code{\link{coef.lmm}}.}

\item{...}{Not used. For compatibility with the generic method.}
}
\value{
A data.frame (LRT) or a list of containing the following elements (Wald):\itemize{
\item \code{multivariate}: data.frame containing the multivariate Wald test.
The column \code{df.num} refers to the degrees of freedom for the numerator (i.e. number of hypotheses)
wherease the column \code{df.denum} refers to the degrees of freedom for the denominator (i.e. Satterthwaite approximation).
\item \code{univariate}: data.frame containing each univariate Wald test.
\item \code{glht}: used internally to call functions from the multcomp package.
\item \code{object}: list containing key information about the linear mixed model.
\item \code{vcov}: variance-covariance matrix associated to each parameter of interest (i.e. hypothesis).
\item \code{iid}: matrix containing the influence function relative to each parameter of interest (i.e. hypothesis).
\item \code{args}: list containing argument values from the function call.
}
}
\description{
Simultaneous tests of linear combinations of the model paramaters using Wald tests or Likelihood Ratio Test (LRT).
}
\details{
By default adjustment of confidence intervals and p-values for multiple comparisons is based on the distribution of the maximum-statistic.
This is refered to as a single-step Dunnett multiple testing procedures in table II of Dmitrienko et al. (2013).
It is performed using the multcomp package with the option \code{test = adjusted("single-step")} with equal degrees of freedom
or by simulation using a Student's t copula with unequal degrees of freedom (more in the note of the details section of \code{\link{confint.Wald_lmm}}).
}
\examples{
#### simulate data in the long format ####
set.seed(10)
dL <- sampleRem(100, n.times = 3, format = "long")

#### fit Linear Mixed Model ####
eUN.lmm <- lmm(Y ~ visit + X1 + X2 + X5,
               repetition = ~visit|id, structure = "UN", data = dL)

#### Multivariate Wald test ####
## F-tests
anova(eUN.lmm)
anova(eUN.lmm, effects = "all")
anova(eUN.lmm, robust = TRUE, df = FALSE)
summary(anova(eUN.lmm))

## user defined F-test
summary(anova(eUN.lmm, effects = c("X1=0","X2+X5=10")))
print(anova(eUN.lmm, effects = "mean_visit"), columns = add("null"))

## chi2-tests
anova(eUN.lmm, df = FALSE)


## with standard contrast
if(require(multcomp)){
amod <- lmm(breaks ~ tension, data = warpbreaks)
e.amod <- anova(amod, effect = mcp(tension = "Tukey"))
summary(e.amod)
}

#### Likelihood ratio test ####
eUN0.lmm <- lmm(Y ~ X1 + X2, repetition = ~visit|id, structure = "UN", data = dL)
anova(eUN.lmm, eUN0.lmm) 

eCS.lmm <- lmm(Y ~ X1 + X2 + X5, repetition = ~visit|id, structure = "CS", data = dL)
anova(eUN.lmm, eCS.lmm)
}
\references{
Dmitrienko, A. and D'Agostino, R., Sr (2013), Traditional multiplicity adjustment methods in clinical trials. Statist. Med., 32: 5172-5218. https://doi.org/10.1002/sim.5990.
}
\seealso{
\code{\link{summary.Wald_lmm}} or \code{\link{confint.Wald_lmm}} for a summary of the results. \cr
\code{\link{autoplot.Wald_lmm}} for a graphical display of the results. \cr
\code{\link{rbind.Wald_lmm}} for combining result across models and adjust for multiple comparisons. \cr
}
\keyword{htest}
