% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wrapper.r
\name{GET.variogram}
\alias{GET.variogram}
\title{Variogram and residual variogram with global envelopes}
\usage{
GET.variogram(
  object,
  nsim = 999,
  data = NULL,
  ...,
  GET.args = NULL,
  savefuns = TRUE
)
}
\arguments{
\item{object}{An object of class \code{gstat} or a \code{variogram.formula}.
In the first case, direct (residual) variograms are calculated for the variable
defined in object. Only one variable allowed.
In the second case, a formula defining the response vector and (possible) regressors,
in case of absence of regressors, use e.g. z~1. See \code{\link[gstat]{variogram}}.}

\item{nsim}{The number of permutations.}

\item{data}{A data frame where the names in formula are to be found. If NULL,
the data are assumed to be found in the \code{object}.}

\item{...}{Additional parameters to be passed to \code{\link[gstat]{variogram}}.}

\item{GET.args}{A named list of additional arguments to be passed to \code{\link{global_envelope_test}}.}

\item{savefuns}{Logical. If TRUE, then the functions from permutations are saved to the attribute
simfuns.}
}
\description{
The function accompanies the function \code{\link[gstat]{variogram}} with global envelopes
that are based on permutations of the variable(s) or residuals for which the variogram is calculated.
Therefore, one can inspect the hypothesis of "no spatial autocorrelation" of the variable or the residuals
of the fitted model.
}
\examples{
if(require("sp", quietly=TRUE) & require("gstat", quietly=TRUE)) {
  # Examples from gstat complemented with global envelopes
  #-------------------------------------------------------
  data(meuse)
  coordinates(meuse) <- ~x+y
  # topsoil zinc concentration, mg kg-1 soil ("ppm")
  bubble(meuse, "zinc",
         col=c("#00ff0088", "#00ff0088"), main="zinc concentrations (ppm)")
  # Variogram can be calculated as follows by the function variogram of the gstat library.
  # The function variogram takes a formula as its first argument:
  # log(zinc)~1 means that we assume a constant trend for the variable log(zinc).
  lzn.vgm <- variogram(object=log(zinc)~1, data=meuse)
  plot(lzn.vgm)
  # Variogram with global envelopes is as easy:
  \donttest{lzn.vgm.GET <- GET.variogram(object=log(zinc)~1, data=meuse)}
  \dontshow{lzn.vgm.GET <- GET.variogram(object=log(zinc)~1, data=meuse, nsim=4, GET.args=list(alpha=0.2))}
  plot(lzn.vgm.GET)

  # Instead of the constant mean, denoted by ~1, a mean function can
  # be specified, e.g. using ~sqrt(dist) as a predictor variable:
  lznr.vgm <- variogram(log(zinc)~sqrt(dist), meuse)
  # In this case, the variogram of residuals with respect
  # to a fitted mean function are shown.
  plot(lznr.vgm)
  # The variogram with global envelopes (obtained by permuting the residuals):
  \donttest{lznr.vgm.GET <- GET.variogram(object=log(zinc)~sqrt(dist), data=meuse)}
  \dontshow{lznr.vgm.GET <- GET.variogram(object=log(zinc)~sqrt(dist), data=meuse, nsim=4, GET.args=list(alpha=0.2))}
  plot(lznr.vgm.GET)

  # Directional variograms
  lzn.dir <- variogram(object=log(zinc)~1, data=meuse, alpha=c(0, 45, 90, 135))
  plot(lzn.dir)
  # with global envelopes
  \donttest{lzn.dir.GET <- GET.variogram(object=log(zinc)~1, data=meuse, alpha=c(0, 45, 90, 135))}
  \dontshow{lzn.dir.GET <- GET.variogram(object=log(zinc)~1, data=meuse, nsim=4, alpha=c(0, 45, 90, 135), GET.args=list(alpha=0.2))}
  plot(lzn.dir.GET, base_size=10)

  # Use instead gstat objects
  g <- gstat(id="ln.zinc", formula=log(zinc)~1, data=meuse)
  # or: g <- gstat(id="ln.zinc", formula=log(zinc)~sqrt(dist), data=meuse)
  # The variogram
  plot(variogram(g))
  # The variogram with global envelopes:
  \donttest{g.GET <- GET.variogram(object=g)}
  \dontshow{g.GET <- GET.variogram(object=g, nsim=4, GET.args=list(alpha=0.2))}
  plot(g.GET)
}
}
