% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/maxlogLreg.R
\encoding{UTF-8}
\name{maxlogLreg}
\alias{maxlogLreg}
\title{Maximum Likelihood Estimation for parametric linear regression models}
\usage{
maxlogLreg(
  formulas,
  y_dist,
  data = NULL,
  subset = NULL,
  fixed = NULL,
  link = NULL,
  start = NULL,
  lower = NULL,
  upper = NULL,
  optimizer = "nlminb",
  control = NULL,
  silent = FALSE,
  ...
)
}
\arguments{
\item{formulas}{a list of formula objects. Each element must have an \code{~}, with the terms
on the right separated by \code{+} operators. The response variable on the left side is optional.
Linear predictor of each parameter must be specified with the name of the parameter followed by
the suffix \code{'.fo'}. See the examples below for further illustration.}

\item{y_dist}{a formula object that specifies the distribution of the response variable.
On the left side of \code{~} must be the response, and in the right side
must be the name o de probability density/mass function. See the section
\strong{Details} and the examples below for further illustration.}

\item{data}{an optional data frame containing the variables in the model. If data is not
specified, the variables are taken from the environment from which
\code{maxlogLreg} is called.}

\item{subset}{an optional vector specifying a subset of observations to be used
in the fitting process.}

\item{fixed}{a list with fixed/known parameters of distribution of interest. Fixed parameters
must be passed with its name and its value (known).}

\item{link}{a list with names of parameters to be linked, and names of the link function object.
For names of parameters, please visit documentation of density/mass function.
There are three link functions available: \code{\link{log_link}},
\code{\link{logit_link}} and \code{\link{NegInv_link}}. Take into account: the order
used in argument \code{over} corresponds to the order in argument \code{link}.}

\item{start}{a numeric vector with initial values for the parameters to be estimated. Zero is the
default value.}

\item{lower}{a numeric vector with lower bounds, with the same lenght of
argument \code{start} (for box-constrained optimization). \code{-Inf} is the default value.}

\item{upper}{a numeric vector with upper bounds, with the same lenght of
argument \code{start} (for box-constrained optimization). \code{Inf} is the default value.}

\item{optimizer}{a lenght-one character vector with the name of optimization routine.
\code{\link{nlminb}}, \code{\link{optim}} and
\code{\link[DEoptim]{DEoptim}} are available; \code{\link{nlminb}}
is the default routine.}

\item{control}{control parameters of the optimization routine. Please, visit documentation of selected
optimizer for further information.}

\item{silent}{logical. If TRUE, warnings of \code{maxlogL} are suppressed.}

\item{...}{Further arguments to be supplied to the optimization routine.}
}
\value{
A list with class \code{maxlogL} containing the following
lists:
\item{fit}{A list with output information about estimation and method used.}
\item{inputs}{A list with all input arguments.}
\item{outputs}{A list with additional information:
\itemize{
\item Number of parameters.
\item Sample size
\item Standard error computation method.
\item Number of regression parameters.
}
}
}
\description{
Function to compute maximum likelihood estimators (MLE) of regression parameters
of any distribution implemented in \code{R} with covariates (linear predictors).
}
\details{
\code{maxlogLreg} calculates computationally the log-likelihood (log L) function
corresponding to the distribution specified in argument \code{y_dist} with linear
predictors specified in argument \code{formulas}. Then, it maximizes the log L through
\code{\link{optim}}, \code{\link{nlminb}} or \code{\link{DEoptim}}. \code{maxlogLreg}
generates an S3 object of class \code{maxlogL}.

Noncentrality parameters must be named as \code{ncp} in the distribution.
}
\note{
The following generic functions can be used with a \code{maxlogL} object:
\code{summary, print, logLik, AIC}.
}
\examples{
library(EstimationTools)

#--------------------------------------------------------------------------------
# Example 1: Estimation in simulated normal distribution
n <- 1000
x <- runif(n = n, -5, 6)
y <- rnorm(n = n, mean = -2 + 3 * x, sd = exp(1 + 0.3* x))
norm_data <- data.frame(y = y, x = x)

# It does not matter the order of distribution parameters
formulas <- list(sd.fo = ~ x, mean.fo = ~ x)

norm_mod <- maxlogLreg(formulas, y_dist = y ~ dnorm, data = norm_data,
                       link = list(over = "sd", fun = "log_link"))
summary(norm_mod)


#--------------------------------------------------------------------------------
# Example 2: Fitting with censorship
# (data from https://www.itl.nist.gov/div898/handbook/apr/section4/apr413.htm)

failures <- c(55, 187, 216, 240, 244, 335, 361, 373, 375, 386)
fails <- c(failures, rep(500, 10))
status <- c(rep(1, length(failures)), rep(0, 10))
Wei_data <- data.frame(fails = fails, status = status)

# Formulas with linear predictors
formulas <- list(scale.fo=~1, shape.fo=~1)

# Bounds for optimization. Upper bound set with default values (Inf)
start <- list(
  scale = list(Intercept = 100),
  shape = list(Intercept = 10)
)
lower <- list(
  scale = list(Intercept = 0),
  shape = list(Intercept = 0)
)

mod_weibull <- maxlogLreg(formulas, y_dist = Surv(fails, status) ~ dweibull,
                          start = start,
                          lower = lower, data = Wei_data)
summary(mod_weibull)


#--------------------------------------------------------------------------------

}
\references{
\insertRef{Nelder1965}{EstimationTools}

\insertRef{Fox1978}{EstimationTools}

\insertRef{Nash1979}{EstimationTools}

\insertRef{Dennis1981}{EstimationTools}
}
\seealso{
\code{\link{summary.maxlogL}}, \code{\link{optim}}, \code{\link{nlminb}},
\code{\link{DEoptim}}, \code{\link{DEoptim.control}},
\code{\link{maxlogL}}, \code{\link{bootstrap_maxlogL}}

Other maxlogL: 
\code{\link{maxlogL}()}
}
\author{
Jaime Mosquera Gutiérrez, \email{jmosquerag@unal.edu.co}
}
\concept{maxlogL}
