library(EpiModel)
library(shiny)

shinyServer(function(input, output) {
  
  ## Main reactive functions
  param <- reactive({
    vital <- ifelse(input$b.rate > 0 |
                    input$ds.rate > 0 |
                    input$di.rate > 0 |
                    input$dr.rate > 0, TRUE, FALSE)
    l <- list(trans.rate = input$trans.rate, 
              act.rate = input$act.rate, 
              rec.rate = input$rec.rate, 
              b.rate = input$b.rate, 
              ds.rate = input$ds.rate, 
              di.rate = input$di.rate, 
              dr.rate = input$dr.rate,
              groups = 1,
              vital = vital)
    class(l) <- "param.icm"
    return(l)
  })
  init <- reactive({
    if (input$modtype == "SIR") {
      l <- list(s.num = input$s.num, 
                i.num = input$i.num, 
                r.num = input$r.num)
    } else {
      l <- list(s.num = input$s.num, 
                i.num = input$i.num)
    }
    l$status.rand <- TRUE
    class(l) <- "init.icm"
    return(l)
  })
  control <- reactive({
    l <- list(type = input$modtype,
              nsteps = input$nsteps,
              nsims = input$nsims,
              rec.rand = TRUE,
              b.rand = TRUE,
              d.rand = TRUE,
              verbose = FALSE,
              initialize.FUN = initialize.icm,
              infection.FUN = infection.icm,
              recovery.FUN = recovery.icm,
              deaths.FUN = deaths.icm,
              births.FUN = births.icm,
              get_prev.FUN = get_prev.icm)
    class(l) <- "control.icm"
    return(l)
  })
  mod <- reactive({
    input$runMod
    isolate(icm(param(), init(), control()))
  })
  showqnts <- reactive({
    ifelse(input$qntsrng == 0, FALSE, input$qntsrng)
  })

  ## Main Plot tab
  output$MainPlot <- renderPlot({
    par(mar = c(3.5, 3.5, 1.2, 1), mgp = c(2.1, 1, 0))
    if (input$compsel == "Compartment Prevalence") {
      plot(mod(), 
           mean.line = input$showmean,
           sim.lines = input$showsims,
           qnts = showqnts(),
           leg = input$showleg,
           leg.cex = 1.1, 
           lwd = 3.5,
           main = "")
    }
    if (input$compsel == "Compartment Size") {
      plot(mod(), 
           popfrac = FALSE, 
           mean.line = input$showmean,
           sim.lines = input$showsims,
           qnts = showqnts(),
           leg = input$showleg,
           leg.cex = 1.1, 
           lwd = 3.5,
           main = "")
    }
    if (input$compsel == "Disease Incidence") {
      plot(mod(), 
           y = "si.flow", 
           popfrac = FALSE, 
           mean.line = input$showmean,
           sim.lines = input$showsims,
           qnts = showqnts(),
           leg = input$showleg,
           leg.cex = 1.1, 
           lwd = 3.5,
           main = "")
    }
  })
  output$dlMainPlot <- downloadHandler(
    filename = "MainPlot.pdf",
    content = function(file) {
      pdf(file = file, height = 6, width = 10)
      par(mar = c(3.5, 3.5, 1.2, 1), mgp = c(2.1, 1, 0))
      if (input$compsel == "Compartment Prevalence") {
        plot(mod(), 
             mean.line = input$showmean,
             sim.lines = input$showsims,
             qnts = showqnts(),
             leg = input$showleg,
             leg.cex = 1.1, 
             lwd = 3.5)
      }
      if (input$compsel == "Compartment Size") {
        plot(mod(), 
             mean.line = input$showmean,
             sim.lines = input$showsims,
             qnts = showqnts(),
             leg = input$showleg,
             popfrac = FALSE, 
             leg.cex = 1.1, 
             lwd = 3.5)
      }
      if (input$compsel == "Disease Incidence") {
        plot(mod(), 
             y = "si.flow", 
             popfrac = FALSE, 
             mean.line = input$showmean,
             sim.lines = input$showsims,
             qnts = showqnts(),
             leg = input$showleg,
             leg.cex = 1.1, 
             lwd = 3.5)
      }
      dev.off()
    }
  )

  ## Summary and Compartment plot tab
  # Outfrom from summary
  output$outSummary <- renderPrint({
    summary(mod(), 
            at = input$summTs, 
            digits = input$summDig)
  })

  # Comp_plot
  output$CompPlot <- renderPlot({
    comp_plot(mod(), 
              at = input$summTs, 
              digits = input$summDig)
  })
  
  # Download for comp_plot
  output$dlCompPlot <- downloadHandler(
    filename = "CompPlot.pdf",
    content = function(file) {
      pdf(file = file, height = 6, width = 10)
      comp_plot(mod(), 
                at = input$summTs, 
                digits = input$summDig)
      dev.off()
    }
  )
  
  
  ## Data tab
  output$simnoControl <- renderUI({
    input$runMod
    maxsims <- isolate(input$nsims)
    sliderInput(inputId = "datasim", 
                label = strong("Simulation Number"), 
                min = 1, 
                max = maxsims, 
                value = 1, 
                step = 1)
  })
  output$outData <- renderDataTable({
    if (input$datasel == "Means") {
      as.data.frame(mod())
    } else if (input$datasel == "Standard Deviations") {
      as.data.frame(mod(), out = "sd")
    } else if (input$datasel == "Simulations") {
      as.data.frame(mod(), out = "vals", sim = max(1, input$datasim))
    }
  }, options = list(aLengthMenu = c(10, 25, 50, 100), iDisplayLength = 10))
  output$dlData <- downloadHandler(
    filename = "ModelData.csv",
    content = function(file) {
      if (input$datasel == "Means") {
        write.csv(as.data.frame(mod()), file)
      } else if (input$datasel == "Standard Deviations") {
        write.csv(as.data.frame(mod(), out = "sd"), file)
      } else if (input$datasel == "Simulations") {
        write.csv(as.data.frame(mod(), out = "vals", sim = input$datasim), file)
      }
      
    }
  )
  

})
