% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/D3mirt.R
\name{D3mirt}
\alias{D3mirt}
\title{3D DMIRT Model Estimation}
\usage{
D3mirt(x, constructs = NULL)
}
\arguments{
\item{x}{Data frame with rows for items and columns for model parameters. The number of columns must be more than or equal to 4, i.e., three columns with (\emph{a}) parameters and at least one column for difficulty (\emph{d}) parameters.}

\item{constructs}{Optional. Nested lists with integer values indicating construct. The default is \code{constructs = NULL}.}
}
\value{
S3 object of class \code{D3mirt} with lists of \emph{a} and \emph{d} parameters from the M2PL or MGRM estimation, multidimensional difficulty (MDIFF), multidimensional discrimination (MDISC), direction cosines and degrees for vector angles, construct lists, and vector coordinates.
}
\description{
Descriptive multidimensional item response theory modeling (DMIRT; Reckase, 2009, 1985, Reckase & McKinley, 1991) for dichotomous and polytomous items restricted to three dimensions.
}
\details{
The \code{D3mirt()} function takes in a data frame of factor slopes and difficulty parameters from a compensatory three-dimensional multidimensional two-parameter logistic model (M2PL) or a multidimensional graded
response model (MGRM), fitted in accordance with descriptive item response theory model specifications described below.
The function returns an S3 object containing descriptive multidimensional item response theory estimates that can be graphed as vector arrows representing item response functions in a three-dimensional space with \link{plotD3mirt}.

Note, model parameters from the multidimensional M2PL or MGRM must be assessed prior to using the \code{D3mirt()} function (see examples section below or the vignette included in the package).
This means that the model must first be identified (see \link{modid} for more on model identification).
}
\section{DMIRT Theory}{
In DMIRT analysis, also called within multidimensional modeling, it is assumed that items in a multidimensional ability space can measure single or multiple latent traits (Reckase, 2009, 1985; Reckase & McKinley, 1991).
The methodology is a type of data reduction technique for the compensatory model (Reckase, 2009), i.e., a type of measurement model that uses linear combinations of ability estimates.
The method seeks to maximize item discrimination and so is \emph{descriptive} because the results describe the extent to which items in a test are unidimensional,
i.e., that the items discriminate on one dimension only, or are within-multidimensional, i.e., that the items discriminate on more than one dimension.

The most central estimates in DMIRT analysis are the single multidimensional discrimination (MDISC) parameter and the multidimensional difficulty (MDIFF) parameters (Reckase2009, 1985; Reckase & McKinley, 1991).
In addition, if constructs are used (see below) the output will also contain the directional discrimination (DDISC) parameters for all items estimated in the direction set by the construct vectors.
This makes it possible to compare item discrimination under the assumption that they measure the same construct.

Using the parameters from the compensatory model, the \code{D3mirt()} function computes parameters describing the location and direction of the highest possible discrimination for each item.
The output can be visualized with the \link{plotD3mirt} function that use vector geometry with vector arrows indicating level of difficulty and direction of the maximum discrimination.
}

\section{Constructs}{
The user has the option of including constructs in the estimation.
Constructs, in this context, refer to the assumption that a subset of items can measure a higher-order latent variable.
To include constructs, the user must create one or more nested lists that indicate what items belong to what construct (see the examples section below).
From this, the \code{D3mirt()} function calculates direction cosines for the constructs by adding and normalizing the direction cosines for the items in the nested lists.
The construct vector arrows can contribute to the analysis by visualizing the average direction of multidimensional discrimination for a subset set of items.
}

\examples{
\donttest{
# Load data
data("anes0809offwaves")
x <- anes0809offwaves
x <- x[,3:22] # Remove columns for age and gender

# Fit a three-dimensional graded response model with orthogonal factors
# Example below uses Likert items from the built-in data set "anes0809offwaves"
# Item W7Q3 and item W7Q20 was selected with `modid()`
# The model specification set all items in the data set (1-20)
# to load on all three factors (F1-F3)
# The START and FIXED commands are used on the two items to identify the DMIRT model
spec <- '  F1 = 1-20
           F2 = 1-20
           F3 = 1-20

           START=(W7Q3,a2,0)
           START=(W7Q3,a3,0)

           START=(W7Q20,a3,0)

           FIXED=(W7Q3,a2)
           FIXED=(W7Q3,a3)

           FIXED=(W7Q20,a3) '


mod1 <- mirt::mirt(x,
                   spec,
                   itemtype = 'graded',
                   SE = TRUE,
                   method = 'QMCEM')

# Assign a data frame with factor loadings (located in the first three columns in mod1),
# and difficulty parameters (columns 4-8 in mod1)
d <- data.frame(mirt::coef(mod1,
                           simplify=TRUE)$'items'[,1:8])

# Call D3mirt() with data frame d
g <- D3mirt(d)
summary(g) # Show summary of results

# Call to D3mirt(), including optional nested lists for three constructs
# Item W7Q16 is not included in any construct because of model violations
# The model violations for the W7Q16 item can be seen when plotting the model
c <- list(list(1,2,3,4),
          list(5,7,8,9,10),
          list(11,12,13,14,15,15,16,17,18,19,20))
g <- D3mirt(d, c)
summary(g)
}
}
\references{
Chalmers, R., P. (2012). mirt: A Multidimensional Item Response Theory Package for the R Environment. \emph{Journal of Statistical Software, 48}(6), 1-29.

Reckase, M. D. (2009). \emph{Multidimensional Item Response Theory}. Springer.

Reckase, M. D. (1985). The Difficulty of Test Items That Measure More Than One Ability. \emph{Applied Psychological Measurement, 9}(4), 401-412–412. https://doi-org.ezp.sub.su.se/10.1177/014662168500900409

Reckase, M. D., & McKinley, R. L. (1991). The Discriminating Power of Items That Measure More Than One Dimension. \emph{Applied Psychological Measurement, 15}(4), 361-373–373. https://doi-org.ezp.sub.su.se/10.1177/014662169101500407
}
\author{
Erik Forsberg
}
