% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/connectivity_estimation.R
\encoding{UTF-8}
\name{d.rel.conn.unif.prior}
\alias{d.rel.conn.unif.prior}
\alias{p.rel.conn.unif.prior}
\alias{q.rel.conn.unif.prior}
\title{Estimate the probability distribution of relative connectivity values 
assuming a uniform prior distribution}
\usage{
d.rel.conn.unif.prior(phi, p, k, n, log = FALSE, ...)

p.rel.conn.unif.prior(phi, p, k, n, log = FALSE, ...)

q.rel.conn.unif.prior(q, p, k, n, log = FALSE, ...)
}
\arguments{
\item{phi}{Vector of fractions of individuals (i.e., eggs) from the source 
population settling at the destination population}

\item{p}{Fraction of individuals (i.e., eggs) marked in the source population}

\item{k}{Number of marked settlers found in sample}

\item{n}{Total number of settlers collected}

\item{log}{If \code{TRUE}, returns natural logarithm of probabilities, except
for \code{\link{q.rel.conn.unif.prior}}, which expects log of probabilities
as inputs}

\item{q}{Vector of quantiles}

\item{\dots}{Extra arguments to Beta distribution functions.  See 
\code{\link{dbeta}} for details.  For expert use only.}
}
\value{
Vector of probabilities or quantiles.
}
\description{
These functions calculate the probability density function 
(\code{d.rel.conn.unif.prior}), the probability distribution function (aka 
the cumulative distribution function; \code{p.rel.conn.unif.prior}) and the 
quantile function (\code{q.rel.conn.unif.prior}) for the relative (to all 
settlers at the destination site) connectivity value for larval transport 
between a source and destination site given a known fraction of marked 
individuals (i.e., eggs) in the source population.  A uniform prior is used
for the relative connectivity value.
}
\details{
Estimations of the probability distribution are derived from the Beta 
distribution (see \code{\link{dbeta}}) and should be exact to great 
precision.
}
\section{Functions}{
\itemize{
\item \code{d.rel.conn.unif.prior}: Returns the probability density for 
relative connectivity between a pair of sites

\item \code{p.rel.conn.unif.prior}: Returns the cumulative probability
distribution for relative connectivity between a paire of sites

\item \code{q.rel.conn.unif.prior}: Estimates quantiles for the probability
distribution function for relative connectivity between a pair of sites
}}
\examples{
library(ConnMatTools)

k <- 10 # Number of marked settlers among sample
n.obs <- 87 # Number of settlers in sample
n.settlers <- 100 # Total size of settler pool

p <- 0.4 # Fraction of eggs that was marked
phi <- seq(0,1,length.out=101) # Values for relative connectivity

# Probability distribution assuming infinite settler pool and uniform prior
drc <- d.rel.conn.unif.prior(phi,p,k,n.obs)
prc <- p.rel.conn.unif.prior(phi,p,k,n.obs)
qrc <- q.rel.conn.unif.prior(c(0.025,0.975),p,k,n.obs) # 95\% confidence interval

# Test with finite settlement function and large (approx. infinite) settler pool
# Can be a bit slow for large settler pools
dis <- d.rel.conn.finite.settlement(0:(7*n.obs),p,k,n.obs,7*n.obs)

# Quantiles
qis <- q.rel.conn.finite.settlement(c(0.025,0.975),p,k,n.obs,7*n.obs)

# Finite settler pool
dfs <- d.rel.conn.finite.settlement(0:n.settlers,p,k,n.obs,n.settlers)

# Quantiles for the finite settler pool
qfs <- q.rel.conn.finite.settlement(c(0.025,0.975),p,k,n.obs,n.settlers)

# Make a plot of different distributions
plot(phi,drc,type="l",main="Probability of relative connectivity values",
     xlab=expression(phi),ylab="Probability density")
lines(phi,prc,col="blue")
lines((0:(7*n.obs))/(7*n.obs),dis*(7*n.obs),col="black",lty="dashed")
lines((0:n.settlers)/n.settlers,dfs*n.settlers,col="red",lty="dashed")
abline(v=qrc,col="black")
abline(v=qis/(7*n.obs),col="black",lty="dashed")
abline(v=qfs/n.settlers,col="red",lty="dashed")

}
\author{
David M. Kaplan \email{dmkaplan2000@gmail.com}
}
\references{
Kaplan DM, Cuif M, Fauvelot C, Vigliola L, Nguyen-Huu T, Tiavouane J and Lett C 
  (in press) Uncertainty in empirical estimates of marine larval connectivity. 
  ICES Journal of Marine Science. doi:10.1093/icesjms/fsw182.
}
\seealso{
Other connectivity.estimation: \code{\link{d.rel.conn.beta.prior}},
  \code{\link{d.rel.conn.dists.func}},
  \code{\link{d.rel.conn.finite.settlement}},
  \code{\link{d.rel.conn.multinomial.unnorm}},
  \code{\link{d.rel.conn.multiple}},
  \code{\link{dual.mark.transmission}},
  \code{\link{optim.rel.conn.dists}},
  \code{\link{r.marked.egg.fraction}}
}

