\name{binomial.bymCAR}
\alias{binomial.bymCAR}
%- Also NEED an '\alias' for EACH other topic documented here.

\title{
Fit the BYM  conditional autoregressive (CAR) model to spatial binomial data
}

\description{
The function fits a binomial logistic random effects models to spatial data, where the random effects are modelled by the BYM conditional autoregressive (CAR) model (Besag et. al. 1991). The model represents the logit transform of the mean function for the set of binomial responses  by a combination of covariates and two sets of random effects. For the latter, the first set are independent, while the second are spatially correlated and come from the IAR model. A set of offsets can also be included on the linear predictor scale. Inference is based on Markov Chain Monte Carlo (MCMC) simulation, using a combination of Gibbs sampling and Metropolis steps.
}

\usage{
binomial.bymCAR(formula, beta = NULL, phi = NULL, theta = NULL, tau2 = NULL, 
sigma2 = NULL, trials, W, burnin = 0, n.sample = 1000, blocksize.beta = 5, 
blocksize.phi = 10, blocksize.theta = 10, prior.mean.beta = NULL, 
prior.var.beta = NULL, prior.max.tau2 = NULL, prior.max.sigma2 = NULL)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{formula}{
A formula for the covariate part of the model, using the same notation as for the lm() function. The offsets should also be included here using the offset() function.
}
  \item{beta}{
A vector of starting values for the regression parameters (including the intercept term). If this argument is not specified the function will randomly generate starting values.
}
  \item{phi}{
A vector of starting values for the correlated random effects. If this argument is not specified the function will randomly generate starting values.
}
  \item{theta}{
A vector of starting values for the independent random effects. If this argument is not specified the function will randomly generate starting values.
}
  \item{tau2}{
A starting value for the variance parameter of the correlated random effects. If this argument is not specified the function will randomly generate a starting value.
}
  \item{sigma2}{
A starting value for the variance parameter of the independent random effects. If this argument is not specified the function will randomly generate a starting value.
}  
  \item{trials}{
A vector the same length as the response containing the total number of trials for each area.
}
   \item{W}{
A binary n by n neighbourhood matrix (where n is the number of spatial units). The jkth element equals one if areas (j, k) are spatially close (e.g. share a common border) and is zero otherwise. 
}
  \item{burnin}{
The number of MCMC samples to discard as the burnin period. Defaults to 0.
}
  \item{n.sample}{
The number of MCMC samples to generate. Defaults to 1,000.
}
  \item{blocksize.beta}{
The size of the blocks in which to update the regression parameters in the MCMC algorithm. Defaults to 5.
}
  \item{blocksize.phi}{
The size of the blocks in which to update the correlated random effects in the MCMC algorithm. Defaults to 10.
}
  \item{blocksize.theta}{
The size of the blocks in which to update the independent random effects in the MCMC algorithm. Defaults to 10.
}
  \item{prior.mean.beta}{
A vector of prior means for the regression parameters beta (Gaussian priors are assumed). Defaults to a vector of zeros.
}
  \item{prior.var.beta}{
A vector of prior variances for the regression parameters beta (Gaussian priors are assumed). Defaults to a vector with values 1000.
}
  \item{prior.max.tau2}{
The maximum allowable value for the correlated random effects variance tau2 (a Uniform(0,M) prior is assumed). Defaults to M=1000.
}
  \item{prior.max.sigma2}{
The maximum allowable value for the independent random effects variance sigma2 (a Uniform(0,M) prior is assumed). Defaults to M=1000.
}
}



\details{
For further details about how to apply the function see the vignette to the CARBayes package.
}


\value{
\item{formula }{The formula for the covariate and offset part of the model.}
\item{samples.beta }{A matrix of MCMC samples for the regression parameters beta.}
\item{samples.phi }{A matrix of MCMC samples for the correlated random effects phi.}
\item{samples.theta }{A matrix of MCMC samples for the independent random effects theta.}
\item{samples.tau2 }{A matrix of MCMC samples for the correlated random effects variance tau2.}
\item{samples.sigma2 }{A matrix of MCMC samples for the independent random effects variance sigma2.}
\item{fitted.values }{A summary matrix of the posterior distributions of the fitted values for each area. The summaries include: Mean, Sd, Median, and credible interval.}
\item{random.effects }{A summary matrix of the posterior distributions of the random effects for each area. The summaries include: Mean, Sd, Median, and credible interval.}
\item{residuals }{A vector of raw residuals from the model.}
\item{DIC }{The Deviance Information Criterion.}
\item{p.d }{The effective number of parameters in the model.}
\item{summary.results }{A summary table of the parameters.}
}

\references{
Besag, J., J. York, and A. Mollie (1991). Bayesian image restoration with two applications in spatial statistics. Annals of the Institute of Statistics and Mathematics 43, 1-59.
}

\author{
Duncan Lee
}


\examples{
##################################################
#### Run the model on simulated data on a lattice
##################################################

#### Set up a square lattice region
x.easting <- 1:10
x.northing <- 1:10
Grid <- expand.grid(x.easting, x.northing)
n <- nrow(Grid)

#### set up distance and neighbourhood (W, based on sharing a common border) matrices
distance <-array(0, c(n,n))
W <-array(0, c(n,n))
  for(i in 1:n)
	{
		for(j in 1:n)
		{
		temp <- (Grid[i,1] - Grid[j,1])^2 + (Grid[i,2] - Grid[j,2])^2
		distance[i,j] <- sqrt(temp)
			if(temp==1)  W[i,j] <- 1 
		}	
	}
	
	
#### Generate the covariates and response data
x1 <- rnorm(n)
x2 <- rnorm(n)
theta <- rnorm(n, sd=0.05)
phi <- mvrnorm(n=1, mu=rep(0,n), Sigma=0.4 * exp(-0.1 * distance))
logit <- x1 + x2 + theta + phi
prob <- exp(logit) / (1 + exp(logit))
trials <- rep(50,n)
Y <- rbinom(n=n, size=trials, prob=prob)


#### Run the BYM model
#### Let the function randomly generate starting values for the parameters
#### Use the default priors specified by the function (for details see the help files)
formula <- Y ~ x1 + x2
\dontrun{model <- binomial.bymCAR(formula=formula, trials=trials, W=W, burnin=5000, 
n.sample=10000)}
\dontshow{model <- binomial.bymCAR(formula=formula, trials=trials, W=W, burnin=20, 
n.sample=50)}
}