% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/binogcp.r
\name{binogcp}
\alias{binogcp}
\title{Binomial sampling with a general continuous prior}
\usage{
binogcp(x, n, density = c("uniform", "beta", "exp", "normal", "user"),
  params = c(0, 1), n.pi = 1000, pi = NULL, pi.prior = NULL,
  plot = TRUE)
}
\arguments{
\item{x}{the number of observed successes in the binomial experiment.}

\item{n}{the number of trials in the binomial experiment.}

\item{density}{may be one of "beta", "exp", "normal", "student", "uniform"
or "user"}

\item{params}{if density is one of the parameteric forms then then a vector
of parameters must be supplied.  beta: a, b exp: rate normal: mean, sd
uniform: min, max}

\item{n.pi}{the number of possible \eqn{\pi}{pi} values in the prior}

\item{pi}{a vector of possibilities for the probability of success in a
single trial. This must be set if density = "user".}

\item{pi.prior}{the associated prior probability mass. This must be set if
density = "user".}

\item{plot}{if \code{TRUE} then a plot showing the prior and the posterior
will be produced.}
}
\value{
A list will be returned with the following components:
\item{likelihood}{the scaled likelihood function for \eqn{\pi}{pi} given
\eqn{x} and \eqn{n}} \item{posterior}{the posterior probability of
\eqn{\pi}{pi} given \eqn{x} and \eqn{n}} \item{pi}{the vector of possible
\eqn{\pi}{pi} values used in the prior} \item{pi.prior}{the associated
probability mass for the values in \eqn{\pi}{pi}}
}
\description{
Evaluates and plots the posterior density for \eqn{\pi}{pi}, the probability
of a success in a Bernoulli trial, with binomial sampling and a general
continuous prior on \eqn{\pi}{pi}
}
\examples{

## simplest call with 6 successes observed in 8 trials and a continuous
## uniform prior
binogcp(6, 8)

## 6 successes, 8 trials and a Beta(2, 2) prior
binogcp(6, 8,density = "beta", params = c(2, 2))

## 5 successes, 10 trials and a N(0.5, 0.25) prior
binogcp(5, 10, density = "normal", params = c(0.5, 0.25))

## 4 successes, 12 trials with a user specified triangular continuous prior
pi = seq(0, 1,by = 0.001)
pi.prior = rep(0, length(pi))
priorFun = createPrior(x = c(0, 0.5, 1), wt = c(0, 2, 0))
pi.prior = priorFun(pi)
results = binogcp(4, 12, "user", pi = pi, pi.prior = pi.prior)

## find the posterior CDF using the previous example and Simpson's rule
myCdf = cdf(results)
plot(myCdf, type = "l", xlab = expression(pi[0]),
	   ylab = expression(Pr(pi <= pi[0])))

## use the quantile function to find the 95\% credible region.
qtls = quantile(results, probs = c(0.025, 0.975))
cat(paste("Approximate 95\% credible interval : ["
	, round(qtls[1], 4), " ", round(qtls, 4), "]\\n", sep = ""))

## find the posterior mean, variance and std. deviation
## using the output from the previous example
post.mean = mean(results)
post.var = var(results)
post.sd = sd(results)

# calculate an approximate 95\% credible region using the posterior mean and
# std. deviation
lb = post.mean - qnorm(0.975) * post.sd
ub = post.mean + qnorm(0.975) * post.sd

cat(paste("Approximate 95\% credible interval : ["
	, round(lb, 4), " ", round(ub, 4), "]\\n", sep = ""))

}
\seealso{
\code{\link{binobp}} \code{\link{binodp}}
}
\keyword{misc}
