#'@title MRMC Dataset Creator From Hit Rate.
#'@description From hit rates, data of hits are created.
#'@param NL  Number of Lesions.
#'@param p.truth Array of dimension ( C, M, Q), where
#'  C = number of confidence levels,
#'  M = number of modalities,
#'  Q = number of readers.
#'@param seed  The seed for creating hits which are generated by the binomial distributions with the specified seed.
#'@return Hits Data
#'@keywords True positives, binomial distributions, hit rate = probability of binomial distributions.
#'@inheritParams fit_Bayesian_FROC
#'@inheritParams DrawCurves
#'@inheritParams validation.dataset_srsc
#'@examples
#' \donttest{
#'
#'#  Using the default hit values, hit data are created as follows;
#'
#'            hits <- hits_creator_from_rate()
#'
#'
#'# If user want to use their own hit rates, then please input as same as
#'# the default values manner.
# ####1#### ####2#### ####3#### ####4#### ####5#### ####6#### ####7#### ####8#### ####9####
#'#================The third example======================================
#'
#'#     The hits rate cannot take any values, since there is a trend that a hit rate of
#'#   a higher confidence level is a higher. So, If it is difficult for user to create
#'#   a true hit rates, then  by taking estimates as true parameters,
#'#   user can replicate datasets.
#'#     To do so, work follow is first fitting, secondly extracting estimates,
#'#   thirdly apply this function (hits_creator_from_rate() ).
#'
#'
#'# * Fitting
#'
#'      fit <- fit_Bayesian_FROC(
#'              dataList.Chakra.Web.orderd,
#'              ite = 1111,  #  For simplicity, we take small MCMC samples.
#'              summary =FALSE)
#'
#'# * Extracting
#'
#'         estimates <- extract_estimates_MRMC(fit)
#'
#'          ppp <- estimates$ppp.EAP
#'
#'#      Note that ppp is an array
#'#     whose dimension is constituted by number of confidence levels, modalities, readers.
#'
#'
#'# *  Replicating as an true values is ppp
#'
#'
#'         hits  <-   hits_creator_from_rate(p.truth = ppp )
#'
#'
#'# <<Remark>>
#'#     ppp is an array. But ignoring its dimension, we can write that
#'
#'#          hits ~ Binomial(ppp, NL)
#'
#'#    Where NL is a number of lesions.
#'
#'#   By writing its component explicitly, we can write
#'
#'#        Hits[c,m,r] ~ Binomial(ppp[c,m,r], NL)
#'
#'#        Where c means the c-th confidence level,
#'#              m  means the m-th modality,
#'#              r means the r-th reader.
#'}# dottest

#' @export hits_creator_from_rate
#'
#'
#'
#'

hits_creator_from_rate <- function(
  # C= dim(p.truth)[1],
  # M= dim(p.truth)[2],
  # Q= dim(p.truth)[3],
  NL=252,
  seed =123,
  p.truth = BayesianFROC::p_truth
  #  array(c(
  #    c(0.03,0.13,0.2,0.3,0.4,   #for M=1 Q=1
  #      0.04,0.23,0.3,0.4,0.5) , #for M=2 Q=1 ,
  #
  #    c(0.05,0.33,0.4,0.5,0.6,   #for M=1 Q=2
  #      0.06,0.43,0.5,0.6,0.7)  ,#for M=2 Q=2 ,
  #
  #    c(0.07,0.53,0.6,0.7,0.8,   #for M=1 Q=3
  #      0.08,0.63,0.7,0.8,0.9)   #for M=2 Q=3 ,
  #      ),
  #
  # dim = c(5,2,3) #C M Q
  # )#array

){#function

  # if ( missing(p.truth)==TRUE) {
  #   ppp_truth_example <- BayesianFROC::ppp_truth_example
  #   p.truth <- ppp_truth_example
  # }
  if(missing(p.truth)==TRUE){message("\n* We use the default hit rates (= BayesianFROC::ppp_truth_example ) .\n\n\n")}

  C <- dim(p.truth)[1]
  M <- dim(p.truth)[2]
  Q <- dim(p.truth)[3]

  hits <- array(NA,dim = c(C,M,Q))

  # ?possible to Vectorization
  # set.seed(seed =  seed); hits <-   stats::rbinom(n=1,size = NL,prob = p.truth)
  for (cd in 1:C) {
    for (md in 1:M) {
      for (qd in 1:Q) {
        set.seed(seed =  seed); hits[cd,md,qd] <-   stats::rbinom(n=1,size = NL,prob = p.truth[cd,md,qd] )
      }
    }
  }





  return(hits)
}#function









#'@title MRMC Hit Rates Creator from Thresholds, Mean and S.D.
#'@description From thresholds, data of hit rate are created.
#'@param z.truth  Vector of dimension = C represents the thresholds of bi-normal assumption.
#'@param mu.truth array of dimension (M,Q). Mean of represents the signal distribution of bi-normal assumption.
#'@param v.truth  array of dimension (M,Q). Standard Deviation of represents the signal distribution of bi-normal assumption.

#'@return  ppp:= hit rate, which is an array with dimension
#'@inheritParams fit_Bayesian_FROC
#'@inheritParams DrawCurves
#'@inheritParams validation.dataset_srsc
#'@inheritParams hits_creator_from_rate
#'@examples
#'
#' \donttest{

#'#================The first example======================================
#'
#'#     Using default values for hit rates, we can create a data of hits as follows:
#'
#'         hits.rate <-hits_rate_creator()
#'
#'#================The second example======================================
#'
#'#     Using the hit rate from the hits_rate_creator(), we can get the hits data:
#'
#'         hits_creator_from_rate(p.truth =hits_rate_creator() )
#'
#'#================The remark for example======================================
#'
#'# The author does not show how to specify the hit rates or threshods.
#'# For the details of it, please see the default values of such a quantities.
#'
#'
#'#================The 4-th example======================================
#'
#'     p.truth.array <- hits_rate_creator()
#'
#'}# dottest

#'
#' @export hits_rate_creator
#'
#'
#'
#'

hits_rate_creator <- function(
  z.truth  = BayesianFROC::z_truth, #c(0.1,0.2,0.3,0.4,0.5),
  mu.truth = BayesianFROC::mu_truth, #array(1:6/10,c(M,Q)),
  v.truth  = BayesianFROC::v_truth #array(1:6/10,c(M,Q)),
){
  C <- length(z.truth)
  M <- dim(mu.truth)[1]
  Q <- dim(mu.truth)[2]
  ppp <- array(NA,dim = c(C,M,Q))

  for(md in 1 : M) {
    for(qd in 1 : Q) {
      for(cd in 1 : C) {
        if (cd==C){
          ppp[cd,md,qd] <- 1- stats::pnorm((z.truth[cd ] - mu.truth[md,qd])/v.truth[md,qd]);
        }else{
          ppp[cd,md,qd] <- stats::pnorm((z.truth[cd+1] - mu.truth[md,qd])/v.truth[md,qd])
          - stats::pnorm((z.truth[cd ] - mu.truth[md,qd])/v.truth[md,qd]);
        }
      }#cd

    }}# for md qd


  return(ppp)

}#function






#'@title MRMC Hit  Creator from thresholds, mean and S.D.
#'@description From threshold, mean and S.D.,  data of hit rate are created.
#'@param z.truth  Vector of dimension = C represents the thresholds of bi-normal assumption.
#'@param mu.truth array of dimension (M,Q). Mean of represents the signal distribution of bi-normal assumption.
#'@param v.truth  array of dimension (M,Q). Standard Deviation of represents the signal distribution of bi-normal assumption.



#'@return Hits Data for MRMC. The reason that hits is multiple reader and multiple modalities arise from the multiple indices of
#'mean and S.D. of signal distribution of the bi-normal assumption.

#'@inheritParams fit_Bayesian_FROC
#'@inheritParams DrawCurves
#'@inheritParams validation.dataset_srsc
#'@inheritParams hits_creator_from_rate
#'@inheritParams hits_rate_creator
#'@examples
#' \donttest{

#' hits.rate.p <-hits_from_thresholds()
#'
#'
#'}#donttest

#' @export hits_from_thresholds
#'
#'
#'
#'
hits_from_thresholds <-function(
  z.truth=BayesianFROC::z_truth, #c(0.1,0.2,0.3,0.4,0.5),
  mu.truth = BayesianFROC::mu_truth, #array(1:6/10,c(M,Q)),
  v.truth  = BayesianFROC::v_truth, #array(1:6/10,c(M,Q)),
  NL=252,
  seed =123
){
  M <- dim(mu.truth)[1]
  Q <- dim(mu.truth)[2]

  p.truth <- hits_rate_creator(
    z.truth=z.truth,
    mu.truth = mu.truth,
    v.truth  = v.truth
  )





  h <- hits_creator_from_rate(
    p.truth =p.truth  ,
    NL=NL,
    seed =seed
  )

  return(
    list(
      h=h,

      p.truth=p.truth,
      z.truth=z.truth,
      M = M,
      Q = Q,
      mu.truth = mu.truth,
      v.truth  = v.truth
    )
  )

}#function















#'@title MRMC False Alarm Creator.
#'@description From threshold, mean and S.D., data of False Alarm are created.
#'@details From threshold, mean and S.D.
#'of the latent Gaussian noise distribution in the bi-normal assumption,
#' data of False Alarm are created.
#' For the process of this drawing false alarm samples, its rate are also created. So, in the return values of the function, the rates for each confidence level is also attached.
#'
#'@param z.truth  Vector of dimension = C represents the thresholds of bi-normal assumption.
#'@param seed  The seed for creating false alarm which are generated by the Poisson  distributions using the specified seed.
#'@param NI  The number of images.
#'@param NL  The number of lesions.
#'@inheritParams fit_Bayesian_FROC
#'@inheritParams DrawCurves
#'@inheritParams validation.dataset_srsc
#'@inheritParams hits_creator_from_rate

#'@examples
#'  \donttest{

#' false.rate <- false_and_its_rate_creator()
#'
#'
#'}
#' @export
#'
#'
#'


false_and_its_rate_creator <- function(
  z.truth=BayesianFROC::z_truth, #c(0.1,0.2,0.3,0.4,0.5),
  NI =333,
  NL=111,
  ModifiedPoisson =FALSE,
  seed =12345
){

  C <-length(z.truth)

  if(ModifiedPoisson==F){ NX <- NI}
  if(ModifiedPoisson==T){ NX <- NL}



  f.inv <- vector()#these should in for sentence
  f <- vector()


  l.truth  <- vector()
  dl.truth <- vector()

  for(cd in 1 : C) {
    l.truth <- -log( stats::pnorm(z.truth))
  }






  for(cd in 1:C){
    if (cd==C) {dl.truth[cd]<-abs(l.truth[cd]-0);
    }else{

      dl.truth[cd]=abs(l.truth[cd]-l.truth[cd+1]);#not fabs for C++
    }
  }





  for (cd in 1:C) {

    if(ModifiedPoisson==F){
      if (cd==C) {
        set.seed(seed =  seed);
        f.inv[cd]  <- stats::rpois(n= 1, lambda = (l.truth[cd]-0)*NI )
      }else{
        set.seed(seed =  seed);
        f.inv[cd]  <- stats::rpois(n= 1, lambda = (l.truth[cd]-l.truth[cd+1])*NI )
      }#else
    }# if  ModifiedPoisson==F




    if(ModifiedPoisson==T){
      if (cd==C) {
        set.seed(seed =  seed);
        f.inv[cd]  <- stats::rpois(n= 1, lambda = (l.truth[cd]-0)*NL )
      }else{
        set.seed(seed =  seed);
        f.inv[cd]  <- stats::rpois(n= 1, lambda = (l.truth[cd]-l.truth[cd+1])*NL)
      }#else      set.seed(seed =  seed); hits <-   stats::rbinom(n=1,size = NL,prob = p.truth[cd])
    }#  if ModifiedPoisson==T

    f[C-cd+1] <- f.inv[cd]

  }#  for cd in 1:C

  return(
    list(
      f =f,
      l.truth  =l.truth,
      dl.truth =dl.truth
    )
  )

}#function









#'@title MRMC: False Alarm Creator For each Modality and each Reader.
#'@description From threshold, mean and S.D., data of False Alarm are created.#'@details
#'In our hierarchical model, false alarm rate does not depend on the readers or modalities. Thus this sampling function merely draws samples from the Poisson distribution of the same false alarm rate. Of course, this same false rate of the Poisson distributions is not desired one. Since we should assume that each reader with different modality should differ. To accomplish this, we have to assume that threshold parameter of Gaussian assumption should depend on the reader and modality. But such model does not converge in the Hamiltonian Monte Carlo simulation.
#'@inheritParams false_and_its_rate_creator
#'@inheritParams fit_Bayesian_FROC
#'@inheritParams DrawCurves
#'@inheritParams validation.dataset_srsc
#'@inheritParams hits_creator_from_rate

#' @param M Number of modalities
#' @param Q Number of readers
#'
#' @return Vector for false alarms as an element of list of MRMC data.

#' @export
#'
#' @examples
#' \donttest{


#'   false_and_its_rate_creator_MRMC()
#'
#'
#'}
false_and_its_rate_creator_MRMC <- function(
  z.truth=BayesianFROC::z_truth, #c(0.1,0.2,0.3,0.4,0.5),
  NI =333,
  NL=111,
  ModifiedPoisson =FALSE,
  seed =12345,
  M=5,
  Q=4,
  summary = TRUE
){


  C <-length(z.truth)
  f<-list()
  s <- 0

  for (md in 1:M) {
    for (qd in 1:Q) {
      s <- s+1
      a<- false_and_its_rate_creator(
        z.truth=z.truth, #c(0.1,0.2,0.3,0.4,0.5),
        NI =NI,
        NL=NL,
        ModifiedPoisson =ModifiedPoisson,
        seed =seed+100*md+1000*qd
      )
      f[[s]] <-a$f

    }}#for md qd

  f <- unlist(f)
  f.mat <-matrix(f, nrow = C, ncol = M*Q)
  # print(knitr::kable(as.data.frame(f.mat), format = "pandoc"))
  d <-as.data.frame(f.mat)
  colnames(d) <-NULL
  rownames <- vector()
  for (cd in 1:C) {
    rownames[cd] <- paste("f[",cd,"]",sep = "")
  }
  rownames(d) <- rownames

  if( summary == TRUE){
    message("\n* Resulting False Alarms \n")
    print(d)
  }

  return(f)

}# function









#'@title Transform from an array to a vector
#'@description In the stan files, the number of hits, false alarms and hit rates in binomial assumption for MRMC case are written in the three indexed array format. Three index are constituted from confidence levels, modality ID, reader ID.
#'However the data substituting the function BayesianFROC::fit_Bayesian_FROC() are written in the vector.
#'So, to connect these different format, the author make this function.

#'
#' @param Three.dim.array Three dimensional array, such as the number of hits for each confidence level, modality and reader. Or false alarms. Since the author construct the substituting data list as one dimensional (one index) array, it needs to reconstruct to the three indexed array  from one dimensional array whose subscript is [confidence level, modality, reader ] or vice versa.
#'
#' @return One dimensional array transformed from user input three dimensional array.

#' @export
#'
#' @examples
#' \donttest{

#'   h.array.etc <- hits_from_thresholds()
#'
#'   h.vector     <- from_array_to_vector(h.array.etc$h)
#'
#'
#'
#'   }
from_array_to_vector <- function(Three.dim.array){

  one.dim.array <-vector()

  C <- dim(Three.dim.array)[1]
  M <- dim(Three.dim.array)[2]
  Q <- dim(Three.dim.array)[3]

  s<-0
  for (cd in 1:C) {
    for (md in 1:M) {
      for (qd in 1:Q) {
        s<-s+1
        one.dim.array[s] <- Three.dim.array[cd,md,qd]

      }
    }
  }

  return(one.dim.array)

}#function



#
# one.dim.array <- vector()
# s<-0
# for (cd in 1:2) {
#   for (md in 1:3) {
#     for (qd in 1:4) {
#       s<-s+1
#       one.dim.array[s] <- paste("h[" , cd,  ",",  md, ",", qd ,"]", sep="")
#     }
#   }
# }












#'@title MRMC: One Dataset Creator (No Replication)
#'@description From threshold, mean and S.D., data of Hits and  False Alarm are created.
#'@param z.truth  Vector of dimension = C represents the thresholds of bi-normal assumption.
#'@param seed  The seed for creating hits which are generated by the binomial distributions with the specified seed.
#'@param NI  The number of images,
#'@param NL  The number of lesions,
#'@inheritParams fit_Bayesian_FROC
#'@inheritParams DrawCurves
#'@inheritParams validation.dataset_srsc
#'@inheritParams hits_creator_from_rate
#'@inheritParams false_and_its_rate_creator
#'@inheritParams hits_from_thresholds
####1#### ####2#### ####3#### ####4#### ####5#### ####6#### ####7#### ####8#### ####9####

#'@examples
#' \donttest{


#'      dataList  <- create_dataList_MRMC()
#'
#'
#'      fit_Bayesian_FROC(dataList,summary = FALSE)
#'
#'
#' #  In the example, we use a default values for true parameters for
#' #  the distributions. The reason why the default values exists is difficulty
#' #  for the user who is not familiar with FROC data nor FROC model parameter regions.
#' #   So, in the Bayesian model is merely model for FROC data.
#' #   If user input the abnormal data, then the model does not fit nor converge
#' #   in the Hamiltonian Monte Carlo simulations.

#'}
#' @export

create_dataList_MRMC <-function(
  z.truth=BayesianFROC::z_truth, #c(0.1,0.2,0.3,0.4,0.5),
  mu.truth = BayesianFROC::mu_truth, #array(1:6/10,c(M,Q)),
  v.truth  = BayesianFROC::v_truth, #array(1:6/10,c(M,Q)),
  NI =57,
  NL=142,
  ModifiedPoisson =FALSE,
  seed =123,
  summary = FALSE
){
  M <- dim(mu.truth)[1]
  Q <- dim(mu.truth)[2]


  f <- false_and_its_rate_creator_MRMC(
    z.truth=z.truth,
    NI =NI,
    NL=NL,
    ModifiedPoisson =ModifiedPoisson,
    seed =seed, # important for replication
    M=M,
    Q=Q,
    summary = FALSE
  )

  f.l.dl <- false_and_its_rate_creator(
    z.truth=z.truth,
    NI =NI,
    NL=NL,
    ModifiedPoisson =ModifiedPoisson,
    seed =seed # important for replication
  )

  h.etc <- hits_from_thresholds(
    z.truth=z.truth,
    mu.truth = mu.truth,
    v.truth  = v.truth,
    NL=NL,
    seed =seed # important for replication
  )


  C <-  length(z.truth)


  m  <-  rep(1:M, each=C*Q)
  q  <-  rep(seq(1,Q,1), M, each=C)
  c  <-  rep(rep(C:1), M*Q)
  h  <- from_array_to_vector(h.etc$h)


  if (summary ==TRUE) {

    print(knitr::kable(data.frame(
      m=m,
      q=q,
      c=c,
      f=f,
      h=h
    ), format = "pandoc")
    )
  }#if summary


  truth=list(
    l.truth = f.l.dl$l.truth,
    dl.truth  = f.l.dl$dl.truth,
    p.truth=h.etc$p.truth,
    z.truth=h.etc$z.truth,
    mu.truth = h.etc$mu.truth,
    v.truth  = h.etc$v.truth
  )



  dataList = list(
    m=m,
    q=q,
    c=c,
    f=f,
    h=h,
    NL=NL,
    NI=NI,
    C=C,
    M=M,
    Q=Q
  )

  return(
    dataList
  )#return


}#function



#'@title MRMC: Replicate Datasets
#'@description From threshold, mean and S.D., it replicate datasets.
#'@param initial.seed The variable \code{initial.seed} is used to replicate datasets. That is, if you take initial.seed = 1234, then the seed 1234, 1235, 1236, 1237, 1238, ….. are for the first replication, the second replication, the third replication, …. If the n-th model does not converge for some n, then such model has no mean and thus the non-convergent models are omitted to calculate the errors.
#'@param No.of.replication For fixed number of lesions, images, the dataset of hits and false alarms are replicated, and the number of replicated datasets are specified by this variable.
#'@inheritParams fit_Bayesian_FROC
#'@inheritParams DrawCurves
#'@inheritParams validation.dataset_srsc
#'@inheritParams hits_creator_from_rate
#'@inheritParams false_and_its_rate_creator
#'@inheritParams hits_from_thresholds
#' @export

replicate_MRMC_dataList <- function(
  initial.seed=123,
  mu.truth = BayesianFROC::mu_truth,
  v.truth =  BayesianFROC::v_truth,
  z.truth  =  BayesianFROC::z_truth,
  NI = 200,
  NL = 142,
  ModifiedPoisson = FALSE,
  No.of.replication = 2,
  summary=FALSE
){
  M <- dim(mu.truth)[1]
  Q <- dim(mu.truth)[2]

  a <-initial.seed
  b <- initial.seed + No.of.replication-1
  list.of.dataList <- list()
  s<-0
  for(seed in a:b){
    s<-s+1
    list.of.dataList[[s]] <-  create_dataList_MRMC(
      z.truth = z.truth,
      mu.truth = mu.truth,
      v.truth  = v.truth,
      NI = NI,
      NL = NL,
      ModifiedPoisson = ModifiedPoisson,
      seed = seed,
      summary=summary
    )

  }# for seed


  return(list.of.dataList)




}#function















#'@title Replicate Models
#'@description Replicate Models For Replicated Data From True Distributions.
#'@param initial.seed The variable \code{initial.seed} is used to replicate datasets. That is, if you take initial.seed = 1234, then the seed 1234, 1235, 1236, 1237, 1238, ….. are for the first replication, the second replication, the third replication, …. If the n-th model does not converge for some n, then such model has no mean and thus the non-convergent models are omitted to calculate the errors.
#'@param No.of.replication For fixed number of lesions, images, the dataset of hits and false alarms are replicated, and the number of replicated datasets are specified by this variable.
#'@inheritParams fit_Bayesian_FROC
#'@inheritParams DrawCurves
#'@inheritParams validation.dataset_srsc
#'@inheritParams hits_creator_from_rate
#'@inheritParams false_and_its_rate_creator
#'@inheritParams hits_from_thresholds
#'@inheritParams replicate_MRMC_dataList
#' @export
#'
#'@examples
#'
#' \donttest{

#'
#'   list.of.fit  <- replicate_model_MRMC()
#'
#'
#'
#'}
replicate_model_MRMC <- function(
  initial.seed=123,
  mu.truth = BayesianFROC::mu_truth,
  v.truth =  BayesianFROC::v_truth,
  z.truth  =  BayesianFROC::z_truth,
  NI = 200,
  NL = 142,
  ModifiedPoisson = FALSE,
  No.of.replication = 2,
  summary=FALSE,
  ite = 1111

){



  list.of.dataList <- replicate_MRMC_dataList(
    initial.seed=initial.seed,
    mu.truth = mu.truth,
    v.truth =  v.truth ,
    z.truth  =  z.truth,
    NI = NI,
    NL = NL,
    ModifiedPoisson = ModifiedPoisson,
    No.of.replication = No.of.replication,
    summary=summary

  )

  list.of.fit <- list()
  s<-0
  message("In the following, the print message from rstan::stan() are restricted:\n")
  for (repl in 1:No.of.replication) {

    message("\n* The ", repl, "-th fitting.\n", sep ="")

    invisible(utils::capture.output(# For hiding the printed message
      fit <- fit_Bayesian_FROC(
        dataList = list.of.dataList[[repl]],
        ite  = ite ,
        summary = FALSE)
    ))


    if( ConfirmConvergence(fit)==TRUE){
      s<-s+1
      list.of.fit[[s]] <-fit # list.of.fit contains only converget models
    }# convergence

  }# for repl

  message("\n -------- Comments --------------\n")
  message("\n* The return value (list of fitting models) contains only converget models.\n\n  ")
  message("\n* Number of Replicated Models : ", No.of.replication, "\n")
  message("\n* Number of Convergent Models : ", s, " \n")

  message("\n* convergence rate :", crayon::cyan(100*s/No.of.replication), "% \n\n")

  size_of_return_value( summary=TRUE,object = list.of.fit)

  invisible(list.of.fit)

}








view_CFP_CTP_for_replicated_models <- function(list.of.fit){

  iii <- length(list.of.fit)
  for (sss in 1:iii) {
    fit <- list.of.fit[[sss]]
    Q <- fit@dataList$Q
    M <- fit@dataList$M

    DrawCurves(fit,
               DrawCFPCTP = T,
               DrawFROCcurve = F,
               DrawAFROCcurve = F,
               modalityID = 1:M,
               readerID = 1:Q,
               new.imaging.device = FALSE
    )
  }

}










#' @title Extract Estimates From Replicated MRMC Model
#'
#'@inheritParams replicate_model_MRMC
#'
#' @return list of estimates
#' @export
#'
#' @examples
#' \donttest{

#'
#'  list.of.estimates <- extract_parameters_from_replicated_models()
#'
#'
#'  }
extract_parameters_from_replicated_models <- function(
  initial.seed=123,
  mu.truth = BayesianFROC::mu_truth,
  v.truth =  BayesianFROC::v_truth,
  z.truth  =  BayesianFROC::z_truth,
  NI = 200,
  NL = 142,
  ModifiedPoisson = FALSE,
  No.of.replication = 2,
  summary=FALSE,
  ite = 1111){


  list.of.fit <- replicate_model_MRMC(
    initial.seed=initial.seed,
    mu.truth = mu.truth,
    v.truth =  v.truth,
    z.truth  =  z.truth,
    NI = NI,
    NL = NL,
    ModifiedPoisson = ModifiedPoisson,
    No.of.replication = No.of.replication,
    summary=summary,
    ite = ite)

  list.of.estimates <- list()

  for (sss in 1:length(list.of.fit)){
    list.of.estimates[[sss]] <- extract_estimates_MRMC(list.of.fit[[sss]] )
  }


  invisible(list.of.estimates)

}#function



#' @title Comparison of Model and Truth
#'@description Comparison of estimates from true distributions and the true model parameters
#'@inheritParams replicate_model_MRMC
#'
#' @return list of estimates
#' @export
#'
comparison_replicated_models_and_truth <- function(
  initial.seed=123,
  mu.truth = BayesianFROC::mu_truth,
  v.truth =  BayesianFROC::v_truth,
  z.truth  =  BayesianFROC::z_truth,
  NI = 200,
  NL = 142,
  ModifiedPoisson = FALSE,
  No.of.replication = 2,
  summary=FALSE,
  ite = 1111){


  list.of.estimates <- extract_parameters_from_replicated_models(
    initial.seed=initial.seed,
    mu.truth = mu.truth,
    v.truth =  v.truth,
    z.truth  =  z.truth,
    NI = NI,
    NL = NL,
    ModifiedPoisson = ModifiedPoisson,
    No.of.replication = No.of.replication,
    summary=summary,
    ite = ite
  )


  w.EAP <- list()
  dz.EAP <- list()
  z.EAP <- list()
  mu.EAP <- list()
  v.EAP <- list()
  ppp.EAP  <- list()
  l.EAP <- list()
  dl.EAP <- list()
  A.EAP <- list()
  AA.EAP <- list()
  aa.EAP <- list()
  bb.EAP <- list()


  for (sss in 1:length(list.of.estimates)) {

    w.EAP[[sss]] <- list.of.estimates[[sss]]$w.EAP
    dz.EAP[[sss]] <- list.of.estimates[[sss]]$dz.EAP
    z.EAP[[sss]] <- list.of.estimates[[sss]]$z.EAP
    mu.EAP[[sss]] <- list.of.estimates[[sss]]$mu.EAP
    v.EAP[[sss]] <- list.of.estimates[[sss]]$v.EAP
    ppp.EAP[[sss]] <- list.of.estimates[[sss]]$ppp.EAP
    l.EAP[[sss]] <- list.of.estimates[[sss]]$l.EAP
    dl.EAP[[sss]] <- list.of.estimates[[sss]]$dl.EAP
    A.EAP[[sss]] <- list.of.estimates[[sss]]$A.EAP
    AA.EAP[[sss]] <- list.of.estimates[[sss]]$AA.EAP
    aa.EAP[[sss]] <- list.of.estimates[[sss]]$aa.EAP
    bb.EAP[[sss]] <- list.of.estimates[[sss]]$bb.EAP

  }

  # Tp summarize over the list, i.e., over sss =1,2,3.... we use the function Reduce form package base.


  w.EAP.sum.over.list   <-  Reduce("+",  w.EAP )
  dz.EAP.sum.over.list  <-  Reduce("+",  dz.EAP )
  z.EAP.sum.over.list   <-  Reduce("+",  z.EAP )
  mu.EAP.sum.over.list  <-  Reduce("+",  mu.EAP )
  v.EAP.sum.over.list   <-  Reduce("+",  v.EAP )
  ppp.EAP.sum.over.list <-  Reduce("+",  ppp.EAP )
  l.EAP.sum.over.list   <-  Reduce("+",  l.EAP )
  dl.EAP.sum.over.list  <-  Reduce("+",  dl.EAP )
  A.EAP.sum.over.list   <-  Reduce("+",  A.EAP )
  AA.EAP.sum.over.list  <-  Reduce("+",  AA.EAP )
  aa.EAP.sum.over.list  <-  Reduce("+",  aa.EAP )
  bb.EAP.sum.over.list  <-  Reduce("+",  bb.EAP )



  w.EAP.mean.over.replicated.models   <-    w.EAP.sum.over.list/length(list.of.estimates)
  dz.EAP.mean.over.replicated.models  <-   dz.EAP.sum.over.list/length(list.of.estimates)
  z.EAP.mean.over.replicated.models   <-    z.EAP.sum.over.list/length(list.of.estimates)
  mu.EAP.mean.over.replicated.models  <-  mu.EAP.sum.over.list/length(list.of.estimates)
  v.EAP.mean.over.replicated.models   <-     v.EAP.sum.over.list/length(list.of.estimates)
  ppp.EAP.mean.over.replicated.models <-   ppp.EAP.sum.over.list/length(list.of.estimates)
  l.EAP.mean.over.replicated.models   <-    l.EAP.sum.over.list/length(list.of.estimates)
  dl.EAP.mean.over.replicated.models  <-    dl.EAP.sum.over.list/length(list.of.estimates)
  A.EAP.mean.over.replicated.models   <-    A.EAP.sum.over.list/length(list.of.estimates)
  AA.EAP.mean.over.replicated.models  <-    AA.EAP.sum.over.list/length(list.of.estimates)
  aa.EAP.mean.over.replicated.models  <-   aa.EAP.sum.over.list/length(list.of.estimates)
  bb.EAP.mean.over.replicated.models  <-     bb.EAP.sum.over.list/length(list.of.estimates)





  mu.error <-  mu.EAP.mean.over.replicated.models  -  mu.truth
  v.error  <- v.EAP.mean.over.replicated.models    -  v.truth
  z.error  <- z.EAP.mean.over.replicated.models    -  z.truth








  # xx <-apply(
  #   array(unlist(x),append( dim(x[[1]]), length(x) )  ),
  #   1:length(dim(x[[1]])),
  #   var
  # )

  foo <- function(x){apply(
    array(unlist(x),append( dim(x[[1]]), length(x) )  ),
    1:length(dim(x[[1]])),
    stats::var
  )}


  w.EAP.variance.over.replicated.models   <- foo_of_a_List_of_Arrays(  w.EAP  ,stats::var)
  dz.EAP.variance.over.replicated.models  <- foo_of_a_List_of_Arrays( dz.EAP   ,stats::var)
  z.EAP.variance.over.replicated.models   <- foo_of_a_List_of_Arrays(  z.EAP  ,stats::var)
  mu.EAP.variance.over.replicated.models  <- foo_of_a_List_of_Arrays( mu.EAP   ,stats::var)
  v.EAP.variance.over.replicated.models   <- foo_of_a_List_of_Arrays( v.EAP   ,stats::var)
  ppp.EAP.variance.over.replicated.models <- foo_of_a_List_of_Arrays(  ppp.EAP  ,stats::var)
  l.EAP.variance.over.replicated.models   <- foo_of_a_List_of_Arrays(  l.EAP  ,stats::var)
  dl.EAP.variance.over.replicated.models  <- foo_of_a_List_of_Arrays( dl.EAP   ,stats::var)
  A.EAP.variance.over.replicated.models   <- foo_of_a_List_of_Arrays(  A.EAP  ,stats::var)
  AA.EAP.variance.over.replicated.models  <- foo_of_a_List_of_Arrays(  AA.EAP  ,stats::var)
  aa.EAP.variance.over.replicated.models  <- foo_of_a_List_of_Arrays(  aa.EAP  ,stats::var)
  bb.EAP.variance.over.replicated.models  <- foo_of_a_List_of_Arrays(  bb.EAP  ,stats::var)























  return(
    list(
      mu.error = mu.error,
      v.error  = v.error,
      z.error  = z.error,



      w.EAP.variance.over.replicated.models   =   w.EAP.variance.over.replicated.models,
      dz.EAP.variance.over.replicated.models  =    dz.EAP.variance.over.replicated.models,
      z.EAP.variance.over.replicated.models   =   z.EAP.variance.over.replicated.models   ,
      mu.EAP.variance.over.replicated.models  =   mu.EAP.variance.over.replicated.models ,
      v.EAP.variance.over.replicated.models   =  v.EAP.variance.over.replicated.models     ,
      ppp.EAP.variance.over.replicated.models =  ppp.EAP.variance.over.replicated.models ,
      l.EAP.variance.over.replicated.models   = l.EAP.variance.over.replicated.models     ,
      dl.EAP.variance.over.replicated.models  =  dl.EAP.variance.over.replicated.models   ,
      A.EAP.variance.over.replicated.models   =   A.EAP.variance.over.replicated.models   ,
      AA.EAP.variance.over.replicated.models  =   AA.EAP.variance.over.replicated.models  ,
      aa.EAP.variance.over.replicated.models  =    aa.EAP.variance.over.replicated.models  ,
      bb.EAP.variance.over.replicated.models  =   bb.EAP.variance.over.replicated.models



    )
  )














  # w.EAP
  # dz.EAP
  # z.EAP
  # mu.EAP
  # v.EAP
  # ppp.EAP
  # l.EAP
  # dl.EAP
  # A.EAP
  # AA.EAP
  # aa.EAP
  # bb.EAP


}#function

