#' @title plot the conditional predictive ordinate
#' @description
#' Plot the conditional predictive ordinate (CPO) for each individual of a fitted model generated by \code{BayesSUR} which is a "BayesSUR" object. CPO is a handy posterior predictive check because it may be used to identify outliers, influential observations, and for hypothesis testing across different non-nested models (Gelfand 1996).
#' @importFrom graphics axis box text par abline
#' @name plotCPO
#' @param object an object of class "BayesSUR"
#' @param sum.responses compute CPOs aggreated in all response variables
#' @param outlier.thresh threshold for the CPOs. The default is 0.01.
#' @param outlier.mark mark the outliers with the response names. The default is \code{FALSE}
#' @param scale.CPO scaled CPOs which is divided by their maximum. The default is \code{TRUE}
#' @param x.loc a vector of features distance
#' @param axis.label a vector of predictor names which are shown in CPO plot. The default is \code{NULL} only showing the indices. The value \code{"auto"} show the predictor names from the orginal data.
#' @param mark.pos the location of the marked text relative to the point
#' @param las graphical parameter of plot.default
#' @param cex.axis graphical parameter of plot.default
#' @param mark.color the color of the marked text. The default color is red.
#' @param mark.cex the fontsize of the marked text. The default fontsize is 0.8.
#' @param xlab a title for the x axis 
#' @param ylab a title for the y axis 
#' @references Statisticat, LLC (2013). \emph{Bayesian Inference.} Farmington, CT: Statisticat, LLC.
#'
#' @details The default threshold for the CPOs to detect the outliers is 0.01 by Congdon (2005). It can be tuned by the argument \code{outlier.thresh}.
#' 
#' @references Gelfand A. (1996). \emph{Model Determination Using Sampling Based Methods}. In Gilks W., Richardson S., Spiegelhalter D. (eds.), Markov Chain Monte Carlo in Practice, pp. 145–161. Chapman & Hall, Boca Raton, FL.
#' @references Congdon P. (2005). \emph{Bayesian Models for Categorical Data}. John Wiley & Sons, West Sussex, England.
#'
#' @examples
#' data("example_eQTL", package = "BayesSUR")
#' hyperpar <- list( a_w = 2 , b_w = 5 )
#' 
#' fit <- BayesSUR(Y = example_eQTL[["blockList"]][[1]], 
#'                 X = example_eQTL[["blockList"]][[2]],
#'                 data = example_eQTL[["data"]], outFilePath = tempdir(),
#'                 nIter = 100, burnin = 50, nChains = 2, gammaPrior = "hotspot",
#'                 hyperpar = hyperpar, tmpFolder = "tmp/" )
#' 
#' ## check output
#' # plot the conditional predictive ordinate (CPO)
#' plotCPO(fit)
#' 
#' @export
plotCPO <- function(object, sum.responses=FALSE, outlier.mark=TRUE, outlier.thresh=0.01, scale.CPO=TRUE, x.loc=FALSE, axis.label=NULL, las=0, cex.axis=1, mark.pos=c(0,-.01), mark.color=2, mark.cex=0.8,
                    xlab="Observations", ylab=NULL){
  
  object$output[-1] <- paste(object$output$outFilePath,object$output[-1],sep="")
  CPO <- as.matrix( read.table(object$output$CPO) )
  
  if(is.null(ylab))
    ylab <- ifelse(scale.CPO,"scaled CPOs","CPOs")
  
  name.predictors <- rownames(as.matrix( read.table(object$output$Y,header=T) ))
    
  if(is.null(axis.label)){
    x.loc <- 1:nrow(CPO)
    names(x.loc) <- 1:nrow(CPO)
  }else{
    if(axis.label[1] == "auto"){
      x.loc <- 1:nrow(CPO)
      names(x.loc) <- name.predictors
    }else{
      if(!x.loc[1]){
        x.loc <- 1:length(axis.label)
      }else{
        if( length(axis.label) != length(x.loc) )
          stop("The given predictor names are not consistent with the data")
      }
      names(x.loc) <- axis.label
    }
  }
  
  if(!sum.responses){
    if(scale.CPO) CPO <- CPO/max(CPO)
    plot.default(as.vector(CPO) ~ rep(1:nrow(CPO), times=ncol(CPO)), xlim=c(1,nrow(CPO)), ylim=c(0,max(CPO)), xaxt = 'n',bty = "n", ylab = ylab, xlab = xlab, main="Conditional predictive ordinate", pch=19)
    axis(1, at=x.loc, labels=names(x.loc), las=las, cex.axis=cex.axis); box()
    
    # mark the names of the specified response variables corresponding to the given responses
    if(outlier.mark){
      if(min(CPO) > outlier.thresh){
        message("NOTE: The minimum CPO is larger than the threshold of the (scaled) CPO!\n")
      }else{
        name.responses <- colnames(as.matrix( read.table(object$output$Y,header=T) ))
        text(rep(1:nrow(CPO), times=ncol(CPO))[which(as.vector(CPO) <= outlier.thresh)]+mark.pos[1], as.vector(CPO[CPO<outlier.thresh])+mark.pos[2], labels=rep(name.responses, each=nrow(CPO))[as.vector(CPO) < outlier.thresh], col=mark.color, cex=mark.cex)
        abline(h=outlier.thresh, lty=2, col=mark.color)
      }
    }
  }else{
    CPO <- as.matrix( read.table(object$output$CPOsumy) )
    #CPO <- rowSums(CPO)
    if(scale.CPO) CPO <- CPO/max(CPO)
    plot.default(CPO ~ c(1:length(CPO)), xaxt = 'n',bty = "n", xlim=c(1,length(CPO)), ylim=c(min(CPO)+mark.pos[2]*2,max(CPO)), ylab = ylab, xlab = xlab, main="Conditional predictive ordinate", pch=19)
    axis(1, at=x.loc, labels=names(x.loc), las=las, cex.axis=cex.axis); box()
    
    # mark the names of the specified response variables corresponding to the given responses
    if(outlier.mark){
      if(min(CPO) > outlier.thresh){
        message("NOTE: The minimum CPO is larger than the threshold of the (scaled) CPO!\n")
      }else{
        opar <- par(no.readonly=TRUE)
        on.exit(par(opar))    
        par(new=T)
        plot.default(CPO[CPO<outlier.thresh] ~ which(CPO<outlier.thresh), xaxt = 'n',bty = "n", xlim=c(1,length(CPO)), ylim=c(min(CPO)+mark.pos[2]*2,max(CPO)), ylab = "", xlab = "", main="", pch=19, col=mark.color)
        text(which(CPO <= outlier.thresh)+mark.pos[1], CPO[CPO<outlier.thresh]+mark.pos[2], labels=name.predictors[CPO < outlier.thresh], col=mark.color, cex=mark.cex)
        abline(h=outlier.thresh, lty=2, col=mark.color)
      }
    }
  }
  
}
