% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayes_mode.R
\name{bayes_mode}
\alias{bayes_mode}
\title{Bayesian mode inference}
\usage{
bayes_mode(
  BayesMix,
  rd = 1,
  tol_x = sd(BayesMix$data)/10,
  tol_conv = 1e-08,
  show_plot = FALSE,
  nb_iter = NULL
)
}
\arguments{
\item{BayesMix}{An object of class \code{BayesMixture}.}

\item{rd}{Rounding parameter.}

\item{tol_x}{Tolerance parameter for distance in-between modes; default is sd(data)/10 where data is an element of argument \code{BayesMix}.
If two modes are closer than \code{tol_x}, only the first estimated mode is kept.}

\item{tol_conv}{Tolerance parameter for convergence of the algorithm; default is 1e-8.
Not needed for mixtures of discrete distributions.}

\item{show_plot}{Show density with estimated mode as vertical bars ?}

\item{nb_iter}{Number of draws on which the mode-finding algorithm is run; default is NULL which means the algorithm is run on all draws.}
}
\value{
A list of class \code{BayesMode} containing
\itemize{
 \item{data}{ - from \code{BayesMix}.}
 \item{dist}{ - from \code{BayesMix}.}
 \item{dist_type}{ - from \code{BayesMix}.}
 \item{pars_names}{ - from \code{BayesMix}.}
 \item{modes}{ - Matrix with a row for each draw and columns showing modes.}
 \item{p1}{ - Posterior probability of unimodality.}
 \item{tb_nb_modes}{ - Matrix showing posterior probabilities for the number of modes.}
 \item{table_location}{ - Matrix showing the posterior probabilities for location points being modes.}
}
}
\description{
This function estimates modes for each mcmc draw and uses these estimates to compute posterior
probabilities for the number of modes and their locations (following the approach of Cross et al. 2023).
The fixed-point algorithm of Carreira-Perpinan (2000) is used for Gaussian mixtures.
The Modal EM algorithm of Li et al. (2007) is used for other continuous mixtures.
A basic algorithm is used for discrete mixtures (see Cross et al. 2023).
}
\details{
Each draw from the MCMC output after burnin, \eqn{\theta^{(d)}, \quad d = 1,...,D}, leads to a posterior predictive probability
density/mass function: 
\deqn{p(y | \theta^{(d)}) =\sum_{k=1}^{K} \pi_k^{(d)} p(y | \theta_k^{(d)}).}
Using this function, the mode in draw \eqn{d} \eqn{y_{m}^{(d)}}, \eqn{m = 1,..., M^{(d)}},
where \eqn{M^{(d)}} is the number of modes, are estimated using the algorithm mentioned
in the description above.

After running this procedure across all retained posterior draws, 
we compute the posterior probability for the number of modes being \eqn{M} as:
\deqn{P(\#\text{modes}=M)=\frac{1}{D}\sum_{d=1}^{D}1(M^{(d)} = M).}
Similarly, posterior probabilities for locations of the modes are given by:
\deqn{P(y=\text{mode})=\frac{1}{D}\sum_{d=1}^{D} \sum_{m=1}^{M^{(d)}} 1(y = y_m^{(d)}),}
for each location \eqn{y} in the range \eqn{[\min(y),\max(y)]}. Obviously,
continuous data are not defined on a discrete support;
it is therefore necessary to choose a rounding decimal to discretize their support (with the \code{rd} argument).
}
\examples{
# Example with galaxy data ================================================
set.seed(123) 

# retrieve galaxy data
y = galaxy

# estimation
bayesmix = bayes_estimation(data = y,
                           K = 5, #not many to run the example rapidly
                           dist = "normal",
                           nb_iter = 500, #not many to run the example rapidly
                           burnin = 100)

# mode estimation
bayesmode = bayes_mode(bayesmix)

# plot 
# plot(bayesmode, max_size = 200)

# summary 
# summary(bayesmode)

# Example with DNA data ================================================
set.seed(123) 

# retrieve galaxy data
y = d4z4

# estimation
bayesmix = bayes_estimation(data = y,
                           K = 5, #not many to run the example rapidly
                           dist = "shifted_poisson",
                           nb_iter = 500, #not many to run the example rapidly
                           burnin = 100)

# mode estimation
bayesmode = bayes_mode(bayesmix)

# plot 
# plot(bayesmode, max_size = 200)

# summary 
# summary(bayesmode)

# Example with a Student t ================================================
mu = c(0.5,6)
sigma = c(1,2)
nu = c(5,5)
p = c(0.8,0.2)
params = c(eta = p, mu = mu, sigma = sigma, nu = nu)
pars_names = c("eta", "mu", "sigma", "nu")
dist_type = "continuous"

data = c(sn::rst(p[1]*1000, mu[1], sigma[1], nu = nu[1]),
         sn::rst(p[2]*1000, mu[2], sigma[2], nu = nu[2]))

fit = c(eta = p, mu = mu, sigma = sigma, nu = nu)
fit = rbind(fit, fit)

pdf_func = function(x, pars) {
  sn::dst(x, pars["mu"], pars["sigma"], pars["xi"], pars["nu"])
}

bayesmix = new_BayesMixture(fit, data, K = 2, burnin = 1,
pars_names = pars_names, pdf_func = pdf_func, dist_type = dist_type)

bayesmode = bayes_mode(bayesmix)

# plot 
# plot(bayesmode, max_size = 200)

# summary 
# summary(bayesmode)

}
\references{
\insertRef{cross_2023}{BayesMultiMode}
\insertRef{carreira-perpinan_mode-finding_2000}{BayesMultiMode}\cr\cr
\insertRef{li_nonparametric_2007}{BayesMultiMode}
}
