#'
#' Generate operating characteristics for drug combination trials
#'
#' Obtain the operating characteristics of the BOIN design or waterfall design for drug combination
#'  trials. The BOIN design is to find a MTD, and the waterfall design is to find the MTD contour
#'  (i.e., multple MTDs in the dose matrix)
#'
#' @usage get.oc.comb(target, p.true, ncohort, cohortsize, n.earlystop=NULL, startdose=c(1, 1),
#'                    p.saf="default", p.tox="default", cutoff.eli=0.95, extrasafe=FALSE,
#'                    offset=0.05, ntrial=1000, MTD.contour=TRUE, Nmax=NULL)
#'
#' @param target target toxicity rate
#' @param p.true a \code{J*K} matrix \code{(J<=K)} containing the true toxicity probabilities of
#'               combinations with \code{J} dose levels of agent A and \code{K} dose levels of agent B
#' @param ncohort the total number of cohorts
#' @param cohortsize the cohort size
#' @param n.earlystop early stopping parameter. If the number of patients treated at the current
#'                    dose reaches \code{n.earlystop}, stop the trial or subtrial and select the MTD based on
#'                    the observed data. When the waterfall design is used to find the MTD contour,
#'                    \code{n.earlystop=12} by default.
#' @param startdose the starting dose combination level for drug combination trial
#' @param p.saf the highest toxicity probability that is deemed subtherapeutic (i.e. below the MTD)
#'              such that dose escalation should be undertaken.
#'              The default value is \code{p.saf=0.6*target}.
#' @param p.tox the lowest toxicity probability that is deemed overly toxic such that deescalation
#'              is required. The default value is \code{p.tox=1.4*target}.
#' @param cutoff.eli the cutoff to eliminate an overly toxic dose for safety. We recommend the
#'                   default value of (\code{cutoff.eli=0.95}) for general use.
#' @param extrasafe set \code{extrasafe=TRUE} to impose a more stringent stopping rule
#' @param offset a small positive number (between 0 and 0.5) to control how strict the stopping
#'               rule is when \code{extrasafe=TRUE}. A larger value leads to a more strict stopping
#'               rule. The default value \code{offset=0.05} generally works well.
#' @param ntrial the total number of trials to be simulated
#' @param MTD.contour set \code{MTD.contour=TRUE} to select the MTD contour (claiming multiple MTDs).
#'                    Otherwise, BOIN design is used to search for a single MTD.
#' @param Nmax this argument is effective only when \code{MTD.contour=TRUE}.
#'             It specifies the maximum number of patients for each subtrial
#'
#' @details The operating characteristics of the BOIN design or waterfall design are generated by
#' simulating trials under the prespecified true toxicity probabilities of the investigational dose
#' combinations. The BOIN and waterfall designs have two built-in stopping rules:
#' (1) stop the trial/subtrial if the lowest dose is eliminated due to toxicity, and no dose should
#' be selected as the MTD; and (2) stop the trial/subtrial and select the MTD if the number of
#' patients treated at the current dose reaches \code{n.earlystop}. The first stopping rule is a safety
#' rule to protect patients from the case in which all doses are overly toxic. The rationale for
#' the second stopping rule is that when there is a large number (i.e., \code{n.earlystop}) of
#' patients assigned to a dose, it means that the dose-finding algorithm has approximately converged.
#' Thus, we can stop the trial/subtrial early and select the MTD to save sample size and reduce the
#' trial duration.
#'
#'
#'  For some applications, investigators may prefer a more strict safety stopping rule than rule
#'  (1) for extra safety when the lowest dose is overly toxic.
#'  This can be achieved by setting \code{extrasafe=TRUE},
#'  which imposes the following more strict safety stopping rule:
#'  stop the trial if (i) the number of patients treated at the lowest dose \code{>=3},
#' and (ii) Pr(toxicity rate of the lowest dose > \code{target} | data) > \code{cutoff.eli-offset}.
#' As a tradeoff, the strong stopping rule will decrease the MTD selection percentage
#'  when the lowest dose actually is the MTD.
#'
#' @return \code{get.oc.comb()} returns the operating characteristics of the BOIN or
#' waterfall design as a list, including (1) selection percentage at each dose level,
#' (2) the number of patients treated at each dose level,
#' (3) the number of toxicities observed at each dose level,
#' (4) the total correct selection of the MTD,
#' (5) the total percentage of patients treated at the MTD.
#'
#' @export
#'
#' @note We should avoid setting the values of \code{p.saf} and \code{p.tox} very close to the target.
#' This is because the small sample sizes of typical phase I trials prevent us from differentiating
#'  the target toxicity rate from the rates close to it. In addition, in most clinical applications,
#'  the target toxicity rate is often a rough guess, and finding a dose level with a toxicity rate
#'  reasonably close to the target rate will still be of interest to the investigator. The default
#'  values provided by \code{get.oc.comb()} are generally reasonable for most clinical applications.
#'
#' @author Suyu Liu and Ying Yuan
#'
#' @references Liu S. and Yuan, Y. (2015). Bayesian Optimal Interval Designs for Phase I Clinical
#'             Trials, Journal of the Royal Statistical Society: Series C, 64, 507-523.
#'
#'            Lin R. and Yin, G. (2016). Bayesian Optimal Interval Designs for Dose Finding in
#'            Drug-combination Trials, Statistical Methods in Medical Research, to appear.
#'
#'            Zhang L. and Yuan, Y. (2016). A Simple Bayesian Design to Identify the Maximum
#'            Tolerated Dose Contour for Drug Combination Trials, under review.
#'
#' @seealso  Tutorial: \url{http://odin.mdacc.tmc.edu/~yyuan/Software/BOIN/BOIN2.1_tutorial.pdf}
#'
#'           Paper: \url{http://odin.mdacc.tmc.edu/~yyuan/Software/BOIN/paper.pdf}
#'
#' @examples
#' #p.true = matrix(c(0.01,0.03,0.15,0.20,0.30,
#' #                  0.03,0.05,0.10,0.30,0.60,
#' #                  0.08,0.10,0.30,0.60,0.75), byrow=TRUE, ncol=5)
#' ## find the MTD contour using waterfall design
#' #get.oc.comb(target=0.3, p.true, ncohort=20, cohortsize=3, n.earlystop=12, startdose=c(1,1),
#' #  		       ntrial=1000, MTD.contour=TRUE)
#'
#' ## find a single MTD using BOIN design
#' #get.oc.comb(target=0.3, p.true, ncohort=20, cohortsize=3, n.earlystop=12, startdose=c(1,1),
#' #			       ntrial=1000, MTD.contour=FALSE)
#'
get.oc.comb <- function(target, p.true, ncohort, cohortsize, n.earlystop=NULL,
                        startdose=c(1,1), p.saf="default", p.tox="default",
                        cutoff.eli=0.95, extrasafe=FALSE, offset=0.05, ntrial=1000, MTD.contour=TRUE, Nmax=NULL){


get.oc.comb.boin <- function(target, p.true, ncohort, cohortsize, n.earlystop=100, startdose=c(1,1), p.saf="default",
                             p.tox="default", cutoff.eli=0.95, extrasafe=FALSE, offset=0.05, ntrial=1000){
    JJ=nrow(p.true); KK=ncol(p.true)
    if(JJ>KK) stop("p.true should be arranged in a way (i.e., rotated) such that
                    the number of rows is less than or equal to the number of columns.")

    if(JJ>KK) p.true = t(p.true)
    ## if the user does not provide p.saf and p.tox, set them to the default values
    if(p.saf=="default") p.saf=0.6*target;
    if(p.tox=="default") p.tox=1.4*target;

    ## simple error checking
    if(target<0.05) {
      cat("Error: the target is too low! \n"); return(1);
    }
    if(target>0.6)  {cat("Error: the target is too high! \n"); return(1);}
    if((target-p.saf)<(0.1*target)) {cat("Error: the probability deemed safe cannot be higher than or too close to the target! \n"); return(1);}
    if((p.tox-target)<(0.1*target)) {cat("Error: the probability deemed toxic cannot be lower than or too close to the target! \n"); return(1);}
    if(offset>=0.5) {cat("Error: the offset is too large! \n"); return();}
    if(n.earlystop<=6) {cat("Warning: the value of n.earlystop is too low to ensure good operating characteristics. Recommend n.earlystop = 9 to 18 \n"); return();}


    set.seed(6);
    ndose=length(p.true);
    npts = ncohort*cohortsize;
    Y<-array(matrix(rep(0,length(p.true)*ntrial),dim(p.true)[1]),dim=c(dim(p.true),ntrial)) # toxicity outcome
    N <-array(matrix(rep(0,length(p.true)*ntrial),dim(p.true)[1]),dim=c(dim(p.true),ntrial)) # number of patients
    dselect =matrix(rep(0, 2*ntrial), ncol=2); # the selected dose level

    ## obtain dose escalation and de-escalation boundaries
    temp=get.boundary(target, ncohort, cohortsize, n.earlystop, p.saf, p.tox, cutoff.eli, extrasafe, print=FALSE);
    b.e=temp[2,];   # escalation boundary
    b.d=temp[3,];   # deescalation boundary
    b.elim=temp[4,];  # elimination boundary

    lambda1  = log((1-p.saf)/(1-target))/log(target*(1-p.saf)/(p.saf*(1-target)));
    lambda2  = log((1-target)/(1-p.tox))/log(p.tox*(1-target)/(target*(1-p.tox)));

    ################## simulate trials ###################
    for(trial in 1:ntrial)
    {
        y<-matrix(rep(0, ndose),dim(p.true)[1],dim(p.true)[2]);    ## the number of DLT at each dose level
        n<-matrix(rep(0, ndose),dim(p.true)[1],dim(p.true)[2]);    ## the number of patients treated at each dose level
        earlystop=0;         ## indiate whether the trial terminates early
        d=startdose;         ## starting dose level
        elimi = matrix(rep(0, ndose),dim(p.true)[1],dim(p.true)[2]);  ## indicate whether doses are eliminated

        for(pp in 1:ncohort)
        {
            ### generate toxicity outcome
            y[d[1],d[2]] = y[d[1],d[2]] + sum(runif(cohortsize)<p.true[d[1],d[2]]);
            n[d[1],d[2]] = n[d[1],d[2]] + cohortsize;
            if(n[d[1],d[2]]>=n.earlystop) break;
            nc = n[d[1],d[2]];

            ## determine if the current dose should be eliminated
            if(!is.na(b.elim[nc]))
            {
                if(y[d[1],d[2]]>=b.elim[nc])
                {
                    for (i in min(d[1],dim(p.true)[1]):dim(p.true)[1])
                    { for (j in min(d[2],dim(p.true)[2]):dim(p.true)[2]) { elimi[i,j]=1;}}
                    if(d[1]==1&&d[2]==1) {d=c(99, 99); earlystop=1; break;}
                }

                ## implement the extra safe rule by decreasing the elimination cutoff for the lowest dose
                if(extrasafe)
                {
                    if(d[1]==1&&d[2]==1 && y[1,1]>=3)
                    {
                        if(1-pbeta(target, y[1,1]+1, n[1,1]-y[1,1]+1)>cutoff.eli-offset) {d=c(99, 99); earlystop=1; break;}
                    }
                }
            }

            ## dose escalation/de-escalation
            if(y[d[1],d[2]]<=b.e[nc])
            {
                elevel=matrix(c(1,0,0,1),2);
                pr_H0=rep(0,length(elevel)/2)
                nn=pr_H0;
                for ( i in seq(1,length(elevel)/2,by=1))
                { if (d[1]+elevel[1,i]<=dim(p.true)[1] && d[2]+elevel[2,i]<=dim(p.true)[2])
                    {
                        if (elimi[d[1]+elevel[1,i],d[2]+elevel[2,i]]==0)
                        {
                            yn=y[d[1]+elevel[1,i],d[2]+elevel[2,i]];
                            nn[i]=n[d[1]+elevel[1,i],d[2]+elevel[2,i]];
                            pr_H0[i]<-pbeta(lambda2,yn+0.5,nn[i]-yn+0.5)-pbeta(lambda1,yn+0.5,nn[i]-yn+0.5)
                        }
                    }
                }
                pr_H0=pr_H0+nn*0.0005;  ## break ties

                if (max(pr_H0)==0) {d=d} else
                {
                    k=which(pr_H0==max(pr_H0))[as.integer(runif(1)*length(which(pr_H0==max(pr_H0)))+1)];
                    d=d+c(elevel[1,k],elevel[2,k]);
                }

            }
            else if(y[d[1],d[2]]>=b.d[nc])
            {
                delevel=matrix(c(-1,0,0,-1),2)
                pr_H0=rep(0,length(delevel)/2)
                nn=pr_H0;
                for ( i in seq(1,length(delevel)/2,by=1))
                {
                    if (d[1]+delevel[1,i]>0 && d[2]+delevel[2,i]>0)
                    {
                        yn=y[d[1]+delevel[1,i],d[2]+delevel[2,i]];
                        nn[i]=n[d[1]+delevel[1,i],d[2]+delevel[2,i]];
                        pr_H0[i]=pbeta(lambda2,yn+0.5,nn[i]-yn+0.5)-pbeta(lambda1,yn+0.5,nn[i]-yn+0.5)
                    }
                }
                pr_H0=pr_H0+nn*0.0005; ## break ties

                if (max(pr_H0)==0) {d=d}  else
                {
                    k=which(pr_H0==max(pr_H0))[as.integer(runif(1)*length(which(pr_H0==max(pr_H0)))+1)];
                    d=d+c(delevel[1,k],delevel[2,k]);
                }
            } else { d=d; }
        }

        Y[,,trial]=y;
        N[,,trial]=n;
        if(earlystop==1) { dselect[trial,]=c(99, 99); }
        else
        {
            selcomb=select.mtd.comb(target, n, y, cutoff.eli, extrasafe, offset, print=FALSE, MTD.contour=FALSE);
			dselect[trial,1]=selcomb[1];
			dselect[trial,2]=selcomb[2];
        }
    }

    # output results
    selpercent=matrix(rep(0, ndose),dim(p.true)[1],dim(p.true)[2]);
    nptsdose  =apply(N,c(1,2),mean, digits=2, format="f");
    ntoxdose  =apply(Y,c(1,2),mean, digits=2, format="f");

    for(i in 1:dim(p.true)[1])
    for (j in 1:dim(p.true)[2]){
    { selpercent[i,j]=sum(dselect[,1]==i&dselect[,2]==j)/ntrial*100; }}


    if(JJ<=KK){
        cat("True toxicity rate of dose combinations:\n");
        for (i in 1:dim(p.true)[1]){
            cat(formatC(p.true[i,], digits=2, format="f", width=5), sep="  ", "\n");
        }
        cat("\n");
        cat("selection percentage at each dose combination (%):\n");
        for (i in 1:dim(p.true)[1]){
            cat(formatC(selpercent[i,], digits=2, format="f", width=5), sep="  ", "\n");
        }
        cat("\n");
        cat("number of patients treated at each dose combination:\n");
        for (i in 1:dim(p.true)[1]){
            cat(formatC(apply(N,c(1,2),mean)[i,], digits=2, format="f", width=5), sep ="  ", "\n");}
        cat("\n");
        cat("number of toxicity observed at each dose combination:\n");
        for (i in 1:dim(p.true)[1]){
            cat(formatC(apply(Y,c(1,2),mean)[i,], digits=2, format="f", width=5), sep ="  ", "\n");}
        cat("\n");
        cat("average number of toxicities:", formatC(sum(Y)/ntrial, digits=1, format="f"), "\n");
        cat("average number of patients:", formatC(sum(N)/ntrial, digits=1, format="f"), "\n");
        cat("selection percentage of MTD:", formatC(sum(selpercent[which(p.true==target, arr.ind = TRUE)]), digits=1, format="f"), "\n");
        cat("percentage of patients treated at MTD:", formatC(sum(nptsdose[which(p.true==target, arr.ind = TRUE)])/npts*100, digits=1, format="f"), "\n");
        cat("percentage of early stopping due to toxicity:", formatC(100-sum(selpercent), digits=2, format="f"), "\n");
    }
    else{
        cat("True toxicity rate of dose combinations:\n");
        for (i in 1:dim(p.true)[2]){
            cat(formatC(p.true[,i], digits=2, format="f", width=5), sep="  ", "\n");
        }
        cat("\n");
        cat("selection percentage at each dose combination (%):\n");
        for (i in 1:dim(p.true)[2]){
            cat(formatC(selpercent[,i], digits=2, format="f", width=5), sep="  ", "\n");
        }
        cat("\n");
        cat("number of patients treated at each dose combination:\n");
        for (i in 1:dim(p.true)[2]){
            cat(formatC(apply(N,c(1,2),mean)[,i], digits=2, format="f", width=5), sep ="  ", "\n");}
        cat("\n");
        cat("number of toxicity observed at each dose combination:\n");
        for (i in 1:dim(p.true)[2]){
            cat(formatC(apply(Y,c(1,2),mean)[,i], digits=2, format="f", width=5), sep ="  ", "\n");}
        cat("\n");
        cat("average number of toxicities:", formatC(t(sum(Y))/ntrial, digits=1, format="f"), "\n");
        cat("average number of patients:", formatC(t(sum(N))/ntrial, digits=1, format="f"), "\n");
        cat("selection percentage of MTD:", formatC(sum(selpercent[which(p.true==target, arr.ind = TRUE)]), digits=1, format="f"), "\n");
        cat("percentage of patients treated at MTD:", formatC(sum(nptsdose[which(p.true==target, arr.ind = TRUE)])/npts*100, digits=1, format="f"), "\n");
        cat("percentage of early stopping due to toxicity:", formatC(100-sum(selpercent), digits=2, format="f"), "\n");
    }

}

waterfall.subtrial.mtd <- function(target, npts, ntox, cutoff.eli=0.95, extrasafe=FALSE, offset=0.05){
  ## obtain dose escalation and deescalation boundaries
  temp=get.boundary(target, ncohort=150, cohortsize=1, n.earlystop=100, p.saf="default", p.tox="default", cutoff.eli, extrasafe, print=FALSE);
  b.e=temp[2,];   # escalation boundary

  ## isotonic transformation using the pool adjacent violator algorithm (PAVA)
  pava <- function (x, wt = rep(1, length(x)))
  {
    n <- length(x)
    if (n <= 1)
      return(x)
    if (any(is.na(x)) || any(is.na(wt))) {
      stop("Missing values in 'x' or 'wt' not allowed")
    }
    lvlsets <- (1:n)
    repeat {
      viol <- (as.vector(diff(x)) < 0)
      if (!(any(viol)))
        break
      i <- min((1:(n - 1))[viol])
      lvl1 <- lvlsets[i]
      lvl2 <- lvlsets[i + 1]
      ilvl <- (lvlsets == lvl1 | lvlsets == lvl2)
      x[ilvl] <- sum(x[ilvl] * wt[ilvl])/sum(wt[ilvl])
      lvlsets[ilvl] <- lvl1
    }
    x
  }
  ## determine whether the dose has been eliminated during the trial
  y=ntox;
  n=npts;
  ndose=length(n);
  elimi=rep(0, ndose);
  is.escalation=0
  for(i in 1:ndose)
  {
    if(n[i]>=3) {if(1-pbeta(target, y[i]+1, n[i]-y[i]+1)>cutoff.eli) {elimi[i:ndose]=1; break;}}
  }
  if(extrasafe)
  {
    if(n[1]>=3) {if(1-pbeta(target, y[1]+1, n[1]-y[1]+1)>cutoff.eli-offset) {elimi[1:ndose]=1;}}
  }

  ## no dose should be selected (i.e., selectdose=99) if the first dose is already very toxic or
  ## all uneliminated doses are never used to treat patients
  if(elimi[1]==1 || sum(n[elimi==0])==0) { selectdose=99; }
  else
  {
    adm.set = (n!=0) & (elimi==0);
    adm.index = which(adm.set==T);
    y.adm = y[adm.set];
    n.adm = n[adm.set];

    ## poster mean and variance of toxicity probabilities using beta(0.05, 0.05) as the prior
    phat = (y.adm+0.05)/(n.adm+0.1);
    phat.var = (y.adm+0.05)*(n.adm-y.adm+0.05)/((n.adm+0.1)^2*(n.adm+0.1+1));

    ## perform the isotonic transformation using PAVA
    phat = pava(phat, wt=1/phat.var)
    phat = phat + (1:length(phat))*1E-10 ## break ties by adding an increasingly small number
    selectd = sort(abs(phat-target), index.return=T)$ix[1]  ## select dose closest to the target as the MTD
    selectdose = adm.index[selectd];

    if(y[selectdose]<=b.e[n[selectdose]]) { is.escalation=1 }
  }

  list(selectdose=selectdose, is.escalation=is.escalation)
}

waterfall.subtrial <- function(target, p.true, dosespace, npts, ntox, elimi, ncohort, cohortsize, n.earlystop=20, startdose=1, Nmax=1000,
							   p.saf="default", p.tox="default", cutoff.eli=0.95, extrasafe=FALSE, offset=0.05,totaln){
  ndoses1 = nrow(p.true); ndoses2=ncol(p.true)
  p.truee = p.true[dosespace];
  npts = npts; ntox=ntox; elimi=elimi;

  ## if the user does not provide p.saf and p.tox, set them to the default values
  if(p.saf=="default") p.saf=0.6*target;
  if(p.tox=="default") p.tox=1.4*target;

  ## simple error checking
  if(target<0.05) {cat("Error: the target is too low! \n"); return();}
  if(target>0.6)  {cat("Error: the target is too high! \n"); return();}
  if((target-p.saf)<(0.1*target)) {cat("Error: the probability deemed safe cannot be higher than or too close to the target! \n"); return();}
  if((p.tox-target)<(0.1*target)) {cat("Error: the probability deemed toxic cannot be lower than or too close to the target! \n"); return();}
  if(offset>=0.5) {cat("Error: the offset is too large! \n"); return();}

  ndose=length(p.truee);

  selectdose = 0; is.escalation=0# store the selected dose level

  ## obtain dose escalation and deescalation boundaries
  temp=get.boundary(target, ncohort=150, cohortsize=1, n.earlystop=100, p.saf, p.tox, cutoff.eli, extrasafe, print=FALSE);
  b.e=temp[2,];   # escalation boundary
  b.d=temp[3,];   # deescalation boundary
  b.elim=temp[4,];  # elimination boundary

  lambda1  = log((1-p.saf)/(1-target))/log(target*(1-p.saf)/(p.saf*(1-target)));
  lambda2  = log((1-target)/(1-p.tox))/log(p.tox*(1-target)/(target*(1-p.tox)));

  #set.seed(6)
  ################## simulate trials ###################
  y<-rep(0, ndose);    ## the number of DLT at each dose level
  n<-rep(0, ndose);    ## the number of patients treated at each dose level
  #ye=rep(0,ndose); ne=rep(0,ndose)
  earlystop=0;         ## indiate whether the trial terminates early
  d=startdose;         ## starting dose level
  elm = rep(0, ndose);  ## indicate whether doses are eliminated

  for(icohort in 1:ncohort){
    ### generate toxicity outcome
    y[d] = y[d] + sum(runif(cohortsize)<p.truee[d]);
    n[d] = n[d] + cohortsize;

    ## determine if the current dose should be eliminated
    if(!is.na(b.elim[n[d]])){
	    if(y[d]>=b.elim[n[d]]){
		    elm[d:ndose]=1;
		    if(d==1) {earlystop=1; break;}
	   }
     ## implement the extra safe rule by decreasing the elimination cutoff for the lowest dose
      if(extrasafe){
	  	  if(d==1 && y[1]>=3){
		      if(1-pbeta(target, y[1]+.05, n[1]-y[1]+.1)>cutoff.eli-offset) {earlystop=1; break;}
		   }
	    }
    }
    ## dose escalation/de-escalation
    if(y[d]<=b.e[n[d]] && d!=ndose) { if(elm[d+1]==0) d=d+1; }
    else if(y[d]>=b.d[n[d]] && d!=1) { d=d-1; }
	  else {d =d}

	if(n[d]==n.earlystop) break;
	if(sum(n)>=Nmax) break;
	if((totaln + sum(n))>=(ncohort*cohortsize)) break;
  }

  if(earlystop==1) {
		selectdose=99; elm = rep(1, ndose)
  }else{
	  wsmtd = waterfall.subtrial.mtd(target, n, y, cutoff.eli, extrasafe, offset)
	  selectdose = wsmtd$selectdose;
	  is.escalation = wsmtd$is.escalation
  }

  ## output results
  npts[dosespace] = n;  ntox[dosespace] = y;  elimi[dosespace] = elm


  list(ncohort=icohort, ntotal=icohort*cohortsize, startdose=startdose,
       npts=npts, ntox=ntox, totaltox=sum(ntox), totaln=sum(npts),
	     pctearlystop=sum(selectdose==99)*100, selectdose=selectdose, is.escalation=is.escalation, elimi=elimi)
}


get.oc.comb.waterfall <- function(p.true, target, ncohort, cohortsize, Nmax=NULL, n.earlystop=12, cutoff.eli=0.95,
                                  p.saf="default", p.tox="default", extrasafe=FALSE, offset=0.05, ntrial=1000){
  JJ=nrow(p.true); KK=ncol(p.true)
  if(JJ>KK) p.true = t(p.true)
  ## epsilon is the tolerance of the true MTD contour, e.g., target = 0.30, epsilon=0.03 means that doses with 0.30-0.04<=Pr(tox)<=0.30+0.04 are located at the MTD contour
  epsilon=0.03
  # for comparison purpose, we need to record the number of true MTDs and get the nselpercent
  nMTDs = paste(sort(which(abs(p.true-target)<=epsilon)), collapse=',')

  if(is.null(Nmax)) Nmax=1000

  aa=function(x) as.numeric(as.character(x))
  ###############################  phase I ###############################
  ## if the user does not provide p.saf and p.tox, set them to the default values
  if(p.saf=="default") p.saf=0.6*target;
  if(p.tox=="default") p.tox=1.4*target;

  ## simple error checking
  #if(npts[dose.curr[1], dose.curr[2]]==0)  {cat("Error: dose entered is not the current dose \n"); return(1);}
  if(target<0.05) {cat("Error: the target is too low! \n"); return(1);}
  if(target>0.6)  {cat("Error: the target is too high! \n"); return(1);}
  if((target-p.saf)<(0.1*target)) {cat("Error: the probability deemed safe cannot be higher than or too close to the target! \n"); return(1);}
  if((p.tox-target)<(0.1*target)) {cat("Error: the probability deemed toxic cannot be lower than or too close to the target! \n"); return(1);}

  # dose levels for agent 1 and 2
  ndoses1 <- nrow(p.true)
  ndoses2 <- ncol(p.true)

  ntrial.phase1=NULL; ntrial.mtd=NULL; ntrial.nt=NULL; ntrial.yt=NULL; ntrial.ne=NULL; ntrial.ye=NULL
  ntrial.ONEMTD=NULL;selonemtdpercent=0# report the selection percentage if only one MTD is recalled
  for(trial in 1:ntrial){
    # run the subtrials sequentially
    trial.result = NULL

	  # store toxicity outcome in y and the number of patients in n
	  ntox=matrix(rep(0, (ndoses2)*ndoses1), ncol=ndoses2); colnames(ntox) = paste('ntoxDoseB',1:ndoses2,sep='')
	  npts=matrix(rep(0, (ndoses2)*ndoses1), ncol=ndoses2); colnames(npts) = paste('nptsDoseB',1:ndoses2,sep='')
	  elimi = matrix(0, nrow=ndoses1, ncol=ndoses2);     colnames(elimi) = paste('elimiDoseB',1:ndoses2,sep='')
	  mtd=cbind('selectdoseA'=1:ndoses1, 'selectdoseB'=rep(NA, ndoses1))


	  trial.result = data.frame(cbind('trial'=rep(trial, ndoses1), mtd, npts, ntox, elimi)) #, n.e, y.e

	  totaln=0; startdose=1; dosespace=c(1:(ndoses1-1), (1:ndoses2)*ndoses1)
	  subtriali=1
	  while(totaln < ncohort*cohortsize){
		  Nmax1 = Nmax
		  subtrial=waterfall.subtrial(target, p.true=p.true,  dosespace=dosespace, npts=npts, ntox=ntox, elimi=elimi,
		                              ncohort=ncohort, cohortsize=cohortsize, n.earlystop=n.earlystop, startdose=startdose, Nmax=Nmax1,
									  p.saf='default', p.tox='default', cutoff.eli, extrasafe, offset, totaln=totaln)

		  # update dosespace for next subtrial if further subtrials are needed
		  selectdose = ifelse(subtrial$selectdose==99, 99, dosespace[subtrial$selectdose])
		  if(selectdose==99) break

	  	dj = ifelse(selectdose%%ndoses1==0, selectdose%/%ndoses1, selectdose%/%ndoses1+1)
		  di = selectdose - (dj-1) * ndoses1

		  # update total number of patients treated in the whole design
      totaln = aa(subtrial$totaln)
      # update npts and ntox
      npts=subtrial$npts; ntox=subtrial$ntox;

     # update elimi based on current information,
      elimi=subtrial$elimi

      if((subtriali==1) & (selectdose<ndoses1)){
        for(a in (di+1):ndoses1) for(b in 1:ndoses2) elimi[a,b]=1

        # need to adjust MTD for the first subtrial, also 'elimi' matrix should be updated differently
        if(subtrial$is.escalation==1){
          startdose=1;dosespace1=c(di + ((2:ndoses2)-1) * ndoses1)
          Nmax1 = Nmax
          subtrial1=waterfall.subtrial(target, p.true=p.true, dosespace=dosespace1, npts=npts, ntox=ntox, elimi=elimi,
                                       ncohort=ncohort, cohortsize=cohortsize, n.earlystop=n.earlystop, startdose=startdose, Nmax=Nmax1,
									   p.saf='default', p.tox='default', cutoff.eli, extrasafe, offset, totaln=totaln)

          selectdose1 = ifelse(subtrial1$selectdose==99, selectdose, dosespace1[subtrial1$selectdose])
          if(selectdose1==99) break

          dj = ifelse(selectdose1%%ndoses1==0, selectdose1%/%ndoses1, selectdose1%/%ndoses1+1)
          di = selectdose1 - (dj-1) * ndoses1

          # update total number of patients treated in the whole design
          totaln = aa(subtrial1$totaln)
          # update npts and ntox
          npts=subtrial1$npts; ntox=subtrial1$ntox;

          # update elimi based on current information,
          elimi=subtrial1$elimi
        }
      }

      subtriali=subtriali+1

      if(dj<ndoses2) elimi[di, (dj+1):ndoses2]=1

      # if current subtrial claimed the last level is mtd, then stop the whole trial
  		if(dj == ndoses2) break
      # otherwise, update the starting dose for the next adjacent subtrial. Since 1st dose level has been removed, thus the index is just dj.
      # and the dose space is ranging from [di-1, 2:ndoses2]
  		startdose = dj  #   #di-1 + (dj + 1 -1) * ndoses1
  		dosespace = di-1 + ((2:ndoses2) -1) * ndoses1

  		# if the current subtrial claims the MTD is located at [1, dj], then stop the whole trial since no additional subtrials are left.
  		if(di-1==0) break;

  	}
  	npts = t(apply(npts, 1, aa)); ntox = t(apply(ntox, 1, aa)); elimi = t(apply(elimi, 1, aa))

  	phat = (ntox+0.05)/(npts+0.1); phat = t(apply(phat, 1, aa))
    colnames(phat) = paste('phat', 1:ndoses2, sep='')
    ## perform the isotonic transformation using PAVA
    phat=Iso::biviso(phat,npts+0.1,warn=TRUE)[,];
    ## break the ties
    phat = phat+(1E-5)*(matrix(rep(1:dim(npts)[1], each = dim(npts)[2], len = length(npts)),dim(npts)[1],byrow=T) +
             matrix(rep(1:dim(npts)[2], each = dim(npts)[1], len = length(npts)),dim(npts)[1]))
    colnames(phat) = paste('phat', 1:ndoses2, sep='')

  	# save mtd information for one simulation based on elimination & phat information
  	for(k in ndoses1:1){
  		kn = npts[k,]; ky = ntox[k,]; kelimi = elimi[k,];
  		kphat = phat[k,]
  		if(kelimi[1]==1 || sum(npts[kelimi==0])==0) {
  			kseldose=99;
  		}else{
  			adm.set = (kn!=0) & (kelimi==0);
  			adm.index = which(adm.set==T);
  			y.adm = ky[adm.set];
  			n.adm = kn[adm.set];
  			selectd = sort(abs(kphat[adm.set]-target), index.return=T)$ix[1]  ## select dose closest to the target as the MTD
  			kseldose = adm.index[selectd];
  		}
  		mtd[k, 2] = kseldose
  		if(k<ndoses1) if(mtd[k+1,2]==ndoses2) mtd[k,2] = ndoses2
  		if(k<ndoses1) if(aa(mtd[k+1,2])==aa(mtd[k,2])) mtd[k,2] = 99
  	}
    # report ONE MTD for comparison
    onephat=aa(as.matrix(phat))  # ONLY report one MTD
    mtdpos=aa(apply(mtd, 1, function(x) (x[2]-1)*ndoses1+x[1]));
    onemtd = 99
    if(sum(mtdpos<=ndoses1*ndoses2)>0){
      onemtdpos = mtdpos[mtdpos<=ndoses1*ndoses2]
      onemtd=onemtdpos[which.min(abs(target-onephat[onemtdpos]))]
    }
    ntrial.ONEMTD=c(ntrial.ONEMTD, onemtd)
    selonemtdpercent=ifelse(is.element(onemtd, which((abs(p.true-target)-epsilon)<=1e-4))==T, selonemtdpercent+1, selonemtdpercent)

  	trial.result[1:ndoses1, grep('nptsDoseB',colnames(trial.result))]	=	npts
  	trial.result[1:ndoses1, grep('ntoxDoseB',colnames(trial.result))]	=	ntox
  	trial.result[1:ndoses1, grep('selectdose',colnames(trial.result))]	=	mtd
  	trial.result[1:ndoses1, grep('elimiDoseB',colnames(trial.result))]	=	elimi

  	ntrial.mtd = rbind(ntrial.mtd, cbind('trial'=rep(trial, nrow(mtd)), mtd))
  	## save npts and ntox
    ntrial.nt = rbind(ntrial.nt, cbind('trial'=rep(trial, nrow(npts)), npts))
    ntrial.yt = rbind(ntrial.yt, cbind('trial'=rep(trial, nrow(ntox)), ntox))

    trial.result = cbind(trial.result, phat)
    ntrial.phase1 = rbind(ntrial.phase1, trial.result)

  } # end: for ntrial
  selonemtdpercent=round(selonemtdpercent*100/ntrial,1)

  ntrial.mtd = data.frame(ntrial.mtd)
  colnames(ntrial.mtd)= c('trial', 'doseA', 'doseB')

  alltoxpercent = round(sum(sapply(1:ntrial, function(x) ifelse(sum(ntrial.mtd$doseB[ntrial.mtd$trial==x]==99)==ndoses1,1,0)),na.rm=T)*100/1000,3)
  stoppercent = round(sum(sapply(1:ntrial, function(x)  sum(ntrial.mtd$doseB[ntrial.mtd$trial==x]==99)))*100/1000,1)
  selpercent = matrix(0, nrow=ndoses1,ncol=ndoses2)
  nselpercent = 0 # for comparison purpose, compute the accuracy for returning all the true MTDs
  selpercent1 = 0; selpercent2 = 0
  mtdtable=NULL; mtdlist=list()

  for(triali in 1:ntrial){
  	mtddata = unique(as.matrix(ntrial.mtd[ntrial.mtd$trial==triali,2:3]))
  	mtddata = mtddata[!is.na(mtddata[,2]),]
  	if(length(mtddata)>0){
  	  if(length(mtddata)==2) mtddata = matrix(mtddata, ncol=2)
    	mtdlevel = aa(t(apply(mtddata, 1, function(x) (x[2]-1)*ndoses1 + x[1])))
    	mtdlevel[mtdlevel>ndoses1*ndoses2] = 99
    	if(sum(mtdlevel<=ndoses1*ndoses2)>0){
    		selpercent[mtdlevel[mtdlevel<=ndoses1*ndoses2]] = selpercent[mtdlevel[mtdlevel<=ndoses1*ndoses2]] + 1
    		mtdlist[[triali]]=mtdlevel[mtdlevel<=ndoses1*ndoses2]
    		mtdtable=c(mtdtable, paste(sort(aa(mtdlevel[mtdlevel<=ndoses1*ndoses2])), collapse=','))
    		if(paste(sort(aa(mtdlevel[mtdlevel<=ndoses1*ndoses2])), collapse=',') == nMTDs) nselpercent = nselpercent + 1
    		if(sum(mtdlevel<=ndoses1*ndoses2)==1 & sum(is.element(mtdlevel[mtdlevel<=ndoses1*ndoses2],which(p.true==target))==F)==0) selpercent1 = selpercent1 + 1
    		if(sum(mtdlevel<=ndoses1*ndoses2)==2 & sum(is.element(mtdlevel[mtdlevel<=ndoses1*ndoses2],which(p.true==target))==F)==0) selpercent2 = selpercent2 + 1
    	}else{
    		mtdlist[[triali]]=99
    		mtdtable = c(mtdtable, 99)
    	}
  	}
  } # end of triali in ntrial

  selpercent = round(selpercent*100/ntrial,2)
  rownames(selpercent) = paste('DoseA',1:ndoses1,sep=''); colnames(selpercent) = paste('DoseB',1:ndoses2,sep='')

  if(JJ>KK){ # need to transpose the matrix and results back
    ## print summary stats: selpercent
  cat("True toxicity rate of dose combinations:\n");
  for (i in 1:dim(p.true)[2]) cat(formatC(p.true[,i], digits=2, format="f", width=5), sep="  ", "\n");
  cat("\n");
  cat("selection percentage at each dose combination (%):\n");
  for (i in 1:dim(p.true)[2]) cat(formatC(selpercent[,i], digits=2, format="f", width=5), sep="  ", "\n");
  cat("\n");

  nptsdose=matrix(0,nrow=ndoses1,ncol=ndoses2)
  for(i in seq(1,nrow(ntrial.nt),by=ndoses1)) nptsdose = nptsdose + ntrial.nt[i+0:(ndoses1-1),-1]
## print summary stats: nptsdose
  cat("number of patients treated at each dose combination:\n");
  for (i in 1:dim(p.true)[2]) cat(formatC(nptsdose[,i]/ntrial, digits=2, format="f", width=5), sep ="  ", "\n");
  cat("\n");

## print summary stats: ntoxdose
  ntoxdose=matrix(0,nrow=ndoses1,ncol=ndoses2)
  for(i in seq(1,nrow(ntrial.yt),by=ndoses1)) ntoxdose = ntoxdose + ntrial.yt[i+0:(ndoses1-1),-1]
  ## print summary stats: ntoxdose
  cat("number of toxicity observed at each dose combination:\n");
  for (i in 1:dim(p.true)[2]) cat(formatC(ntoxdose[,i]/ntrial, digits=2, format="f", width=5), sep ="  ", "\n");
  cat("\n");
  cat("average number of toxicities:", formatC(sum(ntoxdose/ntrial), digits=1, format="f"), "\n");
  cat("average number of patients:", formatC(sum(nptsdose/ntrial), digits=1, format="f"), "\n");
  cat("percentage of patients treated at MTD contour:", paste(round(100*sum(nptsdose[which(abs(p.true-target)<=epsilon)])/sum(nptsdose),1),'%',sep=''), "\n");
  cat("percentage of patients treated above MTD contour:", paste(round(100*sum(nptsdose[which(p.true>target+epsilon)])/sum(nptsdose),1),'%',sep=''), "\n");
  cat("percentage of patients treated below MTD contour:", paste(round(100*sum(nptsdose[which(p.true<target-epsilon)])/sum(nptsdose),1),'%',sep=''), "\n");
  cat("percentage of correct selection of the MTD contour:", paste(round(100*nselpercent/ntrial,1), '%',sep=''), "\n");
  }else{
## print summary stats: selpercent
  cat("True toxicity rate of dose combinations:\n");
  for (i in 1:dim(p.true)[1]) cat(formatC(p.true[i,], digits=2, format="f", width=5), sep="  ", "\n");
  cat("\n");
  cat("selection percentage at each dose combination (%):\n");
  for (i in 1:dim(p.true)[1]) cat(formatC(selpercent[i,], digits=2, format="f", width=5), sep="  ", "\n");
  cat("\n");

  nptsdose=matrix(0,nrow=ndoses1,ncol=ndoses2)
  for(i in seq(1,nrow(ntrial.nt),by=ndoses1)) nptsdose = nptsdose + ntrial.nt[i+0:(ndoses1-1),-1]
## print summary stats: nptsdose
  cat("number of patients treated at each dose combination:\n");
  for (i in 1:dim(p.true)[1]) cat(formatC(nptsdose[i,]/ntrial, digits=2, format="f", width=5), sep ="  ", "\n");
  cat("\n");

## print summary stats: ntoxdose
  ntoxdose=matrix(0,nrow=ndoses1,ncol=ndoses2)
  for(i in seq(1,nrow(ntrial.yt),by=ndoses1)) ntoxdose = ntoxdose + ntrial.yt[i+0:(ndoses1-1),-1]
  ## print summary stats: ntoxdose
  cat("number of toxicity observed at each dose combination:\n");
  for (i in 1:dim(p.true)[1]) cat(formatC(ntoxdose[i,]/ntrial, digits=2, format="f", width=5), sep ="  ", "\n");
  cat("\n");
  cat("average number of toxicities:", formatC(sum(ntoxdose/ntrial), digits=1, format="f"), "\n");
  cat("average number of patients:", formatC(sum(nptsdose/ntrial), digits=1, format="f"), "\n");
  cat("percentage of patients treated at MTD contour:", paste(round(100*sum(nptsdose[which(abs(p.true-target)<=epsilon)])/sum(nptsdose),1),'%',sep=''), "\n");
  cat("percentage of patients treated above MTD contour:", paste(round(100*sum(nptsdose[which(p.true>target+epsilon)])/sum(nptsdose),1),'%',sep=''), "\n");
  cat("percentage of patients treated below MTD contour:", paste(round(100*sum(nptsdose[which(p.true<target-epsilon)])/sum(nptsdose),1),'%',sep=''), "\n");
  cat("percentage of correct selection of the MTD contour:", paste(round(100*nselpercent/ntrial,1), '%',sep=''), "\n");
  }
}

#################################################################################################################################
## the main code for get.oc.comb(.) starts here
## get the oc for drug combination trials
  JJ=nrow(p.true); KK=ncol(p.true)
  if(JJ>KK) { cat("Error: p.true should be arranged in a way (i.e., rotated) such that the number of rows is less than or equal to the number of columns.");
                return();}
  if(JJ*KK<=4 & ncohort<=6) {cat("Warning: the sample size is too small, which may lead to poor operating characteristics. Suggest to increase the number of cohort."); }

  if(JJ*KK>4 & ncohort<=8)  {cat("Warning: the sample size is too small, which may lead to poor operating characteristics. Suggest to increase the number of cohort."); }

  if(MTD.contour==FALSE){
    if(is.null(n.earlystop)==TRUE) n.earlystop=100
	  if(n.earlystop<=6) {cat("Warning: the value of n.earlystop is too low to ensure good operating characteristics. Recommend n.earlystop = 9 to 18 \n"); }
   # cat("\n\n", 'BOIN design is used for evaluating operating characteristics: ', "\n\n")
    get.oc.comb.boin(target, p.true, ncohort, cohortsize, n.earlystop, startdose, p.saf, p.tox, cutoff.eli, extrasafe, offset, ntrial)
  }
  if(MTD.contour==TRUE){
    if(is.null(n.earlystop)==TRUE) n.earlystop=12
	  if(n.earlystop<=6) {cat("Warning: the value of n.earlystop is too low to ensure good operating characteristics. Recommend n.earlystop = 9 to 18 \n"); }
    if(is.null(Nmax)==TRUE) Nmax = round(1.5*ncohort*cohortsize/min(dim(p.true)),0)
  #  cat("\n\n", 'Waterfall design is used for evaluating operating characteristics: ', "\n\n")
    get.oc.comb.waterfall(p.true, target, ncohort, cohortsize, Nmax=Nmax, n.earlystop=n.earlystop, cutoff.eli, p.saf, p.tox, extrasafe, offset, ntrial)
  }
}
